<?php
require_once '../../config/db.php';
ini_set('display_errors', 1);
error_reporting(E_ALL);

$halfDayMinutes = 240; // أقل من هالمدّة = نص يوم غياب
$workDaysMonth  = 30;
$workHoursDay   = 8;
$rate_working   = 1.5; // OT أيام العمل
$rate_offday    = 2.0; // OT أيام الراحة

/** ============== Helpers ============== */
function get_base_component_id(PDO $conn): int {
    $stmt = $conn->prepare("SELECT id FROM salary_components WHERE type = 'basic' LIMIT 1");
    $stmt->execute();
    $row = $stmt->fetch(PDO::FETCH_ASSOC);
    return $row ? (int)$row['id'] : 0;
}
function get_base_salary(PDO $conn, int $employee_id, int $base_component_id): float {
    $stmt = $conn->prepare("SELECT value FROM employee_salary_items WHERE employee_id = ? AND component_id = ? LIMIT 1");
    $stmt->execute([$employee_id, $base_component_id]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);
    return (float)($row['value'] ?? 0);
}
function month_bounds(int $month, int $year): array {
    $num_days = cal_days_in_month(CAL_GREGORIAN, $month, $year);
    $Y = (int)date('Y'); $M = (int)date('n'); $D = (int)date('j');
    if ($year > $Y || ($year == $Y && $month > $M)) return [0,0,true];  // مستقبل
    $last = $num_days;
    if ($year == $Y && $month == $M) $last = max(0, $D-1);              // إلى أمس
    return [$num_days, $last, false];
}
function get_work_weekdays(PDO $conn, int $emp_id): array {
    $stmt = $conn->prepare("SELECT department_id, day_off FROM users WHERE id=?");
    $stmt->execute([$emp_id]);
    $emp = $stmt->fetch(PDO::FETCH_ASSOC);
    $dept_id = (int)($emp['department_id'] ?? 0);
    $day_off = $emp['day_off'] ?? null;

    $stmt = $conn->prepare("SELECT day_of_week FROM department_work_days WHERE department_id=? AND is_working=1");
    $stmt->execute([$dept_id]);
    $names = $stmt->fetchAll(PDO::FETCH_COLUMN);

    $map = ['monday'=>1,'tuesday'=>2,'wednesday'=>3,'thursday'=>4,'friday'=>5,'saturday'=>6,'sunday'=>7];
    $wd = [];
    foreach ($names as $n) { $l=strtolower(trim($n)); if(isset($map[$l])) $wd[]=$map[$l]; }
    if ($day_off !== null && $day_off !== '') $wd = array_values(array_diff($wd, [(int)$day_off]));
    return $wd;
}
function get_leave_dates(PDO $conn, int $emp_id, int $month, int $year, int $last): array {
    if ($last<=0) return [];
    $stmt=$conn->prepare("
        SELECT start_date,end_date FROM leave_requests
        WHERE user_id=? AND status='مقبولة' AND (
          (MONTH(start_date)=? AND YEAR(start_date)=?) OR
          (MONTH(end_date)=? AND YEAR(end_date)=?) OR
          (start_date<=LAST_DAY(CONCAT(?, '-', ?, '-01')) AND end_date>=CONCAT(?, '-', ?, '-01'))
        )
    ");
    $stmt->execute([$emp_id,$month,$year,$month,$year,$year,$month,$year,$month]);
    $rows=$stmt->fetchAll(PDO::FETCH_ASSOC);

    $from=new DateTime(sprintf('%04d-%02d-01',$year,$month));
    $to=new DateTime(sprintf('%04d-%02d-%02d',$year,$month,$last));
    $dates=[];
    foreach($rows as $lv){
        try{$s=new DateTime($lv['start_date']);$e=new DateTime($lv['end_date']);}catch(Exception $e){continue;}
        if($s<$from)$s=clone $from; if($e>$to)$e=clone $to;
        while($s<=$e){ $dates[]=$s->format('Y-m-d'); $s->modify('+1 day'); }
    }
    return $dates;
}
function get_shift_times(PDO $conn, int $emp_id): array {
    $stmt = $conn->prepare("
        SELECT ws.start_time, ws.end_time
        FROM users u
        LEFT JOIN work_shifts ws ON ws.id = u.work_shift_id
        WHERE u.id = ?
    ");
    $stmt->execute([$emp_id]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC) ?: [];
    $start = ($row['start_time'] ?? '08:00:00') ?: '08:00:00';
    $end   = ($row['end_time']   ?? '16:00:00') ?: '16:00:00';
    return [$start, $end];
}
function get_shift_start(PDO $conn, int $emp_id): string { [$s,] = get_shift_times($conn,$emp_id); return $s; }

/* حضور الشهر + worked_minutes */
function get_month_attendance(PDO $conn, int $emp_id, int $month, int $year): array {
    $stmt = $conn->prepare("
        SELECT DATE(`date`) AS day,
               MIN(TIME(`check_in`)) AS in_time,
               MAX(TIME(`check_out`)) AS out_time,
               MAX(status) AS status
        FROM attendance
        WHERE user_id=? AND MONTH(`date`)=? AND YEAR(`date`)=?
        GROUP BY DATE(`date`)
        ORDER BY day
    ");
    $stmt->execute([$emp_id, $month, $year]);
    $out = [];
    while ($r = $stmt->fetch(PDO::FETCH_ASSOC)) {
        $in   = $r['in_time']; $outt = $r['out_time'];
        $worked = 0;
        if ($in && $outt) $worked = max(0, (int)round((strtotime($outt)-strtotime($in))/60));
        $r['worked_minutes'] = $worked;
        $out[$r['day']] = $r;
    }
    return $out;
}
/* غياب كقيمة عشرية (0.5 لنصف يوم) */
function get_employee_absent_days(PDO $conn, int $emp_id, int $month, int $year, int $halfDayMinutes=240): float {
    [$num,$last,$future]=month_bounds($month,$year);
    if($future || $last<=0) return 0.0;

    $wd    = get_work_weekdays($conn,$emp_id);
    $att   = get_month_attendance($conn,$emp_id,$month,$year);
    $leave = get_leave_dates($conn,$emp_id,$month,$year,$last);

    $abs=0.0;
    for($d=1;$d<=$last;$d++){
        $date=sprintf('%04d-%02d-%02d',$year,$month,$d);
        $dow=(int)date('N',strtotime($date));
        if(!in_array($dow,$wd,true)) continue;
        if(in_array($date,$leave,true)) continue;

        $row=$att[$date]??null;
        if(!$row){ $abs += 1.0; continue; }
        $worked=(int)($row['worked_minutes']??0);
        if($worked < $halfDayMinutes) $abs += 0.5;
    }
    return $abs;
}
/** ============== End Helpers ============== */

/** Inputs */
$emp_id = (int)($_GET['id'] ?? 0);
$month  = (int)($_GET['month'] ?? date('n'));
$year   = (int)($_GET['year'] ?? date('Y'));

/** تحقق الموظف */
$stmt = $conn->prepare("SELECT name FROM users WHERE id=?");
$stmt->execute([$emp_id]);
$emp = $stmt->fetch(PDO::FETCH_ASSOC);
if (!$emp) {
    echo "<div class='text-red-600'>الموظف غير موجود</div>
          <button class='move-button px-4 py-2 mt-3' onclick=\"$('#employeeDetailsModal').addClass('hidden')\">إغلاق</button>";
    exit;
}

/** حدود الشهر (لا نحسب المستقبل) */
[$num_days, $lastDayToCount, $isFuture] = month_bounds($month, $year);
if ($isFuture) {
    echo "<div class='text-gray-600'>لا توجد بيانات مستقبلية لهذا الشهر.</div>
          <button class='move-button px-4 py-2 mt-3' onclick=\"$('#employeeDetailsModal').addClass('hidden')\">إغلاق</button>";
    exit;
}

/** بيانات الراتب/الشفت */
$base_component_id = get_base_component_id($conn);
$baseSalary = get_base_salary($conn, $emp_id, $base_component_id);
$salaryDay  = $workDaysMonth ? ($baseSalary / $workDaysMonth) : 0;
$deductionPerAbsence = $salaryDay * 1.5;
$deductionPerHour    = $workHoursDay ? ($deductionPerAbsence / $workHoursDay) : 0;
$hourly = ($workDaysMonth && $workHoursDay) ? $baseSalary / ($workDaysMonth * $workHoursDay) : 0;

[$shift_start, $shift_end] = get_shift_times($conn, $emp_id);
$workWeekdays = get_work_weekdays($conn, $emp_id);
$attendances  = get_month_attendance($conn, $emp_id, $month, $year);
$leaveDates   = get_leave_dates($conn, $emp_id, $month, $year, $lastDayToCount);

/** بناء الصفوف + إجماليات */
$rows = '';
$total_late_minutes    = 0;
$total_late_deduction  = 0;
$total_ot_mins_working = 0;
$total_ot_mins_offday  = 0;
$total_ot_amount       = 0;

for ($d = 1; $d <= $lastDayToCount; $d++) {
    $date    = sprintf('%04d-%02d-%02d', $year, $month, $d);
    $weekday = (int)date('N', strtotime($date));
    $dayName = date('l', strtotime($date));

    $att    = $attendances[$date] ?? null;
    $status = $att['status'] ?? '-';
    $in     = $att['in_time'] ?? '-';
    $out    = $att['out_time'] ?? '-';
    $worked = (int)($att['worked_minutes'] ?? 0);

    $lateTxt = '-'; $dedTxt='-'; $otTxt='-'; $otAmtTxt='-';

    // غير عمل
    if (!in_array($weekday, $workWeekdays, true)) {
        if ($att && $worked > 0) {
            $ot_mins   = $worked;
            $ot_amount = ($hourly * $rate_offday) * ($ot_mins / 60);
            $total_ot_mins_offday += $ot_mins;
            $total_ot_amount      += $ot_amount;
            $otTxt    = floor($ot_mins/60).' س '.($ot_mins%60).' د';
            $otAmtTxt = number_format($ot_amount, 2).' ﷼';
        }
        $rows .= "<tr class='bg-gray-50'>
            <td>$date</td><td>$dayName</td><td>$in</td><td>$out</td>
            <td>غير عمل</td>
            <td class='text-red-600 font-bold'>$lateTxt</td>
            <td class='text-red-600 font-bold'>$dedTxt</td>
            <td class='text-green-700 font-bold'>$otTxt</td>
            <td class='text-green-700 font-bold'>$otAmtTxt</td>
        </tr>";
        continue;
    }

    // إجازة
    if (in_array($date, $leaveDates, true)) {
        $rows .= "<tr class='bg-green-50'>
            <td>$date</td><td>$dayName</td><td>-</td><td>-</td>
            <td>إجازة</td>
            <td class='text-red-600 font-bold'>-</td>
            <td class='text-red-600 font-bold'>-</td>
            <td class='text-green-700 font-bold'>-</td>
            <td class='text-green-700 font-bold'>-</td>
        </tr>";
        continue;
    }

    // غياب كامل
    if (!$att) {
        $rows .= "<tr class='bg-red-50'>
            <td>$date</td><td>$dayName</td><td>-</td><td>-</td>
            <td>غياب</td>
            <td class='text-red-600 font-bold'>-</td>
            <td class='text-red-600 font-bold'>-</td>
            <td class='text-green-700 font-bold'>-</td>
            <td class='text-green-700 font-bold'>-</td>
        </tr>";
        continue;
    }

    // نص يوم
    if ($worked < $halfDayMinutes) {
        $rows .= "<tr class='bg-yellow-50'>
            <td>$date</td><td>$dayName</td><td>$in</td><td>$out</td>
            <td>غياب جزئي (نصف يوم)</td>
            <td class='text-red-600 font-bold'>-</td>
            <td class='text-red-600 font-bold'>-</td>
            <td class='text-green-700 font-bold'>-</td>
            <td class='text-green-700 font-bold'>-</td>
        </tr>";
        continue;
    }

    // حاضر كامل: تأخير
    if ($in !== '-' && $in > $shift_start && in_array($status, ['حاضر','متأخر'], true)) {
        $diff = strtotime($in) - strtotime($shift_start);
        if ($diff > 0) {
            $mins = (int)round($diff/60);
            $h=intdiv($mins,60); $m=$mins%60;
            $lateTxt = $h? "$h ساعة".($m? " $m دقيقة":'') : "$m دقيقة";
            $deduction = $deductionPerHour * ($mins/60);
            $dedTxt = number_format($deduction,2).' ﷼';
            $total_late_minutes   += $mins;
            $total_late_deduction += $deduction;
        }
    } else {
        $lateTxt = '0'; $dedTxt='0';
    }

    // OT يوم عمل: بعد نهاية الشفت وبسقف الزيادة عن ساعات اليوم
    if ($out !== '-' && strtotime($out) > strtotime($shift_end)) {
        $after_end = (int)round((strtotime($out)-strtotime($shift_end))/60);
        $cap       = max(0, $worked - ($workHoursDay*60));
        $ot_mins   = max(0, min($after_end, $cap));
        if ($ot_mins > 0) {
            $ot_amount = ($hourly * $rate_working) * ($ot_mins/60);
            $total_ot_mins_working += $ot_mins;
            $total_ot_amount       += $ot_amount;
            $otTxt    = floor($ot_mins/60).' س '.($ot_mins%60).' د';
            $otAmtTxt = number_format($ot_amount, 2).' ﷼';
        }
    }

    $rows .= "<tr>
        <td>$date</td><td>$dayName</td><td>$in</td><td>$out</td>
        <td>".htmlspecialchars($status)."</td>
        <td class='text-red-600 font-bold'>$lateTxt</td>
        <td class='text-red-600 font-bold'>$dedTxt</td>
        <td class='text-green-700 font-bold'>$otTxt</td>
        <td class='text-green-700 font-bold'>$otAmtTxt</td>
    </tr>";
}

/** إجماليات الغياب والخصومات والصافي */
$absentDays       = get_employee_absent_days($conn, $emp_id, $month, $year, $halfDayMinutes);
$absenceDeduction = $absentDays * $deductionPerAbsence;

$total_ot_hours   = round(($total_ot_mins_working + $total_ot_mins_offday)/60, 2);
$totalAll         = $total_late_deduction + $absenceDeduction;
$grandNet         = $totalAll - $total_ot_amount;

/** ====== Table Styles ====== */
?>
<style>
/* طباعة نظيفة بدون صفحة فاضية */
@media print {
  /* أخفِ كل عناصر الصفحة */
  body > :not(#employeeDetailsModal) { display: none !important; }

  /* أظهر المودال كصفحة عادية */
  #employeeDetailsModal {
    position: static !important;
    display: block !important;
    background: transparent !important; /* بدون طبقة التعتيم */
    inset: auto !important;
  }
  /* بطاقة المودال الداخلية */
  #employeeDetailsModal > div {
    box-shadow: none !important;
    border: 0 !important;
    width: 100% !important;
    max-width: none !important;
    max-height: none !important;
    overflow: visible !important;
    padding: 0 !important;
    margin: 0 !important;
  }

  /* الجدول يمتد 100% ويُلغى الهيدر اللزّاق */
  #employeeDetailsModal table { width: 100% !important; }
  #employeeDetailsModal .sticky-head { position: static !important; top: auto !important; }

  /* لا نطبع الأزرار وعناصر التحكم */
  #employeeDetailsModal .no-print,
  #employeeDetailsModal button { display: none !important; }

  /* إعدادات الصفحة */
  @page { size: A4 landscape; margin: 10mm; }
}


.table-compact { table-layout: fixed; width: 100%; border-collapse: separate; border-spacing: 0; }
.table-minw    { min-width: 980px; }
th, td { vertical-align: middle; }
.sticky-head { position: sticky; top: 0; z-index: 10; }

</style>

<h3 class="text-xl font-bold mb-4">تفاصيل التأخير والإضافي للموظف: <?= htmlspecialchars($emp['name']) ?> (<?= $month ?>/<?= $year ?>)</h3>

<table class="table-compact table-minw border rounded shadow text-sm mb-2">
  <colgroup>
    <col style="width:12%">
    <col style="width:12%">
    <col style="width:10%">
    <col style="width:10%">
    <col style="width:12%">
    <col style="width:10%">
    <col style="width:12%">
    <col style="width:10%">
    <col style="width:12%">
  </colgroup>
  <thead class="bg-gray-100 sticky-head">
    <tr>
      <th class="p-2 border">التاريخ</th>
      <th class="p-2 border">اليوم</th>
      <th class="p-2 border">وقت الحضور</th>
      <th class="p-2 border">وقت الانصراف</th>
      <th class="p-2 border">الحالة</th>
      <th class="p-2 border">دقيقة التأخير</th>
      <th class="p-2 border">خصم التأخير</th>
      <th class="p-2 border">أوفر تايم</th>
      <th class="p-2 border">مبلغ الإضافي</th>
    </tr>
  </thead>
  <tbody>
    <?= $rows ?: "<tr><td colspan='9' class='text-center p-4 text-gray-500'>لا توجد بيانات ضمن الفترة المحسوبة.</td></tr>" ?>

<?php if ($rows): ?>
<tr class="bg-gray-200 font-bold">
  <td colspan="5" class="text-center">إجمالي التأخير</td>
  <td class="text-red-700"><?= (int)$total_late_minutes ?> دقيقة</td>
  <td class="text-red-700"><?= number_format($total_late_deduction, 2) ?> ﷼</td>
  <td colspan="2"></td> <!-- 5 + 1 + 1 + 2 = 9 -->
</tr>

<tr>
  <td colspan="5" class="p-2 text-center font-bold">أيام الغياب</td>
  <td class="p-2 text-red-700"><?= number_format($absentDays, 1) ?> يوم</td>
  <td class="p-2 text-red-700"><?= number_format($absenceDeduction, 2) ?> ﷼</td>
  <td colspan="2"></td> <!-- كان td واحد فقط: صار colspan="2" -->
</tr>

<tr class="bg-green-100 font-bold">
  <td colspan="5" class="text-center">إجمالي الإضافي</td>
  <td class="text-green-700"><?= $total_ot_hours ?> ساعة</td>
  <td class="text-green-700"><?= number_format($total_ot_amount, 2) ?> ﷼</td>
  <td colspan="2"></td> <!-- 5 + 1 + 1 + 2 = 9 -->
</tr>

<tr class="bg-gray-300 font-bold">
  <td colspan="6" class="p-2 text-center">الصافي (الخصومات - الإضافي)</td>
  <td class="p-2" colspan="3"><?= number_format($grandNet, 2) ?> ﷼</td> <!-- 6 + 3 = 9 -->
</tr>
<?php endif; ?>

  </tbody>
</table>

<div class="flex justify-end gap-2 mt-4 no-print">
  <button class="move-button px-4 py-2" onclick="printEmployeeDetails()">طباعة</button>
  <button class="move-button px-4 py-2" onclick="$('#employeeDetailsModal').addClass('hidden')">إغلاق</button>
</div>

<script>
  function printEmployeeDetails() {
    window.print();
  }
</script>