<?php
session_start();
require_once '../../config/db.php';
require_once '../../auth.php';
// ===== PATCH START: helpers & bounds =====
function get_base_component_id(PDO $conn): int {
    $stmt = $conn->prepare("SELECT id FROM salary_components WHERE type = 'basic' LIMIT 1");
    $stmt->execute();
    $row = $stmt->fetch(PDO::FETCH_ASSOC);
    return $row ? (int)$row['id'] : 0;
}
function get_base_salary(PDO $conn, int $employee_id, int $base_component_id): float {
    $stmt = $conn->prepare("SELECT value FROM employee_salary_items WHERE employee_id = ? AND component_id = ?");
    $stmt->execute([$employee_id, $base_component_id]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);
    return (float)($row['value'] ?? 0);
}
function month_bounds(int $month, int $year): array {
    $num_days = cal_days_in_month(CAL_GREGORIAN, $month, $year);
    $Y = (int)date('Y'); $M = (int)date('n'); $D = (int)date('j');
    if ($year > $Y || ($year == $Y && $month > $M)) return [0,0,true];
    $last = $num_days;
    if ($year == $Y && $month == $M) $last = max(0, $D-1); // نحسب إلى أمس
    return [$num_days, $last, false];
}
function get_work_weekdays(PDO $conn, int $emp_id): array {
    $stmt = $conn->prepare("SELECT department_id, day_off FROM users WHERE id = ?");
    $stmt->execute([$emp_id]);
    $emp = $stmt->fetch(PDO::FETCH_ASSOC);
    $dept_id = (int)($emp['department_id'] ?? 0);
    $day_off = $emp['day_off'] ?? null;

    $stmt = $conn->prepare("SELECT day_of_week FROM department_work_days WHERE department_id=? AND is_working=1");
    $stmt->execute([$dept_id]);
    $names = $stmt->fetchAll(PDO::FETCH_COLUMN);

    $map=['monday'=>1,'tuesday'=>2,'wednesday'=>3,'thursday'=>4,'friday'=>5,'saturday'=>6,'sunday'=>7];
    $wd=[]; foreach($names as $n){ $l=strtolower(trim($n)); if(isset($map[$l])) $wd[]=$map[$l]; }
    if ($day_off !== null && $day_off !== '') $wd = array_values(array_diff($wd, [(int)$day_off]));
    return $wd;
}
function get_leave_dates(PDO $conn, int $emp_id, int $month, int $year, int $last): array {
    if ($last<=0) return [];
    $stmt=$conn->prepare("
        SELECT start_date,end_date FROM leave_requests
        WHERE user_id=? AND status='مقبولة' AND (
          (MONTH(start_date)=? AND YEAR(start_date)=?) OR
          (MONTH(end_date)=? AND YEAR(end_date)=?) OR
          (start_date<=LAST_DAY(CONCAT(?, '-', ?, '-01')) AND end_date>=CONCAT(?, '-', ?, '-01'))
        )
    ");
    $stmt->execute([$emp_id,$month,$year,$month,$year,$year,$month,$year,$month]);
    $rows=$stmt->fetchAll(PDO::FETCH_ASSOC);

    $from=new DateTime(sprintf('%04d-%02d-01',$year,$month));
    $to=new DateTime(sprintf('%04d-%02d-%02d',$year,$month,$last));
    $dates=[];
    foreach($rows as $lv){
        try{$s=new DateTime($lv['start_date']);$e=new DateTime($lv['end_date']);}catch(Exception $e){continue;}
        if($s<$from)$s=clone $from; if($e>$to)$e=clone $to;
        while($s<=$e){ $dates[]=$s->format('Y-m-d'); $s->modify('+1 day'); }
    }
    return $dates;
}
function get_shift_start(PDO $conn, int $emp_id): string {
    $stmt=$conn->prepare("SELECT ws.start_time FROM users u LEFT JOIN work_shifts ws ON ws.id=u.work_shift_id WHERE u.id=?");
    $stmt->execute([$emp_id]); $row=$stmt->fetch(PDO::FETCH_ASSOC);
    $raw=$row['start_time']??null; return ($raw && $raw!=='00:00:00')? $raw : '08:00:00';
}
function get_month_attendance(PDO $conn, int $emp_id, int $month, int $year): array {
    $stmt = $conn->prepare("
        SELECT DATE(`date`) AS day,
               MIN(TIME(`check_in`)) AS in_time,
               MAX(TIME(`check_out`)) AS out_time,
               MAX(status) AS status
        FROM attendance
        WHERE user_id=? AND MONTH(`date`)=? AND YEAR(`date`)=?
        GROUP BY DATE(`date`)
        ORDER BY day
    ");
    $stmt->execute([$emp_id, $month, $year]);
    $out = [];
    while ($r = $stmt->fetch(PDO::FETCH_ASSOC)) {
        $in   = $r['in_time'];
        $outt = $r['out_time'];
        $worked = 0;
        if ($in && $outt) {
            $worked = max(0, (int)round((strtotime($outt) - strtotime($in)) / 60));
        }
        $r['worked_minutes'] = $worked;
        $out[$r['day']] = $r;
    }
    return $out;
}

// ===== PATCH END
function get_shift_times(PDO $conn, int $emp_id): array {
    $stmt = $conn->prepare("
        SELECT ws.start_time, ws.end_time
        FROM users u
        LEFT JOIN work_shifts ws ON ws.id = u.work_shift_id
        WHERE u.id = ?
    ");
    $stmt->execute([$emp_id]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC) ?: [];
    $start = ($row['start_time'] ?? '08:00:00') ?: '08:00:00';
    $end   = ($row['end_time']   ?? '16:00:00') ?: '16:00:00';
    return [$start, $end];
}
function get_employee_overtime(PDO $conn, int $emp_id, int $month, int $year, int $halfDayMinutes = 240, int $workHoursDay = 8): array {
    [$num,$last,$future] = month_bounds($month,$year);
    if ($future || $last <= 0) return ['mins_working'=>0,'mins_offday'=>0,'hours_total'=>0.0];

    [$shift_start, $shift_end] = get_shift_times($conn, $emp_id);
    $wd    = get_work_weekdays($conn, $emp_id);
    $att   = get_month_attendance($conn, $emp_id, $month, $year);
    $leave = get_leave_dates($conn, $emp_id, $month, $year, $last);

    $mins_working = 0;  // أوفر تايم في أيام العمل
    $mins_offday  = 0;  // أوفر تايم في أيام الراحة
    $standard_minutes = $workHoursDay * 60;

    for ($d=1; $d <= $last; $d++) {
        $date = sprintf('%04d-%02d-%02d', $year, $month, $d);
        $dow  = (int)date('N', strtotime($date));

        $row = $att[$date] ?? null;
        if (!$row) continue;                   // ما حضر = ما فيه أوفر
        if (in_array($date, $leave, true)) continue;

        $worked = (int)($row['worked_minutes'] ?? 0);

        if (!in_array($dow, $wd, true)) {
            // يوم غير عمل: كل العمل أوفر تايم
            $mins_offday += $worked;
            continue;
        }

        // يوم عمل: نحسب فقط بعد نهاية الشفت، ولليوم المكتمل (مو نصف يوم)
        if ($worked < $halfDayMinutes) continue;

        $out = $row['out_time'] ?? null;
        if (!$out) continue;

        $after_end = max(0, (int)round((strtotime($out) - strtotime($shift_end)) / 60));
        // لا تتجاوز الزيادة عن الساعات النظامية
        $cap = max(0, $worked - $standard_minutes);
        $mins_working += min($after_end, $cap);
    }

    $total_hours = round(($mins_working + $mins_offday) / 60, 2);
    return ['mins_working'=>$mins_working, 'mins_offday'=>$mins_offday, 'hours_total'=>$total_hours];
}

function calc_overtime_amount(float $baseSalary, int $workDaysMonth, int $workHoursDay, int $mins_working, int $mins_offday, float $rate_working = 1.5, float $rate_offday = 2.0): float {
    if ($workDaysMonth <= 0 || $workHoursDay <= 0) return 0.0;
    $hourly = $baseSalary / ($workDaysMonth * $workHoursDay); // أجر الساعة الأساسي
    return ($mins_working/60) * $rate_working * $hourly
         + ($mins_offday/60)  * $rate_offday  * $hourly;
}

// متغيرات الألوان من الجلسة
$user = $_SESSION['user'];
$role = $user['role'];
$color = $user['color'];
$userThemeColor      = $color['hex_code'];
$userThemeTextColor  = $color['text_color'];
$userThemeNavColor   = $color['nav_color'] ?? '#233a58';
$currentUserColorId  = $user['color_id'];
$allColors           = $user['colors'];
$userThemeSecondTextColor = $color['second_text_color'] ?? $userThemeTextColor;

$month = isset($_GET['month']) ? intval($_GET['month']) : intval(date('n'));
$year  = isset($_GET['year'])  ? intval($_GET['year'])  : intval(date('Y'));
$halfDayMinutes = 240; // الحد الأدنى للحضور عشان يُعتبر اليوم حاضر (غير كذا = نص غياب)


/** ساعات التأخير (تحسب فقط أيام العمل الماضية وغير الواقعة ضمن إجازة) **/
function get_employee_late_hours(PDO $conn, int $emp_id, int $month, int $year, int $workHoursDay = 8, int $halfDayMinutes = 240): float {
    [$num,$last,$future] = month_bounds($month,$year);
    if ($future || $last <= 0) return 0.0;

    $shift = get_shift_start($conn,$emp_id);
    $wd    = get_work_weekdays($conn,$emp_id);
    $att   = get_month_attendance($conn,$emp_id,$month,$year);
    $leave = get_leave_dates($conn,$emp_id,$month,$year,$last);

    $mins = 0;
    for ($d=1; $d <= $last; $d++) {
        $date = sprintf('%04d-%02d-%02d', $year, $month, $d);
        $dow  = (int)date('N', strtotime($date));
        if (!in_array($dow, $wd, true)) continue;
        if (in_array($date, $leave, true)) continue;

        $row = $att[$date] ?? null;
        if (!$row) continue;

        // لو أقل من نصف يوم، نطنّش التأخير
        if ((int)($row['worked_minutes'] ?? 0) < $halfDayMinutes) continue;

        $in = $row['in_time'] ?? null;
        $status = $row['status'] ?? '';
        if ($in && in_array($status, ['حاضر','متأخر'], true) && $in > $shift) {
            $diff = strtotime($in) - strtotime($shift);
            if ($diff > 0) $mins += (int)round($diff/60);
        }
    }
    return $mins / 60.0;
}

/** أيام الغياب (نفس منطق الأيام الماضية + غير عمل + إجازات) **/
// ===== PATCH REPLACE: get_employee_absent_days =====
function get_employee_absent_days(PDO $conn, int $emp_id, int $month, int $year, int $halfDayMinutes = 240): float {
    [$num,$last,$future] = month_bounds($month,$year);
    if ($future || $last <= 0) return 0.0;

    $wd    = get_work_weekdays($conn,$emp_id);
    $att   = get_month_attendance($conn,$emp_id,$month,$year);
    $leave = get_leave_dates($conn,$emp_id,$month,$year,$last);

    $abs = 0.0;
    for ($d=1; $d <= $last; $d++) {
        $date = sprintf('%04d-%02d-%02d', $year, $month, $d);
        $dow  = (int)date('N', strtotime($date));
        if (!in_array($dow, $wd, true)) continue;          // غير عمل
        if (in_array($date, $leave, true)) continue;        // إجازة

        $row = $att[$date] ?? null;
        if (!$row) {                                        // لا حضور نهائي = يوم كامل
            $abs += 1.0;
            continue;
        }

        $worked = (int)($row['worked_minutes'] ?? 0);
        if ($worked < $halfDayMinutes) {                    // أقل من الحد = نص يوم
            $abs += 0.5;
        }
        // غير كذا يعتبر حاضر (ما نزيد الغياب)
    }
    return $abs;
}

// ===== PATCH END


/** مدخلات **/
$month = isset($_GET['month']) ? (int)$_GET['month'] : (int)date('n');
$year  = isset($_GET['year'])  ? (int)$_GET['year']  : (int)date('Y');

$workDaysMonth = 30;
$workHoursDay  = 8;
$base_component_id = get_base_component_id($conn);

/** الموظفون **/
$stmt = $conn->prepare("SELECT id, name FROM users WHERE employment_status = 'نشط' ORDER BY name");
$stmt->execute();
$employees = $stmt->fetchAll(PDO::FETCH_ASSOC);

$report = [];
$totalLateHours = 0.0;
$totalDeduction = 0.0;

foreach ($employees as $emp) {
    $emp_id = (int)$emp['id'];
$lateHours  = get_employee_late_hours($conn, $emp_id, $month, $year, $workHoursDay, $halfDayMinutes);
$absentDays = get_employee_absent_days($conn, $emp_id, $month, $year, $halfDayMinutes);

    $baseSalary = get_base_salary($conn, $emp_id, $base_component_id);
    $salaryDay  = $workDaysMonth ? ($baseSalary / $workDaysMonth) : 0;
    $deductionPerAbsence = $salaryDay * 1.5;
    $deductionPerHour    = $workHoursDay ? ($deductionPerAbsence / $workHoursDay) : 0;

    $lateDeduction    = $lateHours * $deductionPerHour;
    $absenceDeduction = $absentDays * $deductionPerAbsence;
    $totalEmployeeDeduction = $lateDeduction + $absenceDeduction;

    $totalLateHours += $lateHours;
    $totalDeduction += $totalEmployeeDeduction;
    
    $ot = get_employee_overtime($conn, $emp_id, $month, $year, $halfDayMinutes, $workHoursDay);
$otAmount = calc_overtime_amount($baseSalary, $workDaysMonth, $workHoursDay, $ot['mins_working'], $ot['mins_offday'], 1.5, 2.0);

$net = $totalEmployeeDeduction - $otAmount;

$totalOtHours  = ($totalOtHours  ?? 0) + $ot['hours_total'];
$totalOtAmount = ($totalOtAmount ?? 0) + $otAmount;
$totalNet      = ($totalNet      ?? 0) + $net;
$totalOtHours  = 0;
$totalOtAmount = 0;
$totalNet      = 0;

    $report[] = [
    'id'         => $emp_id,
    'name'       => $emp['name'],
    'lateHours'  => $lateHours,
    'absentDays' => $absentDays,
    'deduction'  => $totalEmployeeDeduction,
    'otHours'    => $ot['hours_total'],
    'otAmount'   => $otAmount,
    'net'        => $net
];
}
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
  <meta charset="UTF-8">
  <title>تقرير التأخير والغياب</title>
  <script src="https://cdn.tailwindcss.com"></script>
  <style>
    body {
      background-color: <?= $userThemeColor ?>;
      color: <?= $userThemeTextColor ?>;
      font-family: 'Tahoma', sans-serif;
    }
    .table th, .table td {
      text-align: center;
    }
    .row-click { transition: background-color .15s ease; }
.row-click:hover { filter: brightness(0.98); }

  </style>

</head>
    <!-- ناف بار مطابق للثاني (متوافق مع متغيرات الصفحة الأولى) -->
<nav class="bg-white shadow-md fixed w-full z-50">
  <div class="max-w-7xl mx-auto px-4">
    <div class="flex justify-between items-center h-18">
      <!--  قائمة اختيار الألوان -->
      <form method="POST" action="../../tasks/update_color.php" style=" width: 200px;display: flex;">
        <label class="text-lg font-bold py-2 ml-2">النمط:</label>
        <select name="theme_color_id" id="theme_color_select" class="card" onchange="this.form.submit()">
          <?php if (!empty($allColors)): ?>
            <?php foreach ($allColors as $color): ?>
              <option
                value="<?php echo htmlspecialchars($color['id']); ?>"
                data-hex="<?php echo htmlspecialchars($color['hex_code']); ?>"
                <?php echo ($color['id'] == $currentUserColorId) ? 'selected' : ''; ?>>
                <?php echo htmlspecialchars($color['color_name']); ?>
              </option>
            <?php endforeach; ?>
          <?php else: ?>
            <option value="">No colors available</option>
          <?php endif; ?>
        </select>
      </form>

      <!-- القوائم الرئيسية -->
      <div class="hidden md:flex items-center space-x-6 rtl:space-x-reverse text-base font-medium">
        <!-- المعاملات -->
        <?php if (in_array($role, ['employee', 'manager', 'admin'])): ?>
          <div class="relative inline-block text-right">
            <button id="taskMenu" class="text text-lg hover:text-indigo-700">📋 المعاملات </button>
            <div id="taskDropdown" class="card hidden absolute right-0 mt-2 w-56 bg-white shadow-lg rounded-md py-2 text-right z-50">
              <a href="/erp_task_system/tasks/kanban_new.php" class="block px-4 py-2 hover:bg-gray-100">🔄 بالمراحل</a>
              <a href="/erp_task_system/tasks/list.php" class="block px-4 py-2 hover:bg-gray-100">📋 قائمة المعاملات</a>
              <?php if (in_array($role, ['manager', 'admin'])): ?>
                <a href="/erp_task_system/tasks/archive_tasks.php" class="block px-4 py-2 hover:bg-gray-100">📦 الأرشيف</a>
              <?php endif; ?>
              <?php if (in_array($role, ['admin'])): ?>
                <a href="/erp_task_system/tasks/archived_tasks.php" class="block px-4 py-2 hover:bg-gray-100">🗑️ المعاملات المحذوفة</a>
              <?php endif; ?>
            </div>
          </div>
        <?php endif; ?>

        <!-- النماذج -->
        <?php if ($role === 'admin'): ?>
          <div class="relative inline-block text-right">
            <button id="formMenu" class="text text-lg hover:text-indigo-700">📄 النماذج </button>
            <div id="formDropdown" class="card hidden absolute right-0 mt-2 w-56 bg-white shadow-lg rounded-md py-2 text-right z-50">
              <a href="/erp_task_system/templates/index.php" class="block px-4 py-2 hover:bg-gray-100">📄 إدارة نماذج المراحل</a>
              <a href="/erp_task_system/custom_forms/manage_portal_forms.php" class="block px-4 py-2 hover:bg-gray-100">🧾 نماذج الإدارات</a>
              <a href="/erp_task_system/custom_forms/portal.php" class="block px-4 py-2 hover:bg-gray-100">📝 بوابة الطلبات</a>
            </div>
          </div>
        <?php endif; ?>

        <!-- الإدارات -->
        <?php if ($role === 'admin'): ?>
          <div class="relative inline-block text-right">
            <button id="adminMenu" class="text text-lg hover:text-indigo-700">🏢 الإدارات </button>
            <div id="adminDropdown" class="card hidden absolute right-0 mt-2 w-56 bg-white shadow-lg rounded-md py-2 text-right z-50">
              <a href="/erp_task_system/departments/index.php" class="block px-4 py-2 hover:bg-gray-100">📁 الإدارات</a>
              <a href="/erp_task_system/tasks/task_types.php" class="block px-4 py-2 hover:bg-gray-100">🏷️ إدارة التصنيفات</a>
              <a href="/erp_task_system/projects/projects.php" class="block px-4 py-2 hover:bg-gray-100">📁 إدارة المشاريع</a>
              <a href="/erp_task_system/users/index.php" class="block px-4 py-2 hover:bg-gray-100">👥 الموارد البشرية</a>
              <a href="/erp_task_system/fields/index.php" class="block px-4 py-2 hover:bg-gray-100">📑 حقول المعاملات</a>
              <a href="/erp_task_system/stages/index.php" class="block px-4 py-2 hover:bg-gray-100">📌 المراحل</a>
            </div>
          </div>
        <?php endif; ?>

        <!-- الربط الخارجي -->
        <?php if ($role === 'admin'): ?>
          <div class="relative inline-block text-right">
            <button id="integrationMenu" class="text text-lg hover:text-indigo-700">🔗 الربط </button>
            <div id="integrationDropdown" class="card hidden absolute right-0 mt-2 w-56 bg-white shadow-lg rounded-md py-2 text-right z-50">
              <a href="/erp_task_system/Integration/settings.php" class="block px-4 py-2 hover:bg-gray-100">⚙️ إعدادات الربط</a>
              <a href="/erp_task_system/Integration/index.php" class="block px-4 py-2 hover:bg-gray-100">🔗 إدارة الربط الخارجي</a>
              <a href="/erp_task_system/Integration/logs.php" class="block px-4 py-2 hover:bg-gray-100">📄 سجل الطلبات</a>
              <a href="/erp_task_system/Integration/webhook_tasks.php" class="block px-4 py-2 hover:bg-gray-100">📥 معاملات Webhook</a>
            </div>
          </div>
        <?php endif; ?>

        <!-- لوحة التحكم -->
        <?php if (in_array($role, ['manager', 'admin'])): ?>
          <a href="/erp_task_system/tasks/dashboard/index.php" class="text-lg hover:text-indigo-700">🏠 لوحة التحكم</a>
        <?php endif; ?>

        <!-- التقارير -->
        <?php if (in_array($role, ['manager', 'admin'])): ?>
          <a href="/erp_task_system/reports/report_builder.php" class="text-lg hover:text-indigo-700">📈 التقارير</a>
        <?php endif; ?>
      </div>

      <!-- المستخدم -->
      <div class="relative inline-block text-left">
        <button id="userMenuButton" class="text text-lg flex items-center  text-gray-800 font-bold py-2 px-4 rounded">
          <?= htmlspecialchars($user['name'] ?? 'مستخدم') ?>
        </button>
        <div id="userDropdown" class="card origin-top-left absolute left-0 mt-2 w-48 rounded-md shadow-lg bg-white ring-1 ring-black ring-opacity-5 hidden z-50 text-right px-2">
          <a href="/erp_task_system/users/profile.php" class="block px-4 py-2 text-sm text-gray-700 hover:bg-gray-100">👤 الملف الشخصي</a>
          <a href="/erp_task_system/users/logout.php" class="block px-4 py-2 text-sm text-red-600 hover:bg-gray-100">🚪 تسجيل خروج</a>
        </div>
      </div>
      <!-- شعار واسم المشروع -->
      <div class="flex items-center space-x-4 rtl:space-x-reverse">
        <a href="/erp_task_system/tasks/dashboard/index.php" class="flex items-center space-x-2 rtl:space-x-reverse">
          <img src="<?php if ($currentUserColorId != 3) echo '/erp_task_system/assets/logo.png';
                    else echo '/erp_task_system/assets/logo-dark.png'; ?> " alt="شعار أسرع" class="h-16 w-auto mt-2">
        </a>
      </div>
    </div>
  </div>
</nav>

<!-- تعويض ارتفاع الناف بار -->
<div class="h-20"></div>

<script>
  const menus = [{
      button: 'taskMenu',
      dropdown: 'taskDropdown'
    },
    {
      button: 'formMenu',
      dropdown: 'formDropdown'
    },
    {
      button: 'adminMenu',
      dropdown: 'adminDropdown'
    },
    {
      button: 'integrationMenu',
      dropdown: 'integrationDropdown'
    },
    {
      button: 'userMenuButton',
      dropdown: 'userDropdown'
    },
  ];

  menus.forEach(currentMenu => {
    const btn = document.getElementById(currentMenu.button);
    const dd = document.getElementById(currentMenu.dropdown);
    if (btn && dd) {
      btn.addEventListener('click', e => {
        e.stopPropagation();
        menus.forEach(menu => {
          const otherDd = document.getElementById(menu.dropdown);
          if (menu.dropdown !== currentMenu.dropdown && otherDd) {
            otherDd.classList.add('hidden');
          }
        });
        dd.classList.toggle('hidden');
      });
      window.addEventListener('click', e => {
        if (!dd.contains(e.target) && !btn.contains(e.target)) {
          dd.classList.add('hidden');
        }
      });
    }
  });

  $(document).ready(function() {
    $('#theme_color_select').select2({
      templateResult: formatColorOption,
      templateSelection: formatColorOption
    });

    function formatColorOption(option) {
      if (!option.id) {
        return option.text;
      }
      var hexCode = $(option.element).data('hex');
      var $option = $(
        '<span><span style="display: inline-block; width: 20px; height: 20px; background-color: ' + hexCode + '; margin-right: 12px; vertical-align: middle; border: 1px solid #ccc;"></span>' + option.text + '</span>'
      );
      return $option;
    }
    $('#theme_color_select').on('select2:select', function(e) {
      $(this).closest('form').submit();
    });
  });
</script>

<style>
  body,
  a:hover {
    background-color: <?php echo $userThemeColor; ?> !important;
  }

  nav,
  .card {
    background-color: <?php echo $userThemeNavColor; ?> !important;
  }

  .text,
  label,
  option,
  h2,
  select,
  a {
    color: <?php echo $userThemeTextColor; ?> !important;
  }

  option {
    text-align: center;
  }

  .secondary-text {
    color: <?php echo $userThemeSecondTextColor; ?> !important;
  }

  .gr {
    color: #7be667;
  }

  .bl {
    color: #4d70f4;
  }

  .view-tasks {
    background-color: none !important;
    /* background-color: rgb(19, 11, 57) !important; */
    border: 3px solid <?php echo $userThemeColor; ?> !important;
    border-radius: 15px !important;
    color:<?= ($currentUserColorId == 3 )?  htmlspecialchars( '#ffffff'): htmlspecialchars('#000000'); ?> !important;

  }

  .text-button {
    color: #ffffff !important;
  }

  .move-button {
    background-color: #4d70f4;
    border-radius: 15px !important;
  }

  .move-button:hover {
    background-color: rgb(19, 11, 57) !important;
  }

  .view-tasks:hover {
    background-color: <?php echo $userThemeColor; ?> !important;
    border: 3px solid rgb(19, 11, 57) !important;
    color:<?= ($currentUserColorId == 3 )?  htmlspecialchars( '#ffffff'): htmlspecialchars('#000000'); ?> !important;

  }

 
</style>

<div class="max-w-6xl mx-auto mt-12 bg-white rounded shadow p-3">
  <h1 class="text-xl font-bold text-center mb-6">📊 تقرير التأخير والغياب</h1>
  <form method="get" class="flex flex-wrap justify-center items-center gap-4 mb-6">
    <label class="font-bold">الشهر:</label>
    <select name="month" class="border rounded px-3 py-2">
      <?php for($m=1;$m<=12;$m++): ?>
        <option value="<?=$m?>" <?=$m==$month?'selected':''?>><?=$m?></option>
      <?php endfor; ?>
    </select>
    <label class="font-bold">السنة:</label>
    <input type="number" name="year" min="2000" max="2100" value="<?=$year?>" class="border rounded px-3 py-2 w-24">
    <button type="submit" class="px-4 py-2 bg-indigo-600 text-white rounded">عرض التقرير</button>
  </form>

  <div class="overflow-x-auto">
    <table class="w-full table-auto border rounded text-sm">
<thead class="bg-gray-100">
  <tr>
    <th class="p-3 border">#</th>
    <th class="p-3 border">الموظف</th>
    <th class="p-3 border">ساعات التأخير</th>
    <th class="p-3 border">أيام الغياب</th>
    <th class="p-3 border">إجمالي الخصم</th>
    <th class="p-3 border">ساعات الإضافي</th>
    <th class="p-3 border">قيمة الإضافي</th>
    <th class="p-3 border">الصافي (خصم - إضافي)</th>
  </tr>
</thead>
      <tbody>
        <?php foreach ($report as $i=>$row): ?>
        <tr class="row-click <?= $i%2?'bg-gray-50':'' ?>" data-id="<?=$row['id']?>" tabindex="0">
<td class="p-2 border"><?= $i+1 ?></td>
<td class="p-2 border font-medium"><?= htmlspecialchars($row['name']) ?></td>
<td class="p-2 border text-red-600 font-bold"><?= number_format($row['lateHours'], 2) ?></td>
<td class="p-2 border text-red-600 font-bold"><?= number_format($row['absentDays'], 1) ?></td>
<td class="p-2 border text-red-600 font-bold"><?= number_format($row['deduction'], 2) ?> ﷼</td>
<td class="p-2 border text-green-600 font-bold"><?= number_format($row['otHours'], 2) ?></td>
<td class="p-2 border text-green-600 font-bold"><?= number_format($row['otAmount'], 2) ?> ﷼</td>
<td class="p-2 border font-bold"><?= number_format($row['net'], 2) ?> ﷼</td>
        </tr>
        <?php endforeach; ?>
        <?php if(empty($report)): ?>
  <tr><td colspan="8" class="text-center p-4 text-gray-500">لا توجد بيانات</td></tr>
<?php else: ?>

<tr class="bg-gray-200 font-bold">
  <td colspan="2" class="p-3 text-center">الإجمالي</td>
  <td class="p-3 text-red-700"><?= number_format($totalLateHours, 2) ?> ساعة</td>
  <td class="p-3 text-red-700"><?= number_format(array_sum(array_column($report,'absentDays')), 1) ?> يوم</td>
  <td class="p-3 text-red-700"><?= number_format($totalDeduction, 2) ?> ﷼</td>
  <td class="p-3 text-green-700"><?= number_format($totalOtHours ?? 0, 2) ?> ساعة</td>
  <td class="p-3 text-green-700"><?= number_format($totalOtAmount ?? 0, 2) ?> ﷼</td>
  <td class="p-3"><?= number_format($totalNet ?? 0, 2) ?> ﷼</td>
</tr>
        <?php endif; ?>
      </tbody>
    </table>
  </div>
</div>

<!-- مودال التفاصيل -->
<div id="employeeDetailsModal" class="fixed inset-0 hidden items-center justify-center bg-black/40 z-50">
  <div class="bg-white p-6 rounded-lg w-full max-w-3xl shadow-xl relative" style="max-height:85vh;overflow:auto">
    <button onclick="document.getElementById('employeeDetailsModal').classList.add('hidden')" class="absolute top-2 left-3 text-2xl font-bold">&times;</button>
    <div id="employeeDetailsContent">...</div>
  </div>
</div>

<script src="https://code.jquery.com/jquery-3.7.1.min.js"></script>
<script>
$(document).on('click', '.row-click', function(){
  const emp_id = $(this).data('id');
  const month = $('select[name=month]').val();
  const year  = $('input[name=year]').val();
  $('#employeeDetailsContent').html('جاري التحميل...');
  $('#employeeDetailsModal').removeClass('hidden').addClass('flex');
  $.get('employee_late_details.php', {id: emp_id, month, year}, function(html){
    $('#employeeDetailsContent').html(html);
  });
});
$(document).on('keydown', '.row-click', function(e){
  if (e.key === 'Enter' || e.key === ' ') { e.preventDefault(); $(this).click(); }
});
</script>
</body>
</html>
