<?php
function getSickYearStartFor(PDO $conn, int $user_id, string $monthEndYmd): string {
  $q = $conn->prepare("
    SELECT COALESCE(MIN(CASE WHEN medical_year_start IS NOT NULL THEN medical_year_start END),
                    MIN(start_date)) AS year_start
    FROM leave_requests
    WHERE user_id=? AND (leave_type_code='sick' OR type IN ('إجازة مرضية','مرضية'))
      AND status='approved'
      AND start_date BETWEEN DATE_SUB(?, INTERVAL 365 DAY) AND ?
  ");
  $q->execute([$user_id, $monthEndYmd, $monthEndYmd]);
  $ys = $q->fetchColumn();
  return $ys ?: $monthEndYmd;
}

/** يرجّع كم يوم مرضي داخل الشهر (مقسّم: full / p75 / unpaid) */
function getSickImpactMonthly(PDO $conn, int $user_id, string $monthStartYmd, string $monthEndYmd): array {
  $yearStart = getSickYearStartFor($conn, $user_id, $monthEndYmd);

  // كل طلبات المرضية المعتمدة من بداية السنة المرضية حتى نهاية الشهر
  $st = $conn->prepare("
    SELECT start_date, end_date
    FROM leave_requests
    WHERE user_id=? AND (leave_type_code='sick' OR type IN ('إجازة مرضية','مرضية'))
      AND status='approved'
      AND NOT (end_date < ? OR start_date > ?)
    ORDER BY start_date ASC
  ");
  $st->execute([$user_id, $yearStart, $monthEndYmd]);
  $ranges = $st->fetchAll(PDO::FETCH_ASSOC);

  // نبني خريطة أيام مرضية من بداية السنة المرضية إلى نهاية الشهر (<= 366 يوم)
  $sickDays = [];
  foreach ($ranges as $r) {
    $from = max($yearStart, $r['start_date']);
    $to   = min($monthEndYmd, $r['end_date']);
    $d = new DateTime($from);
    $toD = new DateTime($to);
    while ($d <= $toD) {
      $sickDays[$d->format('Y-m-d')] = true;
      $d->modify('+1 day');
    }
  }

  // مر على كل يوم من السنة المرضية واحسب ترتيبه المرضي (عشان نحدد الشريحة)
  $full=$p75=$unpaid=0;
  $ordinal = 0;
  $cursor = new DateTime($yearStart);
  $end    = new DateTime($monthEndYmd);
  while ($cursor <= $end) {
    $dYmd = $cursor->format('Y-m-d');
    if (!empty($sickDays[$dYmd])) {
      $ordinal++;
      $inMonth = ($dYmd >= $monthStartYmd && $dYmd <= $monthEndYmd);
      if ($ordinal <= 30) { if ($inMonth) $full++; }
      elseif ($ordinal <= 90) { if ($inMonth) $p75++; }
      else { if ($inMonth) $unpaid++; } // كل ما بعد 90 إلى 120 -> unpaid، وبعدها برضه unpaid
    }
    $cursor->modify('+1 day');
  }
  return ['full'=>$full, 'p75'=>$p75, 'unpaid'=>$unpaid];
}

/** يجيب الراتب الأساسي من بنود الراتب فقط (بدون رجوع لجدول users) */
function getBasicFromComponents(PDO $conn, int $user_id): float {
  $q = $conn->prepare("
    SELECT esi.amount
    FROM employee_salary_items esi
    JOIN salary_components sc ON sc.id = esi.component_id
    WHERE esi.user_id=? AND (sc.type='basic' OR sc.code='basic')
    ORDER BY esi.id DESC
    LIMIT 1
  ");
  $q->execute([$user_id]);
  return (float)($q->fetchColumn() ?: 0);
}
