<?php
session_start();
require_once '../../config/db.php';
require_once '../auth.php';
ini_set('display_errors',1);
ini_set('display_startup_errors',1);
error_reporting(E_ALL);

// check login
if (!isset($_SESSION['user'])) {
    http_response_code(403);
    exit('Not authorized');
}

// Parameters
$payroll_id = (int)($_GET['payroll_id'] ?? 0);
$month = (int)($_GET['month'] ?? date('n'));
$year = (int)($_GET['year'] ?? date('Y'));
$doSnapshot = (int)($_GET['snapshot'] ?? 0);

if ($payroll_id <= 0) {
    exit('payroll_id required');
}

// get setting
function setting(PDO $c, $key, $def) {
    $stmt = $c->prepare("SELECT setting_value FROM payroll_settings WHERE setting_key = ? LIMIT 1");
    $stmt->execute([$key]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);
    if ($row) {
        $val = $row['setting_value'];
        return is_numeric($val) ? (float)$val : $val;
    }
    return $def;
}

// Helpers
function get_base_component_id(PDO $conn): int {
    $stmt = $conn->query("SELECT id FROM salary_components WHERE type='basic' LIMIT 1");
    $row = $stmt->fetch(PDO::FETCH_ASSOC);
    return $row ? (int)$row['id'] : 0;
}
function get_base_salary(PDO $conn, int $emp_id, int $base_cid): float {
    $stmt = $conn->prepare("SELECT value FROM employee_salary_items WHERE employee_id=? AND component_id=? LIMIT 1");
    $stmt->execute([$emp_id, $base_cid]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);
    return (float)($row['value'] ?? 0);
}
function month_bounds(int $month, int $year): array {
    $num = cal_days_in_month(CAL_GREGORIAN, $month, $year);
    $Y=(int)date('Y');
    $M=(int)date('n');
    $D=(int)date('j');
    if ($year > $Y || ($year == $Y && $month > $M)) return [0, 0, true];
    $last = $num;
    if ($year == $Y && $month == $M) $last = max(0, $D - 1);
    return [$num, $last, false];
}
function get_work_weekdays(PDO $conn, int $emp_id): array {
    $stmt = $conn->prepare("SELECT department_id, day_off FROM users WHERE id=?");
    $stmt->execute([$emp_id]);
    $emp = $stmt->fetch(PDO::FETCH_ASSOC);
    $dept = (int)($emp['department_id'] ?? 0);
    $day_off = $emp['day_off'] ?? null;
    $stmt = $conn->prepare("SELECT day_of_week FROM department_work_days WHERE department_id=? AND is_working=1");
    $stmt->execute([$dept]);
    $names = $stmt->fetchAll(PDO::FETCH_COLUMN);
    $map = ['monday'=>1,'tuesday'=>2,'wednesday'=>3,'thursday'=>4,'friday'=>5,'saturday'=>6,'sunday'=>7];
    $wd=[];
    foreach ($names as $n) {
        $l = strtolower(trim($n));
        if (isset($map[$l])) $wd[] = $map[$l];
    }
    if ($day_off !== null && $day_off !== '') {
        $wd = array_values(array_diff($wd, [(int)$day_off]));
    }
    return $wd;
}
function get_leave_dates(PDO $conn, int $emp_id, int $month, int $year, int $last): array {
    if ($last <= 0) return [];
    $stmt = $conn->prepare("SELECT start_date,end_date FROM leave_requests
        WHERE user_id=? AND status='مقبولة' AND (
          (MONTH(start_date)=? AND YEAR(start_date)=?) OR
          (MONTH(end_date)=? AND YEAR(end_date)=?) OR
          (start_date<=LAST_DAY(CONCAT(?, '-', ?, '-01')) AND end_date>=CONCAT(?, '-', ?, '-01'))
        )");
    $stmt->execute([$emp_id,$month,$year,$month,$year,$year,$month,$year,$month]);
    $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
    $from = new DateTime(sprintf('%04d-%02d-01',$year,$month));
    $to   = new DateTime(sprintf('%04d-%02d-%02d',$year,$month,$last));
    $dates=[];
    foreach ($rows as $lv) {
        try {
            $s = new DateTime($lv['start_date']);
            $e = new DateTime($lv['end_date']);
        } catch (Exception $e) {
            continue;
        }
        if ($s < $from) $s = clone $from;
        if ($e > $to)   $e = clone $to;
        while ($s <= $e) {
            $dates[] = $s->format('Y-m-d');
            $s->modify('+1 day');
        }
    }
    return $dates;
}
function get_shift_times(PDO $conn, int $emp_id): array {
    $stmt = $conn->prepare("SELECT ws.start_time, ws.end_time
        FROM users u
        LEFT JOIN work_shifts ws ON ws.id = u.work_shift_id
        WHERE u.id = ?");
    $stmt->execute([$emp_id]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC) ?: [];
    $start = ($row['start_time'] ?? '08:00:00') ?: '08:00:00';
    $end   = ($row['end_time'] ?? '16:00:00') ?: '16:00:00';
    return [$start, $end];
}
function get_month_attendance(PDO $conn, int $emp_id, int $month, int $year): array {
    $stmt = $conn->prepare("SELECT DATE(`date`) AS day,
               MIN(TIME(`check_in`)) AS in_time,
               MAX(TIME(`check_out`)) AS out_time,
               MAX(status) AS status
        FROM attendance
        WHERE user_id=? AND MONTH(`date`)=? AND YEAR(`date`)=?
        GROUP BY DATE(`date`)
        ORDER BY day");
    $stmt->execute([$emp_id,$month,$year]);
    $out=[];
    while ($r = $stmt->fetch(PDO::FETCH_ASSOC)) {
        $in  = $r['in_time'];
        $out_time = $r['out_time'];
        $worked = 0;
        if ($in && $out_time) {
            $worked = max(0, (int)round((strtotime($out_time) - strtotime($in))/60));
        }
        $r['worked_minutes'] = $worked;
        $out[$r['day']] = $r;
    }
    return $out;
}
function get_employee_absent_days(PDO $conn, int $emp_id, int $month, int $year, int $halfDayMinutes = 240): float {
    [$num, $last, $future] = month_bounds($month, $year);
    if ($future || $last <= 0) return 0.0;
    $wd = get_work_weekdays($conn, $emp_id);
    $att = get_month_attendance($conn, $emp_id, $month, $year);
    $leave = get_leave_dates($conn, $emp_id, $month, $year, $last);
    $abs = 0.0;
    for ($d = 1; $d <= $last; $d++) {
        $date = sprintf('%04d-%02d-%02d',$year,$month,$d);
        $dow = (int)date('N',strtotime($date));
        if (!in_array($dow,$wd,true)) continue;
        if (in_array($date,$leave,true)) continue;
        $row = $att[$date] ?? null;
        if (!$row) { $abs += 1.0; continue; }
        $worked = (int)($row['worked_minutes'] ?? 0);
        if ($worked < $halfDayMinutes) $abs += 0.5;
    }
    return $abs;
}

// calculate attendance summary for employee
function calcAttendanceSummary(PDO $conn, int $emp_id, int $month, int $year): array {
    $halfDayMinutes = (int)setting($conn, 'half_day_minutes', 240);
    $workDaysMonth  = (int)setting($conn, 'work_days_month', 30);
    $workHoursDay   = (int)setting($conn, 'work_hours_day', 8);
    $rate_working   = (float)setting($conn, 'ot_rate_working', 1.5);
    $rate_offday    = (float)setting($conn, 'ot_rate_offday', 2.0);
    [$num,$last,$future] = month_bounds($month, $year);
    if ($future || $last <= 0) {
        return [0,0,0,0,0,0,0,0];
    }
    $base_cid = get_base_component_id($conn);
    $baseSalary = get_base_salary($conn,$emp_id,$base_cid);
    $salaryDay = $workDaysMonth ? ($baseSalary / $workDaysMonth) : 0;
    $dedPerAbs = $salaryDay * 1.5;
    $dedPerHour = $workHoursDay ? ($dedPerAbs / $workHoursDay) : 0;
    $hourly = ($workDaysMonth && $workHoursDay) ? $baseSalary / ($workDaysMonth * $workHoursDay) : 0;
    [$shift_start, $shift_end] = get_shift_times($conn, $emp_id);
    $weekdays = get_work_weekdays($conn, $emp_id);
    $att = get_month_attendance($conn, $emp_id, $month, $year);
    $leaves = get_leave_dates($conn, $emp_id, $month, $year, $last);
    $total_late_minutes = 0;
    $total_late_deduction = 0;
    $ot_work_mins = 0;
    $ot_off_mins = 0;
    $ot_work_amount = 0.0;
    $ot_off_amount = 0.0;
    for ($d = 1; $d <= $last; $d++) {
        $date = sprintf('%04d-%02d-%02d',$year,$month,$d);
        $dow = (int)date('N',strtotime($date));
        $row = $att[$date] ?? null;
        $in = $row['in_time'] ?? null;
        $out_time = $row['out_time'] ?? null;
        $worked = (int)($row['worked_minutes'] ?? 0);
        // non working day
        if (!in_array($dow, $weekdays, true)) {
            if ($row && $worked > 0) {
                $ot_mins = $worked;
                $ot_amount = ($hourly * $rate_offday) * ($ot_mins / 60);
                $ot_off_mins += $ot_mins;
                $ot_off_amount += $ot_amount;
            }
            continue;
        }
        // leave
        if (in_array($date, $leaves, true)) {
            continue;
        }
        // absent entire
        if (!$row) {
            continue;
        }
        // half day
        if ($worked < $halfDayMinutes) {
            continue;
        }
        // late
        if ($in && $in > $shift_start && in_array(($row['status'] ?? ''), ['حاضر','متأخر'], true)) {
            $diff = strtotime($in) - strtotime($shift_start);
            if ($diff > 0) {
                $mins = (int)round($diff / 60);
                $total_late_minutes += $mins;
                $total_late_deduction += $dedPerHour * ($mins / 60);
            }
        }
        // OT on working day
        if ($out_time && strtotime($out_time) > strtotime($shift_end)) {
            $after_end = (int)round((strtotime($out_time) - strtotime($shift_end)) / 60);
            $cap = max(0, $worked - ($workHoursDay * 60));
            $ot_mins = max(0, min($after_end, $cap));
            if ($ot_mins > 0) {
                $ot_work_mins += $ot_mins;
                $ot_work_amount += ($hourly * $rate_working) * ($ot_mins / 60);
            }
        }
    }
    $absentDays = get_employee_absent_days($conn, $emp_id, $month, $year, $halfDayMinutes);
    $absenceDeduction = $absentDays * $dedPerAbs;
    return [
        $total_late_minutes,
        round($total_late_deduction, 2),
        $absentDays,
        round($absenceDeduction, 2),
        $ot_work_mins,
        $ot_off_mins,
        round($ot_work_amount, 2),
        round($ot_off_amount, 2)
    ];
}

// Helper to get component id by code
function compId(PDO $c, $code) {
    $stmt = $c->prepare("SELECT id FROM salary_components WHERE code = ? LIMIT 1");
    $stmt->execute([$code]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);
    return $row ? (int)$row['id'] : 0;
}

// component IDs
$CID_LATE = compId($conn, 'LATE_DED');
$CID_ABS  = compId($conn, 'ABSENCE_DED');
$CID_OTW  = compId($conn, 'OT_WORK');
$CID_OTO  = compId($conn, 'OT_OFF');

// get payroll items
$q = $conn->prepare("SELECT id AS payroll_item_id, user_id FROM payroll_items WHERE payroll_id = ?");
$q->execute([$payroll_id]);
$items = $q->fetchAll(PDO::FETCH_ASSOC);
if (!$items) {
    exit('لا يوجد عناصر في هذا المسير');
}

// prepare deletion and insertion
$del = $conn->prepare("DELETE pic FROM payroll_item_components pic JOIN salary_components sc ON sc.id = pic.component_id WHERE pic.payroll_item_id = ? AND sc.code IN ('LATE_DED','ABSENCE_DED','OT_WORK','OT_OFF')");
$ins = $conn->prepare("INSERT INTO payroll_item_components (payroll_item_id, component_id, amount) VALUES (?, ?, ?)");

$count = 0;
foreach ($items as $it) {
    $emp_id = (int)$it['user_id'];
    [$lateMins, $lateDed, $absDays, $absDed, $otWorkMins, $otOffMins, $otWorkAmt, $otOffAmt] = calcAttendanceSummary($conn, $emp_id, $month, $year);
    // delete existing
    $del->execute([$it['payroll_item_id']]);
    // insert new values
    if ($CID_LATE) $ins->execute([$it['payroll_item_id'], $CID_LATE, $lateDed]);
    if ($CID_ABS)  $ins->execute([$it['payroll_item_id'], $CID_ABS,  $absDed]);
    if ($CID_OTW)  $ins->execute([$it['payroll_item_id'], $CID_OTW,  $otWorkAmt]);
    if ($CID_OTO)  $ins->execute([$it['payroll_item_id'], $CID_OTO,  $otOffAmt]);
    // update absent_days
    $u = $conn->prepare("UPDATE payroll_items SET absent_days = ? WHERE id = ?");
    $u->execute([$absDays, $it['payroll_item_id']]);
    $count++;
}

// update totals and net in payroll_items
$agg = $conn->prepare("UPDATE payroll_items pi
    JOIN (
        SELECT pc.payroll_item_id,
               SUM(CASE WHEN sc.type='basic' THEN pc.amount ELSE 0 END) AS base_salary,
               SUM(CASE WHEN sc.type IN ('allowance','earning') THEN pc.amount ELSE 0 END) AS total_allowances,
               SUM(CASE WHEN sc.type='deduction' THEN pc.amount ELSE 0 END) AS total_deductions,
               SUM(CASE WHEN sc.type='overtime' THEN pc.amount ELSE 0 END) AS overtime_total
        FROM payroll_item_components pc
        JOIN salary_components sc ON sc.id = pc.component_id
        JOIN payroll_items pi2 ON pi2.id = pc.payroll_item_id
        WHERE pi2.payroll_id = ?
        GROUP BY pc.payroll_item_id
    ) x ON x.payroll_item_id = pi.id
    SET
      pi.base_salary = x.base_salary,
      pi.total_allowances = x.total_allowances,
      pi.total_deductions = x.total_deductions,
      pi.overtime_total = x.overtime_total,
      pi.net_salary = x.base_salary + x.total_allowances + x.overtime_total - x.total_deductions");
$agg->execute([$payroll_id]);

// snapshot bank details if requested
if ($doSnapshot) {
    $snap = $conn->prepare("UPDATE payroll_items pi
        JOIN users u ON u.id = pi.user_id
        SET pi.snapshot_name = u.name,
            pi.snapshot_employee_number = u.employee_number,
            pi.snapshot_national_id = u.national_id,
            pi.snapshot_bank_account = u.bank_account,
            pi.snapshot_iban = u.iban,
            pi.snapshot_currency = COALESCE(u.salary_currency,'AED')
        WHERE pi.payroll_id = ?");
    $snap->execute([$payroll_id]);
}

// output JSON
header('Content-Type: application/json; charset=utf-8');
echo json_encode([
    'ok' => true,
    'updated_items' => $count,
    'payroll_id' => $payroll_id,
    'month' => $month,
    'year' => $year,
    'snapshot' => $doSnapshot
]);
