<?php
// /erp_task_system/users/settlements/_helpers.php
if (session_status() === PHP_SESSION_NONE) session_start();

require_once __DIR__ . '/../../config/db.php';
$__authPath = __DIR__ . '/../../authorize.php';
if (is_file($__authPath)) require_once $__authPath;

/* ====== صلاحيات ومعلومات أساسية ====== */
function current_user_id(): int {
  return isset($_SESSION['user']['id']) ? (int)$_SESSION['user']['id'] : 0;
}
function require_roles(array $roles) {
  if (!isset($_SESSION['user'])) { http_response_code(401); exit('login'); }
  $role = $_SESSION['user']['role'] ?? '';
  if (!in_array($role, $roles, true)) { http_response_code(403); exit('forbidden'); }
}

/* ====== أدوات ذكيّة تتأكد من الأعمدة قبل الاستخدام ====== */
function db_has_col(PDO $conn, string $table, string $col): bool {
  static $cache = [];
  $key = $table.'|'.$col;
  if (isset($cache[$key])) return $cache[$key];
  $st = $conn->prepare("SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS
                         WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ? AND COLUMN_NAME = ?");
  $st->execute([$table, $col]);
  return $cache[$key] = ((int)$st->fetchColumn() > 0);
}

/** يبني تعبير COALESCE للمبلغ حسب الأعمدة الموجودة فعليًا بالجدول */
function build_amount_expr(PDO $conn, string $table, string $alias, array $candidates): string {
  $parts = [];
  foreach ($candidates as $c) if (db_has_col($conn, $table, $c)) $parts[] = "$alias.`$c`";
  return $parts ? 'COALESCE('.implode(',', $parts).')' : 'NULL';
}

/** يبني شرط where يدعم user_id و/أو employee_id حسب الموجود */
function build_id_where(PDO $conn, string $table, string $alias, int $eid): array {
  $conds = []; $params = [];
  foreach (['user_id','employee_id'] as $c) {
    if (db_has_col($conn, $table, $c)) { $conds[] = "$alias.$c = ?"; $params[] = $eid; }
  }
  $where = $conds ? '(' . implode(' OR ', $conds) . ')' : '1=0';
  return [$where, $params];
}

/** شرط مطابقة بند الأساسي (type/code وربما is_basic إذا موجود) */
function basic_predicate(PDO $conn, string $scAlias = 'sc'): string {
  $preds = [
    "LOWER(TRIM($scAlias.type)) IN ('basic','أساسي','اساسي','basic_salary','salary_basic')",
    "UPPER(REPLACE(TRIM($scAlias.code),'-','')) IN ('BASIC','BASICSALARY','BASICPAY','ASASI','ASASY')"
  ];
  if (db_has_col($conn, 'salary_components', 'is_basic')) {
    $preds[] = "IFNULL($scAlias.is_basic,0)=1";
  }
  return '(' . implode(' OR ', $preds) . ')';
}

/* ====== دوال المخالصات ====== */
function get_settlement_row(PDO $conn, int $sid): array {
  $st = $conn->prepare("SELECT * FROM settlements WHERE id=?");
  $st->execute([$sid]);
  $row = $st->fetch(PDO::FETCH_ASSOC);
  if (!$row) throw new Exception('not_found');
  return $row;
}

function assert_editable(array $settlement) {
  if (($settlement['status'] ?? '') !== 'draft') throw new Exception('locked'); // يمنع التعديل
}

function recalc_settlement_totals(PDO $conn, int $sid): void {
  $sql = "SELECT
            SUM(CASE WHEN item_type='addition'  THEN amount ELSE 0 END) AS adds,
            SUM(CASE WHEN item_type='deduction' THEN amount ELSE 0 END) AS deds
          FROM settlement_items WHERE settlement_id = ?";
  $st = $conn->prepare($sql);
  $st->execute([$sid]);
  $row  = $st->fetch(PDO::FETCH_ASSOC) ?: ['adds'=>0,'deds'=>0];
  $adds = (float)($row['adds'] ?? 0);
  $deds = (float)($row['deds'] ?? 0);
  $net  = $adds - $deds;

  $up = $conn->prepare("UPDATE settlements
                        SET total_additions=?, total_deductions=?, net_amount=?, updated_at=NOW()
                        WHERE id=?");
  $up->execute([$adds, $deds, $net, $sid]);
}

/* ====== جلب الراتب الأساسي بذكاء من البنود أو من آخر مسير ====== */
function get_basic_salary_info(PDO $conn, int $employee_id): array {
  // 1) من بنود الموظف الثابتة employee_salary_items
  $amtExpr = build_amount_expr($conn, 'employee_salary_items', 'esi', ['value','amount','component_value']);
  [$idWhere, $idParams] = build_id_where($conn, 'employee_salary_items', 'esi', $employee_id);
  $pred = basic_predicate($conn, 'sc');

  $sql1 = "SELECT $amtExpr AS amt
           FROM employee_salary_items esi
           JOIN salary_components sc ON sc.id = esi.component_id
           WHERE $idWhere AND $pred
           ORDER BY esi.id DESC
           LIMIT 1";
  $st1 = $conn->prepare($sql1);
  $st1->execute($idParams);
  $v1 = $st1->fetchColumn();
  if ($v1 !== false && $v1 !== null && (float)$v1 > 0) {
    return ['amount' => (float)$v1, 'source' => 'emp_items'];
  }

  // 2) آخر مسير رواتب payroll_item_components
  $amtExpr2 = build_amount_expr($conn, 'payroll_item_components', 'pic', ['amount','value','component_value']);
  [$piWhere, $piParams] = build_id_where($conn, 'payroll_items', 'pi', $employee_id);

  $sql2 = "SELECT $amtExpr2 AS amt
           FROM payroll_item_components pic
           JOIN salary_components sc ON sc.id = pic.component_id
           JOIN payroll_items pi ON pi.id = pic.payroll_item_id
           JOIN payrolls p ON p.id = pi.payroll_id
           WHERE $piWhere AND $pred
           ORDER BY p.id DESC, pic.id DESC
           LIMIT 1";
  $st2 = $conn->prepare($sql2);
  $st2->execute($piParams);
  $v2 = $st2->fetchColumn();
  if ($v2 !== false && $v2 !== null && (float)$v2 > 0) {
    return ['amount' => (float)$v2, 'source' => 'last_payroll'];
  }

  return ['amount'=>null,'source'=>'none'];
}

function get_basic_salary(PDO $conn, int $employee_id): ?float {
  $info = get_basic_salary_info($conn, $employee_id);
  return $info['amount'];
}

/* ====== وظائف مساعدة للحسابات ====== */
function years_of_service(string $hire_date, string $last_day): float {
  $d1 = new DateTime($hire_date);
  $d2 = new DateTime($last_day);
  if ($d2 < $d1) return 0.0;
  $days = (int)$d1->diff($d2)->format('%a'); // أيام كاملة
  return $days / 365.0;
}

/**
 * مكافأة نهاية الخدمة (صيغة شائعة بالسعودية):
 * أول 5 سنوات: نصف راتب/سنة، بعدها راتب كامل/سنة.
 * الاستقالة: <2 سنوات 0%، 2-5 ثلث، 5-10 ثلثين، 10+ كامل.
 */
function calc_eos_reward(float $monthly_wage, float $years, string $reason): float {
  if ($monthly_wage <= 0 || $years <= 0) return 0.0;

  $first5 = min($years, 5.0);
  $rest   = max(0.0, $years - 5.0);
  $base   = ($first5 * 0.5 * $monthly_wage) + ($rest * 1.0 * $monthly_wage);

  $r = strtolower($reason);
  if (in_array($r, ['resignation','استقالة','istiqala'], true)) {
    if ($years < 2.0)       return 0.0;
    elseif ($years < 5.0)   return $base * (1.0/3.0);
    elseif ($years < 10.0)  return $base * (2.0/3.0);
    else                    return $base;
  }
  return $base; // إنهاء من جهة العمل
}

/* ====== إدخال بند تلقائي في settlement_items ====== */
function insert_auto_item(PDO $conn, int $sid, string $type, string $code, string $label, float $amount, array $calc = []): void {
  $uid = current_user_id();
  $calcJson = $calc ? json_encode($calc, JSON_UNESCAPED_UNICODE) : null;
  $ins = $conn->prepare("INSERT INTO settlement_items
    (settlement_id, item_type, item_code, label, amount, source, calc_json, created_by, created_at)
    VALUES (?,?,?,?,?,'auto',?,?,NOW())");
  $ins->execute([$sid, $type, $code, $label, $amount, $calcJson, $uid ?: null]);
}
