<?php
/* /erp_task_system/users/social_insurance/insurance_import.php */
session_start();
require_once __DIR__ . '/../../config/db.php';

ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

/* صلاحيات */
if (!isset($_SESSION['user']) || !in_array($_SESSION['user']['role'] ?? '', ['admin','manager'])) {
  header('Location: ../../unauthorized.php'); exit;
}

/* تحميل تلقائي لـ PhpSpreadsheet عند الحاجة للـ Excel */
spl_autoload_register(function ($class) {
  $prefix = 'PhpOffice\\PhpSpreadsheet\\';
  if (strncmp($prefix, $class, strlen($prefix)) !== 0) return;
  $base_dir = __DIR__ . '/../../vendor/phpoffice/phpspreadsheet/src/PhpSpreadsheet/';
  $file = $base_dir . str_replace('\\', '/', substr($class, strlen($prefix))) . '.php';
  if (file_exists($file)) require $file;
});

/* خرائط الرؤوس (أضفنا identifier + أعمدة الأجور) */
$COLUMNS_MAP = [
  'national_id'           => ['رقم الهوية','national_id','iqama','id','identifier'],
  'gosi_number'           => ['رقم المشترك','gosi_number','subscriber_no','رقم التأمينات'],
  'salary'                => ['الأجر المسجل','الراتب','salary','wage'],
  'basic_wage'            => ['basic wage','basic_wage','basic'],
  'housing'               => ['housing','بدل سكن','السكن'],
  'commission'            => ['commission','عمولة','عمولات'],
  'other_allowance'       => ['other allowance','other_allowance','بدلات اخرى','بدل آخر'],
  'subscription_date'     => ['بداية الاشتراك','subscription_start','subscription_date','start_date'],
  'end_date'              => ['نهاية الاشتراك','end_date','stop_date'],
  'contribution_employee' => ['حصة الموظف','employee_contribution','emp_share'],
  'contribution_employer' => ['حصة المنشأة','employer_contribution','org_share'],
  'status'                => ['الحالة','status','subscription_status'],
];

/* أدوات */
function norm($s){ return mb_strtolower(trim((string)$s),'UTF-8'); }
function findCol($headers, $aliases) {
  foreach ($headers as $i=>$h) {
    if (in_array(norm($h), array_map('norm', $aliases), true)) return $i;
  }
  return null;
}
function toNum($v){
  $v = str_replace([","," "], ["",""], trim((string)$v));
  return ($v!=='' && is_numeric($v)) ? (float)$v : 0.0;
}
function toMoney($v){ return $v===null? null : number_format((float)$v, 2, '.', ''); }
function toDate($v){
  $v = trim((string)$v);
  if ($v==='' || $v==='0000-00-00') return null;
  foreach ([$v, str_replace('/','-',$v)] as $t) {
    $ts = strtotime($t); if ($ts && $ts>0) return date('Y-m-d',$ts);
  }
  return null;
}
function toStatus($v){
  $v = norm($v);
  if (in_array($v,['نشط','active'])) return 'active';
  if (in_array($v,['موقوف','متوقف','stopped'])) return 'stopped';
  if ($v==='معلق' || $v==='suspended') return 'suspended';
  return 'active';
}
function employeeIdByNational(PDO $conn, $national_id){
  if (!$national_id) return null;
  $q = $conn->prepare("SELECT id FROM users WHERE national_id = ? LIMIT 1");
  $q->execute([$national_id]);
  $id = $q->fetchColumn();
  return $id ? (int)$id : null;
}

/* نتيجة */
$done = ['inserted'=>0,'updated'=>0,'skipped'=>0,'errors'=>0,'notes'=>[]];

/* سياسة احتساب الراتب إذا ما كان فيه عمود salary:
   bh = basic + housing (المعمول به غالبًا بالتأمينات)
   bhco = basic + housing + commission + other_allowance  */
$SUM_POLICY = $_POST['sum_policy'] ?? 'bh';

if ($_SERVER['REQUEST_METHOD']==='POST' && isset($_FILES['file'])) {
  $file = $_FILES['file'];
  if ($file['error'] !== UPLOAD_ERR_OK) {
    $done['notes'][] = 'فشل رفع الملف.'; 
  } else {
    $tmp = $file['tmp_name'];
    $ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
    $headers = []; $rows = [];

    try {
      if ($ext==='csv') {
        /* قراءة CSV مع كشف الفاصل */
        $contents = file_get_contents($tmp);
        if ($contents===false) throw new Exception('تعذر قراءة CSV');
        $contents = preg_replace('/^\xEF\xBB\xBF/', '', $contents); // شيل BOM
        $lines = preg_split("/\r\n|\n|\r/", trim($contents));

        if (!$lines || count($lines)<2) throw new Exception('CSV فارغ');
        $first = $lines[0];

        $cands = [",",";","\t","|"];
        $best=","; $bestC=-1;
        foreach($cands as $d){ $count=substr_count($first,$d); if($count>$bestC){$best=$d;$bestC=$count;} }
        $headers = str_getcsv($first, $best);
        for ($i=1;$i<count($lines);$i++){
          if (trim($lines[$i])==='') continue;
          $rows[] = str_getcsv($lines[$i], $best);
        }
      } else {
        /* Excel */
        if (!class_exists('\\PhpOffice\\PhpSpreadsheet\\IOFactory')) {
          throw new Exception('PhpSpreadsheet غير متوفر لقراءة Excel. استورد CSV أو فعّل المكتبة.');
        }
        $sheet = \PhpOffice\PhpSpreadsheet\IOFactory::load($tmp)->getActiveSheet();
        $highestCol = $sheet->getHighestColumn();
        $highestRow = $sheet->getHighestRow();
        foreach ($sheet->rangeToArray("A1:{$highestCol}1", null, true, true, true) as $row) {
          $headers = array_values($row);
        }
        for ($r=2; $r<=$highestRow; $r++) {
          $rows[] = array_values($sheet->rangeToArray("A{$r}:{$highestCol}{$r}", null, true, true, true)[$r]);
        }
      }

      /* خرائط الفهارس */
      $idx = [];
      foreach ($COLUMNS_MAP as $key=>$aliases) $idx[$key]=findCol($headers,$aliases);

      if ($idx['national_id']===null) {
        throw new Exception('لازم عمود national_id (أو IDENTIFIER).');
      }

      $sql = "
        INSERT INTO social_insurance
          (employee_id, national_id, gosi_number, salary, subscription_date, end_date,
           contribution_employee, contribution_employer, status, last_updated)
        VALUES
          (:employee_id, :national_id, :gosi_number, :salary, :subscription_date, :end_date,
           :contribution_employee, :contribution_employer, :status, NOW())
        ON DUPLICATE KEY UPDATE
          employee_id = VALUES(employee_id),
          gosi_number = VALUES(gosi_number),
          salary = VALUES(salary),
          subscription_date = VALUES(subscription_date),
          end_date = VALUES(end_date),
          contribution_employee = VALUES(contribution_employee),
          contribution_employer = VALUES(contribution_employer),
          status = VALUES(status),
          last_updated = NOW()
      ";
      $up = $conn->prepare($sql);

      foreach ($rows as $i=>$r) {
        try {
          $national_id = preg_replace('/\D+/', '', (string)$r[$idx['national_id']]); // رقمي بس
          if ($national_id===''){ $done['skipped']++; continue; }

          $employee_id = employeeIdByNational($conn, $national_id);
          if (!$employee_id){ $done['skipped']++; $done['notes'][]="سطر ".($i+2).": ما لقيت موظف بهوية ($national_id)."; continue; }

          $gosi_number = $idx['gosi_number']!==null ? trim((string)$r[$idx['gosi_number']]) : null;

          // راتب: إذا ما فيه salary نحسبه
          if ($idx['salary']!==null) {
            $salary = toMoney(toNum($r[$idx['salary']]));
          } else {
            $basic = $idx['basic_wage']!==null ? toNum($r[$idx['basic_wage']]) : 0;
            $house = $idx['housing']   !==null ? toNum($r[$idx['housing']])    : 0;
            $comm  = $idx['commission']!==null ? toNum($r[$idx['commission']]) : 0;
            $other = $idx['other_allowance']!==null ? toNum($r[$idx['other_allowance']]) : 0;
            $calc  = ($SUM_POLICY==='bhco') ? ($basic+$house+$comm+$other) : ($basic+$house);
            $salary = toMoney($calc);
          }

          $sub_date  = $idx['subscription_date']!==null ? toDate($r[$idx['subscription_date']]) : null;
          $end_date  = $idx['end_date']         !==null ? toDate($r[$idx['end_date']])          : null;
          $c_emp     = $idx['contribution_employee']!==null ? toMoney(toNum($r[$idx['contribution_employee']])) : null;
          $c_org     = $idx['contribution_employer']!==null ? toMoney(toNum($r[$idx['contribution_employer']])) : null;
          $status    = $idx['status']           !==null ? toStatus($r[$idx['status']]) : 'active';

          $up->execute([
            ':employee_id' => $employee_id,
            ':national_id' => $national_id,
            ':gosi_number' => $gosi_number,
            ':salary' => $salary,
            ':subscription_date' => $sub_date,
            ':end_date' => $end_date,
            ':contribution_employee' => $c_emp,
            ':contribution_employer' => $c_org,
            ':status' => $status,
          ]);
          $rc = $up->rowCount();
          $done['inserted'] += ($rc===1)?1:0;
          $done['updated']  += ($rc===2)?1:0;

        } catch (Throwable $eRow) {
          $done['errors']++; $done['notes'][]="سطر ".($i+2).": ".$eRow->getMessage();
        }
      }

    } catch (Throwable $e) {
      $done['errors']++; $done['notes'][] = "خطأ أثناء القراءة: ".$e->getMessage();
    }
  }
}
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
  <meta charset="UTF-8">
  <title>استيراد التأمينات الاجتماعية</title>
  <script src="https://cdn.tailwindcss.com"></script>
</head>
<body class="bg-gray-50 p-6">
  <div class="max-w-2xl mx-auto bg-white rounded-xl shadow p-6">
    <h1 class="text-xl font-bold mb-4">🧾 استيراد التأمينات الاجتماعية</h1>
    <form method="post" enctype="multipart/form-data" class="space-y-4">
      <input type="file" name="file" accept=".csv,.xls,.xlsx" required class="block w-full border rounded p-2">
      <div class="text-sm">
        <label class="mr-2">سياسة احتساب الراتب إذا ما فيه عمود salary:</label>
        <select name="sum_policy" class="border rounded p-1">
          <option value="bh">الأساسي + السكن (افتراضي)</option>
          <option value="bhco">الأساسي + السكن + عمولة + بدلات</option>
        </select>
      </div>
      <button class="px-4 py-2 bg-blue-600 text-white rounded hover:bg-blue-700">استيراد</button>
    </form>

    <?php if ($done['inserted']||$done['updated']||$done['errors']||$done['skipped']): ?>
      <div class="mt-6">
        <h2 class="font-semibold mb-2">النتيجة:</h2>
        <ul class="list-disc pr-5 space-y-1">
          <li>المضاف: <?= (int)$done['inserted'] ?></li>
          <li>المحدَّث: <?= (int)$done['updated'] ?></li>
          <li>المتجاوز: <?= (int)$done['skipped'] ?></li>
          <li>الأخطاء: <?= (int)$done['errors'] ?></li>
        </ul>
        <?php if ($done['notes']): ?>
          <details class="mt-3">
            <summary class="cursor-pointer text-sm text-gray-700">ملاحظات</summary>
            <ul class="list-disc pr-5 text-sm mt-2 space-y-1">
              <?php foreach($done['notes'] as $n): ?>
                <li><?= htmlspecialchars($n, ENT_QUOTES, 'UTF-8') ?></li>
              <?php endforeach; ?>
            </ul>
          </details>
        <?php endif; ?>
      </div>
    <?php endif; ?>
  </div>
</body>
</html>