<?php
// /erp_task_system/api/contracts.php
session_start();
require_once '../config/db.php';
header('Content-Type: application/json; charset=utf-8');

if (!isset($_SESSION['user'])) { echo json_encode(['ok'=>false,'msg'=>'login']); exit; }
$me   = $_SESSION['user'];
$role = $me['role'] ?? 'user';

$action  = $_REQUEST['action'] ?? 'list';
$user_id = intval($_REQUEST['user_id'] ?? 0);
if (!$user_id) { echo json_encode(['ok'=>false,'msg'=>'user_id']); exit; }

// صلاحيات: admin/manager يقدرون يعدّلون أي موظف، user نفسه فقط
$canEditAll = in_array($role, ['admin','manager']);
if (!$canEditAll && $user_id !== intval($me['id'])) {
  echo json_encode(['ok'=>false,'msg'=>'forbidden']); exit;
}

$UPLOAD_DIR = realpath(__DIR__ . '/../uploads') . '/contracts/';
$PUBLIC_URL = '/erp_task_system/uploads/contracts/';
if (!is_dir($UPLOAD_DIR)) { @mkdir($UPLOAD_DIR, 0777, true); }

// helper
function safe_date($s){ $s=trim($s??''); return $s!=='' ? $s : null; }
function enum_in($v,$arr){ return in_array($v,$arr) ? $v : $arr[0]; }

try {

  /* ====== LIST عقود الموظف ====== */
  if ($action === 'list') {
    $sql = "SELECT c.*,
                   DATEDIFF(c.end_date, CURDATE())           AS days_to_end,
                   DATEDIFF(c.probation_end_date, CURDATE()) AS days_to_prob_end,
                   p.name AS project_name
            FROM contracts c
            LEFT JOIN projects p ON p.id = c.project_id
            WHERE c.user_id = ?
            ORDER BY c.status='ساري' DESC, c.start_date DESC, c.id DESC";
    $stmt = $conn->prepare($sql);
    $stmt->execute([$user_id]);
    $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // مرفقات لكل عقد (count)
    if ($rows){
      $ids = array_column($rows,'id');
      $in  = implode(',', array_fill(0,count($ids),'?'));
      $q   = $conn->prepare("SELECT contract_id, COUNT(*) cnt FROM contract_files WHERE contract_id IN ($in) GROUP BY contract_id");
      $q->execute($ids);
      $map = [];
      foreach($q->fetchAll(PDO::FETCH_ASSOC) as $r){ $map[$r['contract_id']] = $r['cnt']; }
      foreach($rows as &$r){ $r['files_count'] = intval($map[$r['id']] ?? 0); }
    }

    echo json_encode(['ok'=>true,'data'=>$rows]); exit;
  }

  /* ====== SAVE (إنشاء/تعديل) ====== */
  if ($action === 'save') {
    $id        = intval($_POST['id'] ?? 0);
    $employment_type = enum_in($_POST['employment_type'] ?? '', ['دوام كامل','دوام جزئي','عن بُعد','أجير','تمهير','تدريب تعاوني','أخرى']);
    $contract_type   = enum_in($_POST['contract_type']   ?? '', ['محدد المدة','غير محدد المدة','مؤقت','تدريب']);
    $start_date      = safe_date($_POST['start_date'] ?? '');
    $end_date        = safe_date($_POST['end_date']   ?? '');
    $probation_months = $_POST['probation_months'] !== '' ? intval($_POST['probation_months']) : null;
    $notify_end   = intval($_POST['notify_days_before_end']           ?? 30);
    $notify_prob  = intval($_POST['notify_days_before_probation_end'] ?? 7);
    $status       = enum_in($_POST['status'] ?? 'ساري', ['ساري','منتهي','معلق']);
    $project_id   = $_POST['project_id'] !== '' ? intval($_POST['project_id']) : null;
    $notes        = trim($_POST['notes'] ?? '');

    if (!$start_date) { echo json_encode(['ok'=>false,'msg'=>'start_date']); exit; }
    if ($contract_type !== 'غير محدد المدة' && $end_date && $end_date < $start_date) {
      echo json_encode(['ok'=>false,'msg'=>'end_before_start']); exit;
    }
    if ($contract_type === 'غير محدد المدة') { $end_date = null; }

    // منع تداخل العقود "السارية" لنفس الموظف
    if ($status === 'ساري') {
      // نطاق: [start_date, end_date أو NULL = مفتوح]
      $sqlOverlap = "SELECT COUNT(*) FROM contracts
                     WHERE user_id=? AND status='ساري' ".($id>0?"AND id<>$id":"")."
                       AND (
                           (end_date IS NULL OR end_date >= ?)
                           AND (? IS NULL OR start_date <= ?)
                       )";
      $stmt = $conn->prepare($sqlOverlap);
      $stmt->execute([$user_id, $start_date, $end_date, $end_date]);
      if ($stmt->fetchColumn() > 0) {
        echo json_encode(['ok'=>false,'msg'=>'overlap']); exit;
      }
    }

    if ($id>0) {
      $sql = "UPDATE contracts
              SET employment_type=?, contract_type=?, start_date=?, end_date=?,
                  probation_months=?, notify_days_before_end=?, notify_days_before_probation_end=?,
                  status=?, project_id=?, notes=?
              WHERE id=? AND user_id=?";
      $conn->prepare($sql)->execute([$employment_type,$contract_type,$start_date,$end_date,
                                     $probation_months,$notify_end,$notify_prob,$status,$project_id,$notes,
                                     $id,$user_id]);
      $contract_id = $id;
    } else {
      $sql = "INSERT INTO contracts
                (user_id, employment_type, contract_type, start_date, end_date,
                 probation_months, notify_days_before_end, notify_days_before_probation_end,
                 status, project_id, notes)
              VALUES (?,?,?,?,?,?,?,?,?,?,?)";
      $conn->prepare($sql)->execute([$user_id,$employment_type,$contract_type,$start_date,$end_date,
                                     $probation_months,$notify_end,$notify_prob,$status,$project_id,$notes]);
      $contract_id = $conn->lastInsertId();
    }

    // Sync users.employment_type مع العقد الساري (آخر واحد)
    if ($status === 'ساري') {
      $conn->prepare("UPDATE users SET employment_type=? WHERE id=?")->execute([$employment_type, $user_id]);
    } else {
      // لو ما عاد فيه عقد ساري، فضّله (NULL)
      $q = $conn->prepare("SELECT COUNT(*) FROM contracts WHERE user_id=? AND status='ساري'");
      $q->execute([$user_id]);
      if (intval($q->fetchColumn())===0) {
        $conn->prepare("UPDATE users SET employment_type=NULL WHERE id=?")->execute([$user_id]);
      }
    }

    echo json_encode(['ok'=>true,'id'=>$contract_id]); exit;
  }

  /* ====== RENEW (ينسخ عقد جديد) ====== */
  if ($action === 'renew') {
    $old_id    = intval($_POST['id'] ?? 0);
    $new_start = safe_date($_POST['new_start_date'] ?? '');
    $new_end   = safe_date($_POST['new_end_date'] ?? '');
    $close_old = intval($_POST['close_old'] ?? 1);

    if (!$old_id || !$new_start) { echo json_encode(['ok'=>false,'msg'=>'missing']); exit; }

    $o = $conn->prepare("SELECT * FROM contracts WHERE id=? AND user_id=?");
    $o->execute([$old_id,$user_id]);
    $old = $o->fetch(PDO::FETCH_ASSOC);
    if (!$old) { echo json_encode(['ok'=>false,'msg'=>'not_found']); exit; }

    $employment_type = $old['employment_type'];
    $contract_type   = $old['contract_type'];
    $probation_months= $old['probation_months'];
    $notify_end      = $old['notify_days_before_end'];
    $notify_prob     = $old['notify_days_before_probation_end'];
    $project_id      = $old['project_id'];
    $notes           = 'تجديد للعقد #' . $old_id . ' - ' . ($old['notes'] ?? '');

    if ($contract_type === 'غير محدد المدة') { $new_end = null; }
    if ($new_end && $new_end < $new_start) { echo json_encode(['ok'=>false,'msg'=>'end_before_start']); exit; }

    // منع تداخل الساري
    $sqlOverlap = "SELECT COUNT(*) FROM contracts WHERE user_id=? AND status='ساري'
                   AND ( (end_date IS NULL OR end_date >= ?) AND (? IS NULL OR start_date <= ?) )";
    $stmt=$conn->prepare($sqlOverlap); $stmt->execute([$user_id, $new_start, $new_end, $new_end]);
    if ($stmt->fetchColumn()>0){ echo json_encode(['ok'=>false,'msg'=>'overlap']); exit; }

    // أنشئ عقد جديد (ساري)
    $ins = $conn->prepare("INSERT INTO contracts
      (user_id, employment_type, contract_type, start_date, end_date, probation_months,
       notify_days_before_end, notify_days_before_probation_end, status, project_id, notes)
      VALUES (?,?,?,?,?,?,?,?, 'ساري', ?, ?)");
    $ins->execute([$user_id,$employment_type,$contract_type,$new_start,$new_end,$probation_months,
                   $notify_end,$notify_prob,$project_id,$notes]);
    $new_id = $conn->lastInsertId();

    // أغلق القديم لو طلبت
    if ($close_old) {
      $conn->prepare("UPDATE contracts SET status='منتهي' WHERE id=?")->execute([$old_id]);
    }

    // Sync نوع التوظيف
    $conn->prepare("UPDATE users SET employment_type=? WHERE id=?")->execute([$employment_type,$user_id]);

    echo json_encode(['ok'=>true,'id'=>$new_id]); exit;
  }

  /* ====== EXTEND (تمديد نهاية عقد) ====== */
  if ($action === 'extend') {
    $id      = intval($_POST['id'] ?? 0);
    $new_end = safe_date($_POST['new_end_date'] ?? '');
    if (!$id || !$new_end) { echo json_encode(['ok'=>false,'msg'=>'missing']); exit; }

    $o=$conn->prepare("SELECT start_date, contract_type FROM contracts WHERE id=? AND user_id=?");
    $o->execute([$id,$user_id]);
    $row=$o->fetch(PDO::FETCH_ASSOC);
    if (!$row) { echo json_encode(['ok'=>false,'msg'=>'not_found']); exit; }
    if ($row['contract_type']==='غير محدد المدة'){ echo json_encode(['ok'=>false,'msg'=>'indefinite']); exit; }
    if ($new_end < $row['start_date']) { echo json_encode(['ok'=>false,'msg'=>'end_before_start']); exit; }

    $conn->prepare("UPDATE contracts SET end_date=? WHERE id=? AND user_id=?")->execute([$new_end,$id,$user_id]);
    echo json_encode(['ok'=>true]); exit;
  }

  /* ====== CLOSE (إنهاء عقد) ====== */
  if ($action === 'close') {
    $id      = intval($_POST['id'] ?? 0);
    $end_now = safe_date($_POST['end_date'] ?? date('Y-m-d'));
    if (!$id) { echo json_encode(['ok'=>false,'msg'=>'missing']); exit; }

    $conn->prepare("UPDATE contracts SET status='منتهي', end_date=COALESCE(end_date, ?) WHERE id=? AND user_id=?")
         ->execute([$end_now,$id,$user_id]);

    // لو ما بقي عقد ساري، فضّل employment_type
    $q=$conn->prepare("SELECT COUNT(*) FROM contracts WHERE user_id=? AND status='ساري'");
    $q->execute([$user_id]);
    if (intval($q->fetchColumn())===0) {
      $conn->prepare("UPDATE users SET employment_type=NULL WHERE id=?")->execute([$user_id]);
    }
    echo json_encode(['ok'=>true]); exit;
  }

  /* ====== FILES: list / upload / delete ====== */
  if ($action === 'files') {
    $contract_id = intval($_POST['contract_id'] ?? 0);
    if (!$contract_id){ echo json_encode(['ok'=>false,'msg'=>'contract_id']); exit; }
    $stmt = $conn->prepare("SELECT id, file_name, file_path, CONCAT(?, file_path) AS url, created_at
                            FROM contract_files WHERE contract_id=? ORDER BY id DESC");
    $stmt->execute([$PUBLIC_URL,$contract_id]);
    echo json_encode(['ok'=>true,'data'=>$stmt->fetchAll(PDO::FETCH_ASSOC)]); exit;
  }

  if ($action === 'upload_file') {
    $contract_id = intval($_POST['contract_id'] ?? 0);
    if (!$contract_id){ echo json_encode(['ok'=>false,'msg'=>'contract_id']); exit; }
    if (empty($_FILES['file']['name']) || $_FILES['file']['error']!==0){
      echo json_encode(['ok'=>false,'msg'=>'no_file']); exit;
    }
    $ext = strtolower(pathinfo($_FILES['file']['name'], PATHINFO_EXTENSION));
    $allowed = ['pdf','jpg','jpeg','png','doc','docx'];
    if (!in_array($ext,$allowed)){ echo json_encode(['ok'=>false,'msg'=>'bad_ext']); exit; }
    $stored = uniqid('ctr_').'.'.$ext;
    if (!move_uploaded_file($_FILES['file']['tmp_name'], $UPLOAD_DIR.$stored)) {
      echo json_encode(['ok'=>false,'msg'=>'upload_fail']); exit;
    }
    $conn->prepare("INSERT INTO contract_files (contract_id, file_name, file_path) VALUES (?,?,?)")
         ->execute([$contract_id, $_FILES['file']['name'], $stored]);
    echo json_encode(['ok'=>true]); exit;
  }

  if ($action === 'delete_file') {
    $id = intval($_POST['id'] ?? 0);
    if (!$id) { echo json_encode(['ok'=>false,'msg'=>'id']); exit; }
    $s=$conn->prepare("SELECT file_path FROM contract_files WHERE id=?");
    $s->execute([$id]); $row=$s->fetch(PDO::FETCH_ASSOC);
    if ($row && !empty($row['file_path'])) { @unlink($UPLOAD_DIR.$row['file_path']); }
    $conn->prepare("DELETE FROM contract_files WHERE id=?")->execute([$id]);
    echo json_encode(['ok'=>true]); exit;
  }

  echo json_encode(['ok'=>false,'msg'=>'bad_action']);

} catch (Throwable $e) {
  echo json_encode(['ok'=>false,'msg'=>'error','err'=>$e->getMessage()]);
}
