<?php
// /erp_task_system/api/users_basic_update.php
session_start();
require_once '../config/db.php';
require_once '../authorize.php';
header('Content-Type: application/json; charset=utf-8');

/* السماح فقط للمدير والمدير العام */
if (!isset($_SESSION['user']) || !in_array($_SESSION['user']['role'] ?? '', ['admin','manager'])) {
  echo json_encode(['ok'=>false,'msg'=>'forbidden'], JSON_UNESCAPED_UNICODE); exit;
}

/* ================= Helpers ================= */
function fail($msg, $err=null){ echo json_encode(['ok'=>false,'msg'=>$msg,'err'=>$err], JSON_UNESCAPED_UNICODE); exit; }

/* وجود عمود */
function has_col(PDO $conn, string $table, string $col): bool {
  static $cache = [];
  $k = "$table.$col";
  if (array_key_exists($k, $cache)) return $cache[$k];
  $q = $conn->prepare("
    SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS
    WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ? AND COLUMN_NAME = ?
  ");
  $q->execute([$table, $col]);
  return $cache[$k] = ((int)$q->fetchColumn() > 0);
}

/* رفع ملف لمسار ويب */
function upfile(string $field, string $webDir): ?string {
  if (!isset($_FILES[$field]) || empty($_FILES[$field]['name'])) return null;
  $root = rtrim($_SERVER['DOCUMENT_ROOT'] ?? '', '/');
  $absBase  = $root . rtrim($webDir, '/');
  if (!is_dir($absBase)) @mkdir($absBase, 0775, true);
  $name = $_FILES[$field]['name'];
  $ext  = strtolower(pathinfo($name, PATHINFO_EXTENSION));
  $final = sprintf('%s_%s_%s.%s', $field, date('Ymd_His'), bin2hex(random_bytes(3)), $ext);
  $webPath = rtrim($webDir,'/') . '/' . $final;
  $absPath = $root . $webPath;
  if (!move_uploaded_file($_FILES[$field]['tmp_name'], $absPath)) return null;
  return $webPath;
}

/* Normalizers */
function nstr($v){ $v = trim((string)($v ?? '')); return ($v===''?null:$v); }
function nlower($v){ $s = nstr($v); return $s===null?null:mb_strtolower($s,'UTF-8'); }
function nint($v){ $v = nstr($v); return $v===null?null:(int)$v; }
function nbool01($v){ return !empty($v)?1:0; }
function ndate($v){
  $v = trim((string)($v ?? ''));
  if ($v==='') return null;
  $dt = DateTime::createFromFormat('Y-m-d', $v);
  return ($dt && $dt->format('Y-m-d')===$v) ? $v : null;
}
/* مقارنة منصفة (تعامل NULL/فراغ وأرقام) */
function eqq($a,$b){
  if ($a === '' || $a === null) $a = null;
  if ($b === '' || $b === null) $b = null;
  if (is_numeric($a) && is_numeric($b)) return ((float)$a == (float)$b);
  return ((string)$a === (string)$b);
}

/* يبني مصفوفة تغييرات: ياخذ اللي تغيّر بس */
function diff_changed(PDO $conn, array $curr, array $incoming, array $normMap, string $table='users'): array {
  $changed = [];
  foreach ($incoming as $k=>$newVal) {
    if (!has_col($conn, $table, $k)) continue;
    $normFn = $normMap[$k] ?? 'nstr';
    $oldVal = isset($curr[$k]) ? call_user_func($normFn, $curr[$k]) : null;
    if (!eqq($newVal, $oldVal)) $changed[$k] = $newVal;
  }
  return $changed;
}

/* ================= Inputs ================= */
$uid  = (int)($_POST['user_id'] ?? 0);
$part = trim($_POST['part'] ?? ''); // basic|address|identity|bank|''

if ($uid <= 0) fail('bad_user');

try {
  $conn->beginTransaction();

  // المستخدم الحالي
  $st = $conn->prepare("SELECT * FROM users WHERE id = ?");
  $st->execute([$uid]);
  $U = $st->fetch(PDO::FETCH_ASSOC);
  if (!$U) throw new Exception('User not found');

  /* ================= BASIC ================= */
  if ($part === '' || $part === 'basic') {
    $name   = nstr($_POST['name']  ?? '');
    $email  = nlower($_POST['email'] ?? '');
    $pass   = $_POST['password'] ?? '';

    if ($name===null || $email===null) throw new Exception('name/email required');

    $employee_number = nstr($_POST['employee_number'] ?? ''); // '' -> null by nstr
    $incoming = [
      'name'               => $name,
      'email'              => $email,
      'gender'             => nstr($_POST['gender'] ?? null),
      'religion'           => nstr($_POST['religion'] ?? null),
      'marital_status'     => nstr($_POST['marital_status'] ?? null),
      'personal_email'     => nlower($_POST['personal_email'] ?? null),
      'personal_phone'     => nstr($_POST['personal_phone'] ?? null),
      'role'               => nstr($_POST['role'] ?? 'employee'),
      'department_id'      => nint($_POST['department_id'] ?? null),
      'work_shift_id'      => nint($_POST['work_shift_id'] ?? null),
      'location_id'        => nint($_POST['location_id'] ?? null),
      'day_off'            => nint($_POST['day_off'] ?? null),
      'employee_number'    => $employee_number,
      'job_title'          => nstr($_POST['job_title'] ?? null),
      'hire_date'          => ndate($_POST['hire_date'] ?? null),
      'employment_status'  => nstr($_POST['employment_status'] ?? 'نشط'),
      'nationality'        => nstr($_POST['nationality'] ?? ''),
      'can_sign'           => nbool01($_POST['can_sign'] ?? 0),
      'can_transfer_tasks' => nbool01($_POST['can_transfer_tasks'] ?? 0),
      'secret_level'       => nstr($_POST['secret_level'] ?? 'normal'),
    ];

    $normMap = [
      'name'=>'nstr','email'=>'nlower','gender'=>'nstr','religion'=>'nstr','marital_status'=>'nstr',
      'personal_email'=>'nlower','personal_phone'=>'nstr','role'=>'nstr','department_id'=>'nint',
      'work_shift_id'=>'nint','location_id'=>'nint','day_off'=>'nint','employee_number'=>'nstr',
      'job_title'=>'nstr','hire_date'=>'ndate','employment_status'=>'nstr','nationality'=>'nstr',
      'can_sign'=>'nint','can_transfer_tasks'=>'nint','secret_level'=>'nstr',
    ];

    $changed = diff_changed($conn, $U, $incoming, $normMap, 'users');

    // لو الإيميل ضمن التغييرات، افحص التكرار
    if (array_key_exists('email', $changed)) {
      $chk = $conn->prepare("SELECT COUNT(*) FROM users WHERE LOWER(email) = ? AND id <> ?");
      $chk->execute([$changed['email'], $uid]);
      if ((int)$chk->fetchColumn() > 0) throw new Exception('email_taken');
    }

    // لو الرقم الوظيفي تغيّر وموب NULL، افحص التكرار
    if (array_key_exists('employee_number', $changed) && $changed['employee_number'] !== null && has_col($conn,'users','employee_number')) {
      $ch2 = $conn->prepare("SELECT COUNT(*) FROM users WHERE employee_number = ? AND id <> ?");
      $ch2->execute([$changed['employee_number'], $uid]);
      if ((int)$ch2->fetchColumn() > 0) throw new Exception('employee_number_taken');
    }

    // ملفات جديدة فقط تحدث
    $profile_image  = upfile('profile_image',  '/erp_task_system/users/img');
    $signature_file = upfile('signature_file', '/erp_task_system/users/signatures');
    if ($profile_image)  $changed['profile_image']  = $profile_image;
    if ($signature_file) $changed['signature_file'] = $signature_file;

    // باسوورد إذا انكتب
    if ($pass !== '' && has_col($conn,'users','password')) {
      $changed['password'] = password_hash($pass, PASSWORD_BCRYPT);
    }

    if (!empty($changed)) {
      if (has_col($conn,'users','updated_at')) $changed['updated_at'] = date('Y-m-d H:i:s');
      $set=[]; $vals=[];
      foreach($changed as $k=>$v){ $set[]="`$k`=?"; $vals[]=$v; }
      $vals[] = $uid;
      $sql = "UPDATE users SET ".implode(',',$set)." WHERE id = ?";
      $stmt = $conn->prepare($sql);
      $stmt->execute($vals);
      // جِدّد نسخة $U في الذاكرة للأجزاء التالية إذا بنحفظ الكل
      $U = array_merge($U, $changed);
    }
  }

  /* ================= ADDRESS ================= */
  if ($part === '' || $part === 'address') {
    $incoming = [
      'address_building'          => nstr($_POST['address_building'] ?? ''),
      'address_street'            => nstr($_POST['address_street'] ?? ''),
      'address_district'          => nstr($_POST['address_district'] ?? ''),
      'address_city'              => nstr($_POST['address_city'] ?? ''),
      'address_zip'               => nstr($_POST['address_zip'] ?? ''),
      'address_additional_number' => nstr($_POST['address_additional_number'] ?? ''),
      'work_email'                => nlower($_POST['work_email'] ?? ''),
      'work_phone'                => nstr($_POST['work_phone'] ?? ''),
    ];
    $normMap = [
      'address_building'=>'nstr','address_street'=>'nstr','address_district'=>'nstr','address_city'=>'nstr',
      'address_zip'=>'nstr','address_additional_number'=>'nstr','work_email'=>'nlower','work_phone'=>'nstr',
    ];
    $changed = diff_changed($conn, $U, $incoming, $normMap, 'users');
    $na = upfile('national_address_certificate','/erp_task_system/users/docs');
    if ($na) $changed['national_address_certificate'] = $na;

    if (!empty($changed)) {
      if (has_col($conn,'users','updated_at')) $changed['updated_at'] = date('Y-m-d H:i:s');
      $set=[]; $vals=[];
      foreach($changed as $k=>$v){ $set[]="`$k`=?"; $vals[]=$v; }
      $vals[]=$uid;
      $stmt = $conn->prepare("UPDATE users SET ".implode(',',$set)." WHERE id=?");
      $stmt->execute($vals);
      $U = array_merge($U, $changed);
    }
  }

  /* ================= IDENTITY ================= */
  if ($part === '' || $part === 'identity') {
    $national_id = nstr($_POST['national_id'] ?? '');
    if ($national_id !== null && !preg_match('/^\d{10}$/', $national_id)) {
      throw new Exception('bad_national_id');
    }
    $incoming = [
      'national_id'            => $national_id,
      'national_id_issue_date' => ndate($_POST['national_id_issue_date'] ?? null),
      'national_id_place'      => nstr($_POST['national_id_place'] ?? ''),
      'passport_number'        => nstr($_POST['passport_number'] ?? ''),
      'passport_place'         => nstr($_POST['passport_place'] ?? ''),
      'passport_country'       => nstr($_POST['passport_country'] ?? ''),
      'passport_issue_date'    => ndate($_POST['passport_issue_date'] ?? null),
      'passport_expire_date'   => ndate($_POST['passport_expire_date'] ?? null),
    ];
    $normMap = [
      'national_id'=>'nstr','national_id_issue_date'=>'ndate','national_id_place'=>'nstr',
      'passport_number'=>'nstr','passport_place'=>'nstr','passport_country'=>'nstr',
      'passport_issue_date'=>'ndate','passport_expire_date'=>'ndate',
    ];
    $changed = diff_changed($conn, $U, $incoming, $normMap, 'users');

    if (!empty($changed)) {
      if (has_col($conn,'users','updated_at')) $changed['updated_at'] = date('Y-m-d H:i:s');
      $set=[]; $vals=[];
      foreach($changed as $k=>$v){ $set[]="`$k`=?"; $vals[]=$v; }
      $vals[]=$uid;
      $stmt = $conn->prepare("UPDATE users SET ".implode(',',$set)." WHERE id=?");
      $stmt->execute($vals);
      $U = array_merge($U, $changed);
    }
  }

  /* ================= BANK ================= */
  if ($part === '' || $part === 'bank') {
    $incoming = [
      'bank_account' => nstr($_POST['bank_account'] ?? ''),
      'iban'         => ($tmp = nstr($_POST['iban'] ?? '')) ? mb_strtoupper(str_replace(' ','',$tmp)) : null,
      'bank_name'    => nstr($_POST['bank_name'] ?? ''),
    ];
    $normMap = [ 'bank_account'=>'nstr','iban'=>'nstr','bank_name'=>'nstr' ];
    $changed = diff_changed($conn, $U, $incoming, $normMap, 'users');
    $iban_cert = upfile('iban_certificate','/erp_task_system/users/docs');
    if ($iban_cert) $changed['iban_certificate'] = $iban_cert;

    if (!empty($changed)) {
      if (has_col($conn,'users','updated_at')) $changed['updated_at'] = date('Y-m-d H:i:s');
      $set=[]; $vals=[];
      foreach($changed as $k=>$v){ $set[]="`$k`=?"; $vals[]=$v; }
      $vals[]=$uid;
      $stmt = $conn->prepare("UPDATE users SET ".implode(',',$set)." WHERE id=?");
      $stmt->execute($vals);
      $U = array_merge($U, $changed);
    }
  }

  $conn->commit();
  echo json_encode(['ok'=>true,'id'=>$uid], JSON_UNESCAPED_UNICODE);

} catch (Throwable $e) {
  if ($conn->inTransaction()) $conn->rollBack();
  $msg = $e->getMessage();
  $nice = $msg==='email_taken' ? 'البريد مستخدم لمستخدم آخر'
        : ($msg==='employee_number_taken' ? 'الرقم الوظيفي مستخدم'
        : ($msg==='bad_national_id' ? 'رقم الهوية غير صحيح (10 أرقام)'
        : 'error'));
  echo json_encode(['ok'=>false,'msg'=>$nice,'err'=>$msg], JSON_UNESCAPED_UNICODE);
  exit;
}
