<?php
require_once '../config/db.php';
require_once '../auth.php';
require_once '../authorize.php';

ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

/* ✅ تحقق الجلسة والصلاحيات (منع المستخدم العادي من الدخول) */
if (
  !isset($_SESSION['user']) ||
  !isset($_SESSION['user']['role']) ||
  !in_array($_SESSION['user']['role'], ['admin', 'manager'])
) {
  session_unset();
  session_destroy();
  header('Location: ../unauthorized.php');
  exit;
}

/* ✅ بيانات المستخدم والثيم */
$user          = $_SESSION['user'];
$role          = $user['role'];
$user_id       = $user['id'] ?? null;
$department_id = $user['department_id'] ?? null;
$isGeneralManager = (int)($user['is_general_manager'] ?? $user['is_super_manager'] ?? 0);

$color = $user['color'] ?? [
  'hex_code'   => '#0b1023',
  'text_color' => '#e9eefb',
  'nav_color'  => '#121b35',
];

$userThemeColor     = $color['hex_code'];
$userThemeTextColor = $color['text_color'];
$userThemeNavColor  = $color['nav_color'];

/* ✅ CSRF */
if (empty($_SESSION['csrf_token'])) {
  $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}
$csrf_token = $_SESSION['csrf_token'];

/* ✅ الحقول الأساسية */
$base_fields = [
  'title'        => 'عنوان المهمة',
  'description'  => 'الوصف',
  'priority'     => 'الأولوية',
  'secret_level' => 'درجة السرية',
  'due_date'     => 'تاريخ الاستحقاق'
];

/* ✅ جلب الإدارات حسب الدور */
try {
  if ($role === 'admin' || $isGeneralManager === 1) {
    $stmt = $conn->query("SELECT id, name FROM departments ORDER BY name ASC");
    $departments = $stmt->fetchAll(PDO::FETCH_ASSOC);
  } else {
    $stmt = $conn->prepare("SELECT id, name FROM departments WHERE id = ?");
    $stmt->execute([$department_id]);
    $departments = $stmt->fetchAll(PDO::FETCH_ASSOC);
  }
} catch (Exception $e) {
  $departments = [];
}
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
  <meta charset="UTF-8">
  <title>بوابة الطلبات - إدارة النماذج</title>
  <meta name="viewport" content="width=device-width, initial-scale=1" />
  <script src="https://cdn.tailwindcss.com"></script>

  <style>
    :root{
      --bg:  <?= htmlspecialchars($userThemeColor) ?>;
      --nav: <?= htmlspecialchars($userThemeNavColor) ?>;
      --txt: <?= htmlspecialchars($userThemeTextColor) ?>;
      --accent:#4d70f4;
    }
    body{ background-color: var(--bg); color: var(--txt); }

    /* نلوّن الكرت بس، بدون ما نأثر على عناصر عامة مثل a/li/tr:hover */
    .main-card{
      background-color: var(--nav);
      color: var(--txt);
    }
    .main-card select,
    .main-card input,
    .main-card textarea{
      background-color: var(--bg);
      color: var(--txt);
      border-color: rgba(255,255,255,.18);
    }
    .btn-primary{
      background-color: var(--accent);
      color:#fff;
      border-radius:12px;
    }
    .btn-primary:hover{ background: rgb(19,11,57); }
    .rounded-12{ border-radius:12px; }
  </style>
</head>

<?php include $_SERVER['DOCUMENT_ROOT'] . '/erp_task_system/navbar/navbar.php'; ?>

<body class="p-8 text-right">
  <?php $helpHref = file_exists(__DIR__.'/help_portal_form.php') ? 'help_portal_form.php' : 'help.php'; ?>

  <div class="max-w-3xl mx-auto main-card p-6 rounded shadow">
    <div class="flex items-center gap-3 mb-4">
      <h2 class="text-xl font-bold">🛠️ تصميم نموذج جديد للإدارة</h2>

      <!-- زر التعليمات يسار -->
      <a href="<?= htmlspecialchars($helpHref) ?>" target="_blank" rel="noopener"
         class="btn-primary px-4 py-2 ml-auto inline-flex items-center gap-1 no-print"
         title="تعليمات بوابة الطلبات">❓ تعليمات</a>
    </div>

    <?php if (empty($departments)): ?>
      <div class="bg-red-500/10 text-red-200 rounded-12 p-3 text-sm">
        لا توجد إدارات متاحة لك لإنشاء نماذج حالياً.
      </div>
    <?php else: ?>
      <form action="store_portal_form.php" method="POST" onsubmit="return prepareFieldsBeforeSubmit();" class="space-y-4">
        <input type="hidden" name="csrf_token" value="<?= htmlspecialchars($csrf_token) ?>">

        <label class="block font-semibold">اختر الإدارة:</label>
        <select name="department_id" id="departmentSelect" onchange="loadCustomFields()" required class="w-full border p-2 rounded">
          <option value="">-- اختر إدارة --</option>
          <?php foreach ($departments as $dep): ?>
            <option value="<?= (int)$dep['id'] ?>"><?= htmlspecialchars($dep['name']) ?></option>
          <?php endforeach; ?>
        </select>

        <label class="block font-semibold">عنوان النموذج:</label>
        <input type="text" name="title" required class="w-full border p-2 rounded">

        <h3 class="font-bold">الحقول:</h3>
        <div id="fieldsContainer" class="space-y-4"></div>

        <button type="button" onclick="addField()" class="btn-primary px-4 py-2">➕ إضافة حقل</button>

        <input type="hidden" name="fields_json" id="fieldsJson">

        <div class="pt-4">
          <button type="submit" class="btn-primary px-6 py-2">💾 حفظ النموذج</button>
        </div>
      </form>
    <?php endif; ?>
  </div>

  <script>
    // نمرّر الحقول الأساسية من PHP إلى JS
    window.baseFields = <?= json_encode($base_fields, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES) ?>;

    let customFields = [];

    function loadCustomFields() {
      const depId = document.getElementById('departmentSelect').value;
      if (!depId) { customFields = []; return; }

      fetch('load_field_options.php?department_id=' + encodeURIComponent(depId), {
        cache: 'no-store',
        credentials: 'same-origin'
      })
      .then(res => { if (!res.ok) throw new Error('Network'); return res.json(); })
      .then(data => { customFields = Array.isArray(data) ? data : []; })
      .catch(() => { customFields = []; });
    }

    function buildFieldOptions() {
      let html = '<optgroup label="🧩 الحقول الأساسية">';
      for (const [name, label] of Object.entries(window.baseFields)) {
        html += `<option value="${name}">${label}</option>`;
      }
      html += '</optgroup>';

      html += '<optgroup label="🧩 الحقول الإضافية">';
      for (const f of customFields) {
        html += `<option value="custom_${f.id}">${f.label}</option>`;
      }
      html += '</optgroup>';
      return html;
    }

    function addField() {
      const depId = document.getElementById('departmentSelect').value;
      if (!depId) {
        alert('❗ اختر الإدارة أولاً.');
        return;
      }

      const container = document.getElementById('fieldsContainer');
      const div = document.createElement('div');
      div.className = 'border p-4 rounded space-y-2';

      div.innerHTML = `
        <label>عنوان الحقل (يظهر للمستخدم):</label>
        <input type="text" class="w-full border p-2 rounded" data-type="label" required>

        <label>نوع الحقل:</label>
        <select class="w-full border p-2 rounded" data-type="type">
          <option value="text">نص</option>
          <option value="number">رقم</option>
          <option value="select">قائمة خيارات</option>
          <option value="date">تاريخ</option>
        </select>

        <label>الحقل البرمجي (المصدر):</label>
        <select class="w-full border p-2 rounded" data-type="name" required>
          ${buildFieldOptions()}
        </select>

        <label>هل الحقل إلزامي؟</label>
        <select class="w-full border p-2 rounded" data-type="required">
          <option value="true">نعم</option>
          <option value="false">لا</option>
        </select>

        <div class="optionsBox hidden">
          <label>خيارات القائمة (افصلها بفاصلة):</label>
          <input type="text" class="w-full border p-2 rounded" data-type="options" placeholder="مثال: خيار 1, خيار 2, خيار 3">
        </div>

        <button type="button" class="text-red-400 hover:underline" data-action="remove">❌ حذف الحقل</button>
      `;

      const typeSel   = div.querySelector('[data-type="type"]');
      const optionsBx = div.querySelector('.optionsBox');
      const removeBtn = div.querySelector('[data-action="remove"]');

      function toggleOptions() {
        if (typeSel.value === 'select') optionsBx.classList.remove('hidden');
        else optionsBx.classList.add('hidden');
      }
      typeSel.addEventListener('change', toggleOptions);
      toggleOptions();
      removeBtn.addEventListener('click', () => div.remove());

      container.appendChild(div);
    }

    function prepareFieldsBeforeSubmit() {
      const fields = [];
      const seen = new Set();
      const containers = document.querySelectorAll('#fieldsContainer > div');

      for (const c of containers) {
        const label = c.querySelector('[data-type="label"]')?.value?.trim();
        const name  = c.querySelector('[data-type="name"]')?.value;
        const type  = c.querySelector('[data-type="type"]')?.value;
        const req   = c.querySelector('[data-type="required"]')?.value === 'true';
        const optEl = c.querySelector('[data-type="options"]');

        if (!label || !name || !type) { alert('⚠️ يرجى إكمال جميع حقول كل عنصر.'); return false; }

        if (seen.has(name)) { alert('⚠️ تكرار في الحقل البرمجي: ' + name); return false; }
        seen.add(name);

        let options = [];
        if (type === 'select' && optEl?.value.trim()) {
          options = optEl.value.split(',').map(s => s.trim()).filter(Boolean);
        }

        fields.push({ label, name, type, required: req, options });
      }

      document.getElementById('fieldsJson').value = JSON.stringify(fields);

      if (fields.length === 0 && !confirm('النموذج بدون حقول — تبي تكمل؟')) return false;
      return true;
    }

    // لو الإدارة مختارة مسبقًا
    window.addEventListener('DOMContentLoaded', () => {
      const selectedDep = document.getElementById('departmentSelect').value;
      if (selectedDep) loadCustomFields();
    });
  </script>
</body>
</html>