<?php
require_once '../config/db.php';
require_once '../auth.php';
require_once '../authorize.php';

ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

/* ✅ تحقق الجلسة والصلاحيات */
if (
  !isset($_SESSION['user']) ||
  !isset($_SESSION['user']['role']) ||
  !in_array($_SESSION['user']['role'], ['admin', 'manager', 'user'])
) {
  session_unset();
  session_destroy();
  header('Location: ../unauthorized.php');
  exit;
}

/* ✅ بيانات المستخدم + الثيم */
$user          = $_SESSION['user'];
$role          = $user['role'];
$user_id       = $user['id'] ?? null;
$department_id = $user['department_id'] ?? null;

/* فلاغ المدير العام (ينشاف من جلسة المستخدم) */
$isGeneralManager = (int)($user['is_general_manager'] ?? $user['is_super_manager'] ?? 0);

$color = $user['color'] ?? [
  'hex_code'   => '#0b1023',
  'text_color' => '#e9eefb',
  'nav_color'  => '#121b35',
];

$userThemeColor     = $color['hex_code'];
$userThemeTextColor = $color['text_color'];
$userThemeNavColor  = $color['nav_color'];

/* ✅ جلب الإدارات المرتبطة بنماذج حسب الدور/الفلاغ */
try {
  if ($role === 'admin' || ($role === 'manager' && $isGeneralManager === 1)) {
    // الأدمن + المدير العام: يشوفون كل الإدارات اللي عندها نماذج
    $sql = "
      SELECT DISTINCT d.id, d.name
      FROM departments d
      INNER JOIN custom_forms f ON f.department_id = d.id
      ORDER BY d.name ASC
    ";
    $stmt = $conn->query($sql);
    $departments = $stmt->fetchAll(PDO::FETCH_ASSOC);

  } elseif ($role === 'manager') {
    // مدير عادي: فقط إدارته إذا عندها نماذج
    $sql = "
      SELECT DISTINCT d.id, d.name
      FROM departments d
      INNER JOIN custom_forms f ON f.department_id = d.id
      WHERE d.id = ?
      ORDER BY d.name ASC
    ";
    $stmt = $conn->prepare($sql);
    $stmt->execute([$department_id]);
    $departments = $stmt->fetchAll(PDO::FETCH_ASSOC);

  } else {
    // مستخدم: فقط إدارته إذا عندها نماذج (أو تقدر تمنعه تمامًا إذا ودك)
    $sql = "
      SELECT DISTINCT d.id, d.name
      FROM departments d
      INNER JOIN custom_forms f ON f.department_id = d.id
      WHERE d.id = ?
      ORDER BY d.name ASC
    ";
    $stmt = $conn->prepare($sql);
    $stmt->execute([$department_id]);
    $departments = $stmt->fetchAll(PDO::FETCH_ASSOC);
  }
} catch (Exception $e) {
  $departments = [];
}
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
  <meta charset="UTF-8" />
  <title>بوابة الطلبات</title>
  <meta name="viewport" content="width=device-width, initial-scale=1" />
  <script src="https://cdn.tailwindcss.com"></script>

  <style>
    :root{
      --bg:  <?= htmlspecialchars($userThemeColor) ?>;
      --nav: <?= htmlspecialchars($userThemeNavColor) ?>;
      --txt: <?= htmlspecialchars($userThemeTextColor) ?>;
      --accent: #4d70f4;
    }
    body{ background-color: var(--bg); }
    .main-card{
      background-color: var(--nav);
      color: var(--txt);
    }
    .main-card select,
    .main-card input,
    .main-card textarea{
      background-color: var(--bg);
      color: var(--txt);
      border-color: rgba(255,255,255,0.15);
    }
    .btn-primary{
      background-color: var(--accent);
      color:#fff; border-radius:12px;
    }
    .btn-primary:hover{ background-color: rgb(19,11,57); }
    .rounded-12{ border-radius:12px; }
  </style>
</head>

<?php include $_SERVER['DOCUMENT_ROOT'] . '/erp_task_system/navbar/navbar.php'; ?>

<body class="text-right">

  <div class="p-6">
    <div class="max-w-2xl mx-auto main-card p-6 rounded shadow">
      <div class="flex items-center gap-2 mb-4">
        <h2 class="text-xl font-bold">📩 بوابة الطلبات</h2>

        <a href="help_requests_portal.php" target="_blank" rel="noopener"
           class="btn-primary px-3 py-1 ml-auto inline-flex items-center gap-1"
           title="تعليمات بوابة الطلبات">
          ❓ تعليمات
        </a>
      </div>

      <?php if (empty($departments)): ?>
        <div class="bg-red-500/10 text-red-200 rounded-12 p-3 text-sm">
          لا توجد إدارات مرتبطة بنماذج متاحة لك حالياً.
        </div>
      <?php else: ?>
        <label for="departmentSelect" class="block mb-2">اختر الإدارة:</label>
        <select id="departmentSelect" onchange="loadForm()" class="w-full border p-2 rounded mb-4">
          <option value="">-- اختر --</option>
          <?php foreach ($departments as $dep): ?>
            <option value="<?= (int)$dep['id'] ?>"><?= htmlspecialchars($dep['name']) ?></option>
          <?php endforeach; ?>
        </select>

        <div id="formArea" class="mt-4 min-h-12"></div>
      <?php endif; ?>
    </div>
  </div>

<script>
  const BASE = '/erp_task_system'; // عدّلها إذا مسارك مختلف

  function loadForm() {
    const depId = document.getElementById('departmentSelect').value;
    const formArea = document.getElementById('formArea');
    if (!depId) { formArea.innerHTML = ''; return; }

    formArea.textContent = '⏳ جاري تحميل النموذج...';

    fetch(`${BASE}/custom_forms/load_portal_form.php?department_id=${encodeURIComponent(depId)}`, {
      cache: 'no-store',
      credentials: 'same-origin',
      headers: { 'X-Requested-With': 'XMLHttpRequest' }
    })
    .then(async (res) => {
      const text = await res.text();
      if (!res.ok) throw new Error(`HTTP ${res.status} – ${text.slice(0, 300)}`);
      return text;
    })
    .then(html => { formArea.innerHTML = html; })
    .catch(err => {
      console.error(err);
      formArea.textContent = '❌ تعذر تحميل النموذج: ' + err.message;
    });
  }
</script>

</body>
</html>
