<?php
require_once '../config/db.php';
require_once '../auth.php';
require_once '../authorize.php';

if (!isset($_SESSION['user']) || !isset($_SESSION['user']['role']) || !in_array($_SESSION['user']['role'], ['admin', 'manager', 'user'])) {
    session_unset();
    session_destroy();
    header('Location: ../unauthorized.php');
    exit;
}

$user = $_SESSION['user'];
$role = $user['role'];
$department_id = $user['department_id'];

$color = $user['color'];
$userThemeColor = $color['hex_code'];
$userThemeTextColor = $color['text_color'];
$userThemeNavColor = $color['nav_color'];
$currentUserColorId = $user['color_id'];

$defaultFields = [
    'title' => 'عنوان المهمة',
    'description' => 'وصف المهمة',
    'created_at' => 'تاريخ الإنشاء',
    'due_date' => 'تاريخ الاستحقاق',
    'assigned_to' => 'المسؤول'
];

if ($role === 'admin') {
    $departments = $conn->query("SELECT id, name FROM departments ORDER BY name ASC")->fetchAll(PDO::FETCH_ASSOC);
    $customFields = [];
} else {
    $stmt = $conn->prepare("SELECT id, name FROM departments WHERE id = :id OR parent_id = :id ORDER BY name ASC");
    $stmt->execute(['id' => $department_id]);
    $departments = $stmt->fetchAll(PDO::FETCH_ASSOC);

    $stmt2 = $conn->prepare("SELECT id, label FROM custom_fields WHERE department_id = ?");
    $stmt2->execute([$department_id]);
    $customFields = $stmt2->fetchAll(PDO::FETCH_ASSOC);
}

$projects = $conn->query("SELECT id, name FROM projects ORDER BY name ASC")->fetchAll(PDO::FETCH_ASSOC);
$types = $conn->query("SELECT id, name FROM task_types ORDER BY id ASC")->fetchAll(PDO::FETCH_ASSOC);
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <title>📊 إنشاء تقرير مخصص</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <script src="https://cdn.jsdelivr.net/npm/axios/dist/axios.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/xlsx/0.18.5/xlsx.full.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/chartjs-plugin-datalabels@2"></script>
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <link href="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/css/select2.min.css" rel="stylesheet" />
    <script src="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.min.js"></script>
    <style>
        body, a:hover {background-color: <?= $userThemeColor ?> !important;}
        nav, .select2-results__option, .select2-search, .select2-selection--multiple, li, .card {background-color: <?= $userThemeNavColor ?> !important;}
        .text, label, option, h2, li, select, input, a {color: <?= $userThemeTextColor ?> !important;}
        option {text-align: center;}
        .move-button {background-color: #4d70f4; border-radius: 15px !important;}
        .view-tasks {background-color: none !important; border: 3px solid rgb(110, 118, 56) !important; border-radius: 15px !important; color: <?= ($currentUserColorId == 3) ?  htmlspecialchars('#fff') : htmlspecialchars('#000'); ?> !important;}
        .view-tasks:hover {background-color: <?= $userThemeColor ?> !important; border: 3px solid <?= ($currentUserColorId == 3) ?  htmlspecialchars('#fff') : htmlspecialchars('#130B39'); ?> !important;}
        .group-btn.active, .group-btn.bg-green-700 {background: #22c55e !important;}
    </style>
</head>
<body>
    <?php include $_SERVER['DOCUMENT_ROOT'] . '/erp_task_system/navbar/navbar.php'; ?>
    <div class="p-10">
        <div class="card flex justify-center mt-12">
            <form id="reportForm" class="card space-y-6 bg-white p-6 rounded shadow w-full">
                <?php if (in_array($role, ['admin', 'manager'])): ?>
                    <div>
                        <label class="block mb-2 font-medium">الإدارة</label>
                        <select id="department_id" name="department_id" class="card w-full border px-3 py-2 rounded" required>
                            <option value="">اختر الإدارة</option>
                            <?php foreach ($departments as $dep): ?>
                                <option value="<?= $dep['id'] ?>"><?= htmlspecialchars($dep['name']) ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                <?php else: ?>
                    <input type="hidden" id="department_id" name="department_id" value="<?= $department_id ?>">
                <?php endif; ?>

                <div>
                    <label class="block mb-2 font-medium">المشروع</label>
                    <select id="project_id" name="project_id" class="card w-full border px-3 py-2 rounded">
                        <option value="">كل المشاريع</option>
                        <?php foreach ($projects as $proj): ?>
                            <option value="<?= $proj['id'] ?>"><?= htmlspecialchars($proj['name']) ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>

                <div>
                    <label class="block mb-2 font-medium">التصنيف</label>
                    <select id="task_type_id" name="task_type_id" class="card w-full border px-3 py-2 rounded">
                        <option value="">كل التصنيفات</option>
                        <?php foreach ($types as $type): ?>
                            <option value="<?= $type['id'] ?>"><?= htmlspecialchars($type['name']) ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>

                <div id="fieldsContainer" class="mb-6">
                    <label class="block mb-2 font-bold">📋 اختر الأعمدة:</label>
                    <div id="fieldsGrid" class="card grid grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4">
                        <?php foreach ($defaultFields as $field => $label): ?>
                            <label class="inline-flex items-center">
                                <input type="checkbox" name="fields[]" value="<?= $field ?>" class="mr-2" checked>
                                <?= $label ?>
                            </label>
                        <?php endforeach; ?>
                        <?php if ($role !== 'admin' && !empty($customFields)): ?>
                            <?php foreach ($customFields as $field): ?>
                                <label class="inline-flex items-center">
                                    <input type="checkbox" name="fields[]" value="custom_<?= $field['id'] ?>" class="mr-2">
                                    <?= htmlspecialchars($field['label']) ?>
                                </label>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </div>
                </div>

                <div class="my-2">
                    <label><input type="checkbox" id="show_totals"> عرض الإجماليات</label>
                </div>

                <div id="stageContainer" class="card mt-4 hidden">
                    <label class="card block mb-2 font-bold">🌟 اختر المرحلة (اختياري)</label>
                    <select name="stage_ids[]" id="stage_id" class="card w-full border px-3 py-2 rounded" multiple></select>
                </div>

                <div class="card grid grid-cols-2 gap-4">
                    <div>
                        <label class="block mb-2 font-medium">من تاريخ</label>
                        <input type="date" name="from_date" class="card w-full border px-3 py-2 rounded">
                    </div>
                    <div>
                        <label class="block mb-2 font-medium">إلى تاريخ</label>
                        <input type="date" name="to_date" class="card w-full border px-3 py-2 rounded">
                    </div>
                </div>

                <div class="flex gap-4">
                    <button type="submit" class=" move-button text-white px-6 py-2 rounded hover:bg-blue-700">🔍 عرض التقرير</button>
                    <button type="reset" class=" text view-tasks text-white px-4 py-2 rounded">♻️ إعادة تعيين</button>
                </div>
            </form>
        </div>

        <div id="exportButtons" class="flex gap-4 mt-8"></div>
        <div id="reportTable" class="mt-6"></div>
        <div class="flex justify-center mb-4">
            <button id="toggleChartType" type="button" class="move-button text-white px-4 py-2 rounded" style="background-color: #7be667; color:black;">🔄 تغيير نوع الرسم</button>
        </div>
        <div class="flex justify-center gap-4 my-4" id="chartGroupButtons">
            <button type="button" class="move-button group-btn text-white px-4 py-2 rounded" data-group="assigned_to">حسب المسؤول</button>
            <button type="button" class="move-button group-btn text-white px-4 py-2 rounded" data-group="project_id">حسب المشروع</button>
            <button type="button" class="move-button group-btn text-white px-4 py-2 rounded" data-group="task_type_id">حسب التصنيف</button>
        </div>
        <canvas id="reportChart" class="mt-8 bg-white p-4 rounded shadow" style="max-height: 400px;"></canvas>

<script>
let latestReportTable = null;
let chartType = 'bar';
let reportChart = null;
let chartGroupBy = 'assigned_to'; // افتراضي أول ما تفتح الصفحة

$(document).ready(function() {
    // تغيير نوع الرسم
    $('#toggleChartType').on('click', function() {
        chartType = (chartType === 'bar') ? 'pie' : 'bar';
        updateChartOnly();
    });

    // تحديث الحقول الإضافية مع تغيير الإدارة
    $('#department_id').on('change', function() {
        const depId = $(this).val();
        if (!depId) return;
        axios.get('get_fields_api.php?department_id=' + depId)
            .then(response => {
                const { fields, stages } = response.data;
                let defaultFields = {
                    'title': 'عنوان المهمة',
                    'description': 'وصف المهمة',
                    'created_at': 'تاريخ الإنشاء',
                    'due_date': 'تاريخ الاستحقاق',
                    'assigned_to': 'المسؤول'
                };
                $('#fieldsGrid').empty();
                $.each(defaultFields, function(key, value) {
                    $('#fieldsGrid').append(`<label class="card inline-flex items-center"><input type="checkbox" name="fields[]" value="${key}" class="mr-2" checked>${value}</label>`);
                });
                $.each(fields, function(index, field) {
                    $('#fieldsGrid').append(`<label class="card inline-flex items-center"><input type="checkbox" name="fields[]" value="custom_${field.id}" class="mr-2">${field.label}</label>`);
                });
                $('#stage_id').empty();
                stages.forEach(stage => {
                    $('#stage_id').append(`<option class="card" value="${stage.id}">${stage.name}</option>`);
                });
                $('#stage_id').select2({
                    placeholder: 'اختر مرحلة أو أكثر',
                    allowClear: true,
                    width: '100%'
                });
                $('#stageContainer').addClass('card').removeClass('hidden');
            });
    });

    // عرض التقرير (Submit)
    $('#reportForm').on('submit', function(e) {
        e.preventDefault();
        const form = new FormData(this);
        form.append('show_totals', document.getElementById('show_totals').checked ? 'true' : 'false');
        // حفظ آخر تجميع مختار للرسم البياني
        form.append('chart_group_by', chartGroupBy);

        axios.post('run_report_api.php', form)
            .then(response => {
                $('#reportTable').html(response.data);
                latestReportTable = document.querySelector('#reportTable table');
                $('#exportButtons').html(`
                    <button onclick="exportToExcel()" class="text-white px-4 py-1 rounded" style="background-color: #7be667; border-radius:15px ;">📅 Excel</button>
                    <button onclick="printReport()" class="bg-gray-700 text-white px-4 py-1 rounded" style="border-radius:15px;">🖨️ طباعة</button>
                `);
                setTimeout(() => {
                    updateChartOnly();
                }, 500);
            })
            .catch(() => alert('❌ حدث خطأ أثناء توليد التقرير'));
    });

    // زر اختيار التجميع أسفل الرسم البياني
    $(document).on('click', '.group-btn', function () {
        chartGroupBy = $(this).data('group');
        $('.group-btn').removeClass('bg-green-700').addClass('bg-blue-700');
        $(this).removeClass('bg-blue-700').addClass('bg-green-700');
        updateChartOnly();
    });
});

// دالة تحديث الرسم البياني فقط (من دون إعادة تحميل الجدول)
function updateChartOnly() {
    const form = new FormData(document.getElementById('reportForm'));
    form.append('chart_group_by', chartGroupBy);
    axios.post('report_chart_api.php', form)
        .then(res => {
            const { labels, counts } = res.data;
            renderChart(labels, counts);
        })
        .catch(() => console.log('❌ فشل جلب بيانات الرسم البياني'));
}

function renderChart(labels, counts) {
    const ctx = document.getElementById('reportChart').getContext('2d');
    if (reportChart) reportChart.destroy();
    reportChart = new Chart(ctx, {
        type: chartType,
        data: {
            labels: labels,
            datasets: [{
                label: 'عدد المهام',
                data: counts,
                backgroundColor: generateColors(labels.length),
                borderColor: 'rgba(0,0,0,0.1)',
                borderWidth: 1
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                legend: {display: true, position: 'top'},
                tooltip: {enabled: true},
                datalabels: {
                    display: chartType === 'bar',
                    color: '#000',
                    font: {weight: 'bold', size: 12}
                }
            },
            scales: chartType === 'bar' ? {y: {beginAtZero: true}} : {}
        },
        plugins: [ChartDataLabels]
    });
}
function generateColors(length) {
    const colors = [
        '#4F46E5','#06B6D4','#10B981','#F59E0B','#EF4444','#8B5CF6',
        '#14B8A6','#F43F5E','#3B82F6','#22C55E'
    ];
    return Array.from({length}, (_, i) => colors[i % colors.length]);
}
function exportToExcel() {
    if (!latestReportTable) return;
    const wb = XLSX.utils.table_to_book(latestReportTable, {sheet: 'تقرير'});
    if (wb.Sheets['تقرير']) wb.Sheets['تقرير']['!rtl'] = true;
    XLSX.writeFile(wb, 'تقرير.xlsx');
}
function printReport() {
    if (!latestReportTable) return;
    const printWindow = window.open('', '_blank');
    printWindow.document.write(`<html lang="ar" dir="rtl"><head><title>📋 طباعة التقرير</title>
        <style>body{font-family:'Tahoma',sans-serif;direction:rtl;padding:20px;}
        table{width:100%;border-collapse:collapse;}
        th,td{border:1px solid #000;padding:8px;text-align:center;}
        th{background-color:#f1f1f1;}@page{size:A4 landscape;margin:10mm;}</style>
        </head><body><h1>📋 تقرير المهام</h1>${latestReportTable.outerHTML}</body></html>`);
    printWindow.document.close();
    printWindow.focus();
    printWindow.print();
}
</script>
</body>
</html>