<?php
require_once '../../config/db.php';
require_once '../../auth.php';
require_once '../../authorize.php';

/* ====== حماية الجلسة ====== */
if (
  !isset($_SESSION['user']) ||
  !isset($_SESSION['user']['name']) ||
  !isset($_SESSION['user']['email']) ||
  !isset($_SESSION['user']['role']) ||
  !in_array($_SESSION['user']['role'], ['admin','manager','user'])
) {
  session_unset();
  session_destroy();
  header('Location: ../../unauthorized.php');
  exit;
}

/* ====== بيانات المستخدم والثيم ====== */
$user   = $_SESSION['user'];
$role   = $user['role'];
$department_id = (int)($user['department_id'] ?? 0);
$user_secret_level = $user['secret_level'] ?? 'normal';

$color = $user['color'] ?? [];
$userThemeColor            = $color['hex_code']         ?? '#f8fafc';
$userThemeTextColor        = $color['text_color']       ?? '#111827';
$userThemeSecondTextColor  = $color['second_text_color']?? '#6b7280';
$userThemeNavColor         = $color['nav_color']        ?? '#233a58';
$chartColor                = $color['chart_color']      ?? '#10b981';
$allColors                 = $user['colors']            ?? [];
$currentUserColorId        = $user['color_id']          ?? null;

allow_roles(['admin','manager']);

/* ====== مستويات السرّية (آمن) ====== */
$levels = ['normal'=>1,'secret'=>2,'top_secret'=>3];
$max_level = $levels[$user_secret_level] ?? 1;

/* اختيار الإدارة من القائمة */
$selected_dept_id = isset($_GET['department_id']) && $_GET['department_id'] !== ''
  ? (int)$_GET['department_id'] : null;

/* ====== دالة: كل الإدارات التابعة (شجرة) ====== */
function get_descendant_dept_ids(PDO $conn, array $rootIds): array {
  $rootIds = array_values(array_unique(array_map('intval', $rootIds)));
  if (empty($rootIds)) return [];
  $all   = [];
  $queue = $rootIds;

  while (!empty($queue)) {
    foreach ($queue as $id) {
      if (!in_array($id, $all, true)) $all[] = $id;
    }
    $ph = implode(',', array_fill(0, count($queue), '?'));
    $stmt = $conn->prepare("SELECT id FROM departments WHERE parent_id IN ($ph)");
    $stmt->execute($queue);
    $children = array_map('intval', $stmt->fetchAll(PDO::FETCH_COLUMN));
    $queue = array_values(array_diff($children, $all));
  }
  return $all;
}

/* ====== تحميل الإدارات المتاحة للمستخدم (لعرض الأسماء بالقائمة وغيرها) ====== */
if ($role === 'manager') {
  $stmt = $conn->prepare("SELECT id, name FROM departments WHERE manager_id = ? OR parent_id = ? OR id = ?");
  $stmt->execute([$user['id'], $department_id, $department_id]);
  $departments = $stmt->fetchAll(PDO::FETCH_ASSOC);
} else {
  $departments = $conn->query("SELECT id, name FROM departments")->fetchAll(PDO::FETCH_ASSOC);
}

/* ====== قائمة الإدارات المُطبّق عليها الفلترة (الإدارة + التابعين) ====== */
if ($role === 'manager') {
  // جذور المدير
  $stmt = $conn->prepare("SELECT id FROM departments WHERE manager_id = ? OR parent_id = ? OR id = ?");
  $stmt->execute([$user['id'], $department_id, $department_id]);
  $managerDepartments = array_map('intval', $stmt->fetchAll(PDO::FETCH_COLUMN));
  $roots = !empty($managerDepartments) ? $managerDepartments : [$department_id ?: 0];
  $filterDeptIds = get_descendant_dept_ids($conn, $roots);
} elseif ($role === 'admin' && $selected_dept_id) {
  $filterDeptIds = get_descendant_dept_ids($conn, [$selected_dept_id]);
} else {
  $filterDeptIds = array_map('intval', array_column($departments, 'id'));
}
if (empty($filterDeptIds)) $filterDeptIds = [-1]; // يمنع IN () فارغ

/* ====== تجميعة (الإدارة × المراحل) مع فلترة السرّية ====== */
$query = "
  SELECT 
    d.id AS department_id,
    d.name AS department_name,
    s.id AS stage_id,
    s.name AS stage_name,
    s.order_number,
    COUNT(t.id) AS total
  FROM departments d
  LEFT JOIN stages s ON s.department_id = d.id
  LEFT JOIN tasks t ON t.stage_id = s.id
    AND t.is_deleted = 0
    AND (CASE t.secret_level
          WHEN 'normal' THEN 1
          WHEN 'secret' THEN 2
          WHEN 'top_secret' THEN 3
        END) <= ?
";
$params = [$max_level];

$phDepts = implode(',', array_fill(0, count($filterDeptIds), '?'));
$query  .= " WHERE d.id IN ($phDepts) ";
$params = array_merge($params, $filterDeptIds);

$query .= "
  GROUP BY d.id, d.name, s.id, s.name, s.order_number
  ORDER BY d.name, s.order_number
";

$stmt = $conn->prepare($query);
$stmt->execute($params);
$stats = $stmt->fetchAll(PDO::FETCH_ASSOC);

/* تجهيز بيانات الرسم حسب الإدارة */
$data = [];
foreach ($departments as $dept) {
  $data[(int)$dept['id']]['name']   = $dept['name'];
  $data[(int)$dept['id']]['stages'] = [];
}
foreach ($stats as $row) {
  $deptId = (int)$row['department_id'];
  $stage  = $row['stage_name'] ?? 'غير محددة';
  $total  = (int)$row['total'];
  if (!isset($data[$deptId])) $data[$deptId] = ['name'=>'', 'stages'=>[]];
  $data[$deptId]['stages'][$stage] = $total;
}

/* ====== مراحل محتسبة + نسبة إنجاز الإدارات ====== */
$countedStages     = [];
$totalCountedTasks = 0;
$totalTasksInDept  = 0;

/* IDs المراحل المحتسبة داخل الإدارات المُ فلترة */
$allCountedStageIds = [];
if (!empty($filterDeptIds)) {
  $ph = implode(',', array_fill(0, count($filterDeptIds), '?'));
  $stmt = $conn->prepare("SELECT id FROM stages WHERE is_counted = 1 AND department_id IN ($ph)");
  $stmt->execute($filterDeptIds);
  $allCountedStageIds = array_map('intval', $stmt->fetchAll(PDO::FETCH_COLUMN));
}

/* حساب عدد المهام في كل المراحل المحتسبة (إجمالي) */
if (!empty($allCountedStageIds)) {
  $phCounted = implode(',', array_fill(0, count($allCountedStageIds), '?'));
  $paramsWithLevel = array_merge($allCountedStageIds, [$max_level]);

  $countedTasksStmt = $conn->prepare("
    SELECT COUNT(*)
    FROM tasks
    WHERE stage_id IN ($phCounted)
      AND is_deleted = 0
      AND (CASE secret_level
            WHEN 'normal' THEN 1
            WHEN 'secret' THEN 2
            WHEN 'top_secret' THEN 3
          END) <= ?
  ");
  $countedTasksStmt->execute($paramsWithLevel);
  $totalCountedTasks = (int)$countedTasksStmt->fetchColumn();
}

/* نسبة إنجاز كل إدارة (يُحسب فقط للإدارات ضمن الفلترة) */
$departmentProgress = [];
foreach ($departments as $dept) {
  $deptId = (int)$dept['id'];
  if (!in_array($deptId, $filterDeptIds, true)) continue;

  // مراحل محتسبة لهذه الإدارة
  $stageStmt = $conn->prepare("SELECT id FROM stages WHERE is_counted = 1 AND department_id = ?");
  $stageStmt->execute([$deptId]);
  $stageIds = array_map('intval', $stageStmt->fetchAll(PDO::FETCH_COLUMN));

  if (!empty($stageIds)) {
    $ph   = implode(',', array_fill(0, count($stageIds), '?'));
    $paramsWithLevel = array_merge($stageIds, [$max_level]);

    $countedStmt = $conn->prepare("
      SELECT COUNT(*) FROM tasks
      WHERE stage_id IN ($ph)
        AND is_deleted = 0
        AND (CASE secret_level
              WHEN 'normal' THEN 1
              WHEN 'secret' THEN 2
              WHEN 'top_secret' THEN 3
            END) <= ?
    ");
    $countedStmt->execute($paramsWithLevel);
    $countedTasks = (int)$countedStmt->fetchColumn();

    $totalStmt = $conn->prepare("
      SELECT COUNT(*) FROM tasks
      WHERE department_id = ?
        AND is_deleted = 0
        AND (CASE secret_level
              WHEN 'normal' THEN 1
              WHEN 'secret' THEN 2
              WHEN 'top_secret' THEN 3
            END) <= ?
    ");
    $totalStmt->execute([$deptId, $max_level]);
    $totalTasks = (int)$totalStmt->fetchColumn();

    $percentage = ($totalTasks > 0) ? round(($countedTasks / $totalTasks) * 100, 2) : 0;

    $departmentProgress[] = [
      'name'       => $dept['name'],
      'percentage' => $percentage
    ];
  }
}

/* إجمالي المهام داخل الإدارات المُ فلترة (للدونات الكلية) */
if (!empty($filterDeptIds)) {
  $ph = implode(',', array_fill(0, count($filterDeptIds), '?'));
  $sqlTotal = "
    SELECT COUNT(*) FROM tasks
    WHERE department_id IN ($ph)
      AND is_deleted = 0
      AND (CASE secret_level
            WHEN 'normal' THEN 1
            WHEN 'secret' THEN 2
            WHEN 'top_secret' THEN 3
          END) <= ?
  ";
  $totalTasksStmt = $conn->prepare($sqlTotal);
  $totalTasksStmt->execute(array_merge($filterDeptIds, [$max_level]));
  $totalTasksInDept = (int)$totalTasksStmt->fetchColumn();
}
$progressPercentage = ($totalTasksInDept > 0) ? round(($totalCountedTasks / $totalTasksInDept) * 100, 2) : 0;

/* ================== مشاريع: إنجاز كل مشروع (مكتمل/غير مكتمل) ================== */
/* الإدارات للمشاريع = الفلترة الموحّدة مع التابعين */
$availableDeptIds = $filterDeptIds;

$projectLabels         = [];
$projectCompletedData  = [];
$projectPendingData    = [];
$projectPercentages    = [];

if (!empty($availableDeptIds)) {
  // IDs المراحل الخضراء داخل الإدارات المحددة
  $deptPH = implode(',', array_fill(0, count($availableDeptIds), '?'));
  $stmtCompletedStages = $conn->prepare("
      SELECT id FROM stages
      WHERE department_id IN ($deptPH)
        AND LOWER(color) = LOWER('#10b981')
  ");
  $stmtCompletedStages->execute($availableDeptIds);
  $completedStageIds = array_map('intval', $stmtCompletedStages->fetchAll(PDO::FETCH_COLUMN));
  if (empty($completedStageIds)) $completedStageIds = [0];
  $completedPH = implode(',', array_fill(0, count($completedStageIds), '?'));

  // الإحصائية: فلترة الإدارة على المهام t.department_id
  $sqlProjects = "
    SELECT
      p.id,
      p.name,
      COUNT(t.id) AS total_tasks,
      SUM(
        CASE
          WHEN t.id IS NOT NULL
               AND t.is_deleted = 0
               AND t.stage_id IN ($completedPH)
               AND (CASE t.secret_level
                      WHEN 'normal' THEN 1
                      WHEN 'secret' THEN 2
                      WHEN 'top_secret' THEN 3
                   END) <= ?
          THEN 1 ELSE 0
        END
      ) AS completed_tasks
    FROM projects p
    LEFT JOIN tasks t
      ON t.project_id = p.id
     AND t.is_deleted = 0
     AND t.department_id IN ($deptPH)
     AND (CASE t.secret_level
            WHEN 'normal' THEN 1
            WHEN 'secret' THEN 2
            WHEN 'top_secret' THEN 3
         END) <= ?
    GROUP BY p.id, p.name
    HAVING COUNT(t.id) > 0
    ORDER BY p.name ASC
  ";
  $paramsProjects = array_merge($completedStageIds, [$max_level], $availableDeptIds, [$max_level]);

  $stmtProjects = $conn->prepare($sqlProjects);
  $stmtProjects->execute($paramsProjects);
  $rows = $stmtProjects->fetchAll(PDO::FETCH_ASSOC);

  foreach ($rows as $r) {
    $total     = (int)$r['total_tasks'];
    $completed = (int)$r['completed_tasks'];
    $pending   = max(0, $total - $completed);
    $pct       = ($total > 0) ? round(($completed / $total) * 100, 2) : 0;

    $projectLabels[]        = $r['name'];
    $projectCompletedData[] = $completed;
    $projectPendingData[]   = $pending;
    $projectPercentages[]   = $pct;
  }
}
/* ================== خط زمني للمشاريع (من جدول projects فقط) ================== */
/* خمس سنوات من أول السنة الحالية — بدون فلترة إدارة */
$yearNow    = (int)date('Y');
$startRange = $yearNow . '-01-01';
$endRange   = ($yearNow + 5) . '-12-31';

$ganttTasks = [];

/* استعلام محصّن ضد القيم الفارغة/الصفرية */
$sqlTimeline = "
  SELECT p.id, p.name, p.start_date, p.end_date
  FROM projects p
  WHERE
    COALESCE(
      STR_TO_DATE(NULLIF(NULLIF(SUBSTRING_INDEX(p.end_date,   ' ', 1), ''), '0000-00-00'), '%Y-%m-%d'),
      STR_TO_DATE('1000-01-01', '%Y-%m-%d')
    ) >= STR_TO_DATE(?, '%Y-%m-%d')
    AND
    COALESCE(
      STR_TO_DATE(NULLIF(NULLIF(SUBSTRING_INDEX(p.start_date, ' ', 1), ''), '0000-00-00'), '%Y-%m-%d'),
      STR_TO_DATE('9999-12-31', '%Y-%m-%d')
    ) <= STR_TO_DATE(?, '%Y-%m-%d')
  ORDER BY p.start_date ASC
";
$stmtTimeline = $conn->prepare($sqlTimeline);
$stmtTimeline->execute([$startRange, $endRange]);
$timelineProjects = $stmtTimeline->fetchAll(PDO::FETCH_ASSOC);

/* فالباك بسيط — لو ما رجّع شيء، جرّب بدون فلترة التاريخ (تشخيص) */
if (empty($timelineProjects)) {
  $stmtAlt = $conn->query("
    SELECT id, name, start_date, end_date
    FROM projects
    WHERE (start_date IS NOT NULL AND start_date <> '' AND start_date <> '0000-00-00')
      AND (end_date   IS NOT NULL AND end_date   <> '' AND end_date   <> '0000-00-00')
    ORDER BY start_date ASC
    LIMIT 100
  ");
  $timelineProjects = $stmtAlt->fetchAll(PDO::FETCH_ASSOC);
}

/* حوّل النتائج لمهام Gantt */
foreach ($timelineProjects as $p) {
  $start = substr((string)$p['start_date'], 0, 10);
  $end   = substr((string)$p['end_date'],   0, 10);
  if (!$start || !$end) continue;

  $ganttTasks[] = [
    'id'       => 'proj_' . (int)$p['id'],
    'name'     => (string)($p['name'] ?: 'مشروع #' . (int)$p['id']),
    'start'    => $start,   // YYYY-MM-DD
    'end'      => $end,     // YYYY-MM-DD
    'progress' => 0,
  ];
}

/* مهمة مرجعية لضبط الشبكة على 5 سنوات (مخفية بالـ CSS) */
$ganttTasks[] = [
  'id'           => 'range_anchor',
  'name'         => 'نطاق 5 سنوات',
  'start'        => $startRange,
  'end'          => $endRange,
  'progress'     => 0,
  'custom_class' => 'gantt-range-anchor'
];

?>


<!DOCTYPE html>
<html lang="ar" dir="rtl">

<head>
  <meta charset="UTF-8">
  <title>لوحة تحكم</title>
  <script src="https://cdn.tailwindcss.com"></script>
  <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>

  <!-- Include Select2 CSS -->
  <link href="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/css/select2.min.css" rel="stylesheet" />

  <!-- Include jQuery (required by Select2) -->
  <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>

  <!-- Include Select2 JS -->
  <script src="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.min.js"></script>

  <style>
    /* Custom styles to make the Select2 dropdown larger */
    .select2-container .select2-selection--single {
      height: 48px;
      /* Increase height of the select box */
      line-height: 48px;
      /* Align text vertically */
      font-size: 12px;
      /* Larger font size for selected item */
    }

    .select2-container--default .select2-selection--single {
      background-color: <?php echo $userThemeNavColor; ?> !important;
      border: none;
    }

    .select2-container--default .select2-selection--single .select2-selection__rendered {
      line-height: 48px;
      color: <?php echo $userThemeTextColor; ?> !important;
      /* Ensure text aligns with height */
    }

    .select2-container--default .select2-selection--single .select2-selection__arrow {
      height: 48px;
      /* Adjust arrow height to match */
    }

    .select2-dropdown {
      font-size: 12px;
      /* Larger font size for dropdown options */
    }

    .select2-results__option {
      padding: 12px;
      /* More padding for larger options */
      line-height: 24px;
      /* Adjust line height for larger square and text */
    }

    /* Optional: Ensure the select takes full width or a specific width */
    .select2-container {
      width: 100% !important;
      /* Full width, or set a specific width like 300px */
    }
  </style>
  <style id="asraa-theme">
:root{
  --asraa-primary:        <?= htmlspecialchars($chartColor, ENT_QUOTES, 'UTF-8') ?>;
  --asraa-bg:             <?= htmlspecialchars($userThemeColor, ENT_QUOTES, 'UTF-8') ?>;
  --asraa-nav:            <?= htmlspecialchars($userThemeNavColor, ENT_QUOTES, 'UTF-8') ?>;
  --asraa-text:           <?= htmlspecialchars($userThemeTextColor, ENT_QUOTES, 'UTF-8') ?>;
  --asraa-text-muted:     <?= htmlspecialchars($userThemeSecondTextColor, ENT_QUOTES, 'UTF-8') ?>;
  --asraa-success:        #10b981; /* أخضر إنجاز */
  --asraa-warning:        #f59e0b; /* أصفر */
  --asraa-danger:         #ef4444; /* أحمر */
  --asraa-neutral:        #e5e7eb; /* رمادي */
  --asraa-grid:           #e9ecef; /* خطوط الشبكة */
  --asraa-today:          #94a3b8; /* خط اليوم */
}

/* صندوق الـ Gantt */
.gantt-box{ overflow-x:auto; direction:ltr; }

/* رأس السنة/الشهور أصغر */
#ganttProjects .upper-text{ font-size:10px!important; fill:var(--asraa-text-muted)!important; }
#ganttProjects .lower-text{ font-size:9px!important;  fill:var(--asraa-text-muted)!important; }

/* اسم المشروع داخل الشريط */
#ganttProjects .bar-label{
  display:block; font-size:10px!important; fill:var(--asraa-text-muted)!important; font-weight:600;
}

/* شبكة وألوان Gantt */
#ganttProjects .grid-row rect{ fill:#fff; }
#ganttProjects .grid-header{ fill:#fff; }
#ganttProjects .tick{ stroke:var(--asraa-grid); }
#ganttProjects .grid-row:nth-child(odd) rect{ fill:#fafafa; }
#ganttProjects .arrow{ stroke:var(--asraa-primary); }
#ganttProjects .bar{ rx:6; ry:6; }
#ganttProjects .bar-progress{ fill:var(--asraa-primary)!important; }
#ganttProjects .today-highlight{ fill:var(--asraa-today); opacity:.15; }

/* أسطورة */
.gantt-legend{ display:flex; gap:.75rem; align-items:center; margin-bottom:.5rem; color:var(--asraa-text-muted); font-size:12px; }
.gantt-dot{ width:10px; height:10px; border-radius:999px; display:inline-block; }
.gantt-dot.primary{ background:var(--asraa-primary); }
.gantt-dot.success{ background:var(--asraa-success); }
.gantt-dot.warn{    background:var(--asraa-warning); }
.gantt-dot.danger{  background:var(--asraa-danger); }
.gantt-dot.neutral{ background:var(--asraa-neutral); }

/* إخفاء مهمة anchor */
.gantt-range-anchor .bar{ opacity:0; height:0; }
</style>

</head>


<?php include $_SERVER['DOCUMENT_ROOT'] . '/erp_task_system/navbar/navbar.php'; ?>
<div class="h-20"></div>

<style>
  body,
  a:hover {
    background-color: <?php echo $userThemeColor; ?> !important;
  }

  nav,
  .card {
    background-color: <?php echo $userThemeNavColor; ?> !important;
  }

  .text,
  label,
  option,
  h2,
  select,
  a {
    color: <?php echo $userThemeTextColor; ?> !important;
  }

  option {
    text-align: center;
  }

  .secondary-text {
    color: <?php echo $userThemeSecondTextColor; ?> !important;
  }

  .gr {
    color: #7be667;
  }

  .bl {
    color: #4d70f4;
  }

  .view-tasks {
    background-color: none !important;
    /* background-color: rgb(19, 11, 57) !important; */
    border: 3px solid <?php echo $userThemeColor; ?> !important;
    border-radius: 15px !important;
    color:<?= ($currentUserColorId == 3 )?  htmlspecialchars( '#ffffff'): htmlspecialchars('#000000'); ?> !important;

  }

  .text-button {
    color: #ffffff !important;
  }

  .move-button {
    background-color: #4d70f4;
    border-radius: 15px !important;
  }

  .move-button:hover {
    background-color: rgb(19, 11, 57) !important;
  }

  .view-tasks:hover {
    background-color: <?php echo $userThemeColor; ?> !important;
    border: 3px solid rgb(19, 11, 57) !important;
    color:<?= ($currentUserColorId == 3 )?  htmlspecialchars( '#ffffff'): htmlspecialchars('#000000'); ?> !important;

  }

 
</style>
<?php if (!empty($departmentProgress)): ?>

<style>
  <?php foreach ($departmentProgress as $index => $dept): ?>

  /* CSS class for legend labels of each chart */
  .chart-legend-<?= $index ?>.chartjs-legend-text {
    color: <?= htmlspecialchars($userThemeSecondTextColor) ?> !important;
  }

  <?php endforeach; ?>
</style>
<?php endif; ?>


<body class="bg-gray-100">



  <!-- المحتوى -->
  <?php if (!empty($_SESSION['success'])): ?>
    <div class="bg-green-100 gr text-center px-4 py-3 rounded mb-4 shadow max-w-3xl mx-auto">
      <?= $_SESSION['success'] ?>
      <?php unset($_SESSION['success']); ?>
    </div>
  <?php endif; ?>

  <div class="max-w-6xl mx-auto mt-6 px-4">


    <?php if ($role === 'admin') { ?>
      <form method="GET" class="mb-6">
        <label class="block text-sm mb-1">اختر الإدارة:</label>
        <select name="department_id" onchange="this.form.submit()" class="card border rounded px-3 py-2 w-full md:w-1/3">
          <option value="">-- كل الإدارات --</option>
          <?php foreach ($departments as $dept): ?>
            <option value="<?= $dept['id'] ?>" <?= $selected_dept_id == $dept['id'] ? 'selected' : '' ?>>
              <?= htmlspecialchars($dept['name']) ?>
            </option>
          <?php endforeach; ?>
        </select>
  <?php $helpHref = file_exists(__DIR__.'/help_dashboard.php') ? 'help_dashboard.php' : 'help.php'; ?>
  <a href="<?= htmlspecialchars($helpHref) ?>" target="_blank" rel="noopener"
     class="move-button text-button text-white px-4 py-2 rounded no-print inline-flex items-center gap-1
            md:rtl:mr-auto md:ltr:ml-auto"
     title="تعليمات لوحة التحكّم">
    ❓ تعليمات
  </a>

      </form>
    <?php } ?>
    

<!-- بطاقة نسبة إنجاز المشاريع (مع الكانفس) - خليه دايم يطلع -->
<div class="max-w-6xl mx-auto mt-6 px-4">
  <div class="card rounded shadow p-6 mb-6">
    <h2 class="text-xl font-bold mb-2">📊 نسبة إنجاز المشاريع</h2>
    <p class="secondary-text mb-4" id="projectsEmptyMsg" style="display:none;">
      لا توجد بيانات معاملات لهذه الإدارة.
    </p>
    <canvas id="projectsProgressChart" height="260" style="max-height:300px;"></canvas>
  </div>
</div>

<script>
  (function(){
    const labels  = <?= json_encode($projectLabels, JSON_UNESCAPED_UNICODE) ?>;
    const done    = <?= json_encode($projectCompletedData) ?>;
    const pending = <?= json_encode($projectPendingData) ?>;

    const canvas = document.getElementById('projectsProgressChart');
    const emptyMsg = document.getElementById('projectsEmptyMsg');

    // لا ترسم إذا ما فيه بيانات
    if (!labels || labels.length === 0) {
      if (canvas) canvas.style.display = 'none';
      if (emptyMsg) emptyMsg.style.display = 'block';
      return;
    } else {
      if (emptyMsg) emptyMsg.style.display = 'none';
      if (canvas) canvas.style.display = 'block';
    }

    new Chart(canvas, {
      type: 'bar',
      data: {
        labels: labels,
datasets: [
  {
    label: 'مكتملة',
    data: done,
    backgroundColor: getComputedStyle(document.documentElement).getPropertyValue('--asraa-success').trim(),
    borderWidth: 0,
    stack: 'total'
  },
  {
    label: 'غير مكتملة',
    data: pending,
    backgroundColor: getComputedStyle(document.documentElement).getPropertyValue('--asraa-neutral').trim(),
    borderWidth: 0,
    stack: 'total'
  }
]
      },
      options: {
        responsive: true,
        maintainAspectRatio: false,
        plugins: {
          legend: {
            position: 'bottom',
            labels: { color: '<?= htmlspecialchars($userThemeSecondTextColor, ENT_QUOTES, "UTF-8") ?>' }
          },
          tooltip: {
            callbacks: {
              afterBody: function(items) {
                const i = items[0].dataIndex;
                const c = done[i], p = pending[i], t = c + p;
                const pct = t > 0 ? Math.round((c / t) * 100) : 0;
                return 'نسبة الإنجاز: ' + pct + '%';
              }
            },
            titleColor: '<?= htmlspecialchars($userThemeSecondTextColor, ENT_QUOTES, "UTF-8") ?>',
            bodyColor:  '<?= htmlspecialchars($userThemeSecondTextColor, ENT_QUOTES, "UTF-8") ?>'
          }
        },
        scales: {
          x: { stacked: true, ticks: { color: '<?= htmlspecialchars($userThemeSecondTextColor, ENT_QUOTES, "UTF-8") ?>' } },
          y: { stacked: true, beginAtZero: true, ticks: { color: '<?= htmlspecialchars($userThemeSecondTextColor, ENT_QUOTES, "UTF-8") ?>' } }
        }
      }
    });
  })();
</script>

<!-- بطاقة الخط الزمني للمشاريع (5 سنوات) -->
<div class="max-w-6xl mx-auto mt-6 px-4">
  <div class="card rounded shadow p-6 mb-6">
    <h2 class="text-xl font-bold mb-2">🗓️ الخط الزمني للمشاريع (5 سنوات)</h2>
    <p class="secondary-text mb-4">من جدول المشاريع: تاريخ البداية والنهاية.</p>

    <div class="gantt-legend">
      <span class="gantt-dot success"></span> إنجاز ≥ 80%
      <span class="gantt-dot warn"></span> 40% – 79%
      <span class="gantt-dot danger"></span> < 40%
      <span class="gantt-dot primary"></span> لون الثيم
    </div>

    <div id="ganttProjects" class="gantt-box"></div>
    <p id="ganttEmptyMsg" class="secondary-text mt-3" style="display:none;">ما فيه مشاريع ضمن النطاق.</p>
  </div>
</div>

<link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/frappe-gantt@0.6.1/dist/frappe-gantt.css">
<script src="https://cdn.jsdelivr.net/npm/frappe-gantt@0.6.1/dist/frappe-gantt.min.js"></script>

<script>
(function(){
  const tasks  = <?= json_encode($ganttTasks, JSON_UNESCAPED_UNICODE) ?>;
  const box    = document.getElementById('ganttProjects');
  const empty  = document.getElementById('ganttEmptyMsg');

  if (!tasks || tasks.length === 0) {
    box.style.display = 'none';
    empty.style.display = 'block';
    return;
  } else {
    empty.style.display = 'none';
    box.style.display = 'block';
  }

  const gantt = new Gantt("#ganttProjects", tasks, {
    view_mode: 'Month',
    bar_height: 24,
    column_width: 32,
    header_height: 36,
    padding: 12,
    custom_popup_html: (task)=>{
      const s = task._start ? task._start.toISOString().slice(0,10) : (task.start||'');
      const e = task._end   ? task._end.toISOString().slice(0,10)   : (task.end||'');
      return `<div class="card p-3" dir="rtl" style="min-width:220px">
        <div class="text-base font-bold mb-1">المشروع: ${task.name}</div>
        <div class="text-sm">البداية: ${s}</div>
        <div class="text-sm">النهاية: ${e}</div>
        <div class="text-sm">الإنجاز: ${task.progress||0}%</div>
      </div>`;
    }
  });

  /* الشهور 01-12 بدل الأسماء */
  (function replaceMonthNames(){
    const map = {January:'01',February:'02',March:'03',April:'04',May:'05',June:'06',July:'07',August:'08',September:'09',October:'10',November:'11',December:'12'};
    document.querySelectorAll('#ganttProjects .lower-text, #ganttProjects .lower-text tspan')
      .forEach(el=>{ const t=(el.textContent||'').trim(); if(map[t]) el.textContent = map[t]; });
  })();

  /* تلوين البارات حسب نسبة الإنجاز */
  const pctByName = <?= json_encode(array_combine($projectLabels, $projectPercentages), JSON_UNESCAPED_UNICODE) ?> || {};
  const cs  = getComputedStyle(document.documentElement);
  const ok  = cs.getPropertyValue('--asraa-success').trim();
  const mid = cs.getPropertyValue('--asraa-warning').trim();
  const bad = cs.getPropertyValue('--asraa-danger').trim();
  const pri = cs.getPropertyValue('--asraa-primary').trim();

  document.querySelectorAll('#ganttProjects .bar-group').forEach(g=>{
    const bar   = g.querySelector('.bar');
    const label = g.querySelector('.bar-label');
    if (!bar) return;
    let name = label ? label.textContent.trim() : '';
    if (!name && g.getAttribute('data-id')) name = g.getAttribute('data-id');
    const pct = pctByName[name];
    bar.style.fill = (pct==null) ? pri : (pct>=80? ok : pct>=40? mid : bad);
  });

  /* ابدأ التمرير من اليسار (أصغر سنة) */
  const scroller = document.querySelector('.gantt-box') || box;
  const scrollToLeft = el => el.scrollLeft = 0;
  requestAnimationFrame(()=> scrollToLeft(scroller));
  setTimeout(()=> scrollToLeft(scroller), 60);
  window.addEventListener('resize', ()=> scrollToLeft(scroller));
})();
</script>


    <?php if (!empty($departmentProgress)): ?>
      <div class=" grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6 mb-8">
        <?php foreach ($departmentProgress as $index => $dept): ?>
          <div class="card rounded shadow p-6">
            <h2 class="text-lg font-bold text-center mb-4">📈 نسبة الإنجاز - <?= htmlspecialchars($dept['name']) ?></h2>
            <div class="flex justify-center">
              <canvas id="progressChart_<?= $index ?>" width="120" height="150" style="max-width:180px; max-height:180px;"></canvas>
            </div>
            <p class="text-center mt-2 text-base font-bold secondary-text">
              تم إنجاز <?= $dept['percentage'] ?>٪ من المعاملات.
            </p>
          </div>
        <?php endforeach; ?>
      </div>

      <script>
        <?php foreach ($departmentProgress as $index => $dept): ?>
          new Chart(document.getElementById('progressChart_<?= $index ?>'), {
            type: 'doughnut',
            data: {
              labels: ['منجز', 'متبقي'],
              datasets: [{
                data: [<?= $dept['percentage'] ?>, <?= 100 - $dept['percentage'] ?>],
                backgroundColor: ['<?= $chartColor ?>', '#E5E7EB'],
                borderWidth: 1
              }]
            },
            options: {
              plugins: {
                legend: {
                  position: 'bottom',
                  labels: {
                    font: {
                      size: 14 // Optional: Adjust font size if needed
                    },
                    color: '<?= $userThemeSecondTextColor ?>' // Set the desired label color (e.g., black)
                  }
                },
                tooltip: {
                  callbacks: {
                    label: function(context) {
                      return context.label + ': ' + context.parsed + '%';
                    }
                  }
                }
              },
              cutout: '70%'
            }
          });
        <?php endforeach; ?>
      </script>
    <?php endif; ?>



    <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
      <?php foreach ($data as $id => $info):
        $chartId = 'chart_' . $id;
        $stages = $info['stages'];
        $total = array_sum($stages);
      ?>
        <div class="card rounded shadow p-4">
          <h2 class="text-xl font-semibold mb-3">🏢 <?= htmlspecialchars($info['name']) ?> (<?= $total ?> معاملة)</h2>
          <canvas id="<?= $chartId ?>" height="270" style="max-height: 270px;"></canvas>
          <div class="flex items-center gap-2 mt-4">
            <form method="POST" action="../../stages/run_auto_move.php" onsubmit="return confirm('هل أنت متأكد من تنفيذ النقل التلقائي لهذه الإدارة؟');">
              <input type="hidden" name="department_id" value="<?= $id ?>">
              <button type="submit" class=" move-button text-button   text-white px-4 py-2 rounded shadow">
                🚀 تنفيذ النقل التلقائي
              </button>
            </form>

            <a href="../list.php?department_id=<?= $id ?>" class="text view-tasks text-white px-4 py-2 rounded shadow">
              📋 عرض المعاملات
            </a>
          </div>


          <script>
            new Chart(document.getElementById("<?= $chartId ?>"), {
              type: 'bar',
              data: {
                labels: <?= json_encode(array_keys($stages)) ?>,
                datasets: [{
                  label: 'عدد المعاملات',
                  backgroundColor: '<?= $chartColor ?>',
                  borderColor: 'rgb(255, 255, 255)',
                  borderWidth: 2, // Border thickness in pixels
                  data: <?= json_encode(array_values($stages)) ?>
                }]
              },
              options: {
                responsive: true,
                plugins: {
                  legend: {
                    display: false
                  },
                  tooltip: {
                    callbacks: {
                      label: function(context) {
                        return context.dataset.label + ': ' + context.parsed.y;
                      }
                    },
                    titleColor: '<?= htmlspecialchars($userThemeSecondTextColor, ENT_QUOTES, 'UTF-8') ?>',
                    bodyColor: '<?= htmlspecialchars($userThemeSecondTextColor, ENT_QUOTES, 'UTF-8') ?>'
                  }
                },
                scales: {
                  x: {
                    ticks: {
                      color: '<?= htmlspecialchars($userThemeSecondTextColor, ENT_QUOTES, 'UTF-8') ?>'
                    }
                  },
                  y: {
                    beginAtZero: true,
                    ticks: {
                      color: '<?= htmlspecialchars($userThemeSecondTextColor, ENT_QUOTES, 'UTF-8') ?>'
                    }
                  }
                }
              }
            });
          </script>
        </div>
      <?php endforeach; ?>
    </div>

  </div>



</body>

</html>