<?php
session_start();
require_once '../config/db.php';
require_once '../authorize.php';

ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// السماح فقط للمدير والمدير العام بإضافة مستخدمين
if (!isset($_SESSION['user']) || !in_array($_SESSION['user']['role'] ?? '', ['admin','manager'])) {
  header('Location: ../unauthorized.php'); exit;
}

$user = $_SESSION['user'] ?? [];
$color = $user['color'] ?? ['hex_code'=>'#f8fafc','text_color'=>'#111827','nav_color'=>'#233a58','second_text_color'=>'#94a3b8'];
$userThemeColor            = $color['hex_code'];
$userThemeTextColor        = $color['text_color'];
$userThemeSecondTextColor  = $color['second_text_color'];
$userThemeNavColor         = $color['nav_color'] ?? '#233a58';

// قوائم مساعدة
$departments   = $conn->query("SELECT id, name FROM departments ORDER BY name ASC")->fetchAll(PDO::FETCH_ASSOC);
$work_shifts   = $conn->query("SELECT id, name, start_time, end_time, type FROM work_shifts")->fetchAll(PDO::FETCH_ASSOC);
$locations     = $conn->query("SELECT id, name FROM locations ORDER BY name ASC")->fetchAll(PDO::FETCH_ASSOC);
$salaryComponents = $conn->query("
  SELECT id, name, code, type
  FROM salary_components
  WHERE is_active = 1
  ORDER BY type, id
")->fetchAll(PDO::FETCH_ASSOC);
$projects = $conn->query("
  SELECT id, name
  FROM projects
  /* WHERE is_active = 1 */
  ORDER BY name ASC
")->fetchAll(PDO::FETCH_ASSOC);

// رقم وظيفي مقترح
$startAt = 1001;
try {
  $max = (int)$conn->query("
    SELECT COALESCE(MAX(CAST(employee_number AS UNSIGNED)),0)
    FROM users
    WHERE employee_number REGEXP '^[0-9]+$'
  ")->fetchColumn();
  if ($max < $startAt - 1) $max = $startAt - 1;
  $nextEmpNumber = (string)($max + 1);
} catch (Throwable $e) {
  $nextEmpNumber = '';
}
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
  <meta charset="UTF-8">
  <title>إضافة مستخدم</title>
  <script src="https://cdn.tailwindcss.com"></script>
  <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
  <link href="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/css/select2.min.css" rel="stylesheet" />
  <script src="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.min.js"></script>
  <style>
  
    body{ background: <?= $userThemeColor ?> }
    .card{ background: <?= $userThemeNavColor ?> }
    .text, label, h3, h2, th, td, select, input, textarea { color: <?= $userThemeTextColor ?> }
    .step-btn{ border:1px solid #cbd5e1; border-radius:.375rem; padding:.4rem .8rem; background:#fff; color:#111827; }
    .step-btn.active{ background:#2563eb; color:#fff; border-color:#2563eb; }
    .step-btn[disabled]{ opacity:.45; cursor:not-allowed; }
    .seg-btn{ border:1px solid #cbd5e1; border-radius:.375rem; padding:.25rem .75rem; background:#fff; color:#111827; }
    .seg-btn.active{ background:#2563eb; color:#fff; border-color:#2563eb; }
    #wizardBar{ position:sticky; bottom:0; z-index:30; background:rgba(17,24,39,.75); backdrop-filter: blur(6px); border-top:1px solid rgba(255,255,255,.1); }
  </style>
</head>
<body class="text-right">
  <?php include $_SERVER['DOCUMENT_ROOT'] . '/erp_task_system/navbar/navbar.php'; ?>

  <div class="max-w-5xl mx-auto space-y-6">

    <!-- Stepper -->
    <div id="stepper" class="flex flex-wrap gap-2">
      <button class="step-btn" data-step="1">١. الأساسية</button>
      <button class="step-btn" data-step="2">٢. العنوان والاتصال</button>
      <button class="step-btn" data-step="3">٣. الهوية والجواز</button>
      <button class="step-btn" data-step="4">٤. الحساب البنكي</button>
      <button class="step-btn" data-step="5" disabled>٥. المواقع</button>
      <button class="step-btn" data-step="6" disabled>٦. الطوارئ</button>
      <button class="step-btn" data-step="7" disabled>٧. العائلة</button>
      <button class="step-btn" data-step="8" disabled>٨. المؤهلات</button>
      <button class="step-btn" data-step="9" id="btn-salary" disabled>٩. الراتب</button>
      <button class="step-btn" data-step="10" id="btn-contracts" disabled>١٠. العقود</button>
    </div>

    <!-- نموذج الإنشاء: خطوات 1–4 -->
    <form id="basicForm" enctype="multipart/form-data" class="space-y-6">

      <!-- 1) الأساسية -->
      <section data-step="1" class="card rounded shadow p-4">
        <h1 class="text-xl font-bold mb-4">➕ البيانات الأساسية</h1>
        <div class="grid md:grid-cols-3 gap-3">
          <div><label class="block text-sm mb-1">الاسم</label><input class="w-full border rounded px-3 py-2" name="name" required></div>
          <div>
            <label class="block text-sm mb-1">البريد</label>
            <input type="email" class="w-full border rounded px-3 py-2" name="email" required>
            <small id="email_hint" class="block mt-1 text-xs"></small>
          </div>
          <div><label class="block text-sm mb-1">كلمة المرور</label><input type="password" class="w-full border rounded px-3 py-2" name="password" required></div>

          <div><label class="block text-sm mb-1">الجنس</label>
            <select name="gender" class="w-full border rounded px-3 py-2"><option value="">اختر</option><option>ذكر</option><option>أنثى</option></select>
          </div>
          <div><label class="block text-sm mb-1">الديانة</label>
            <select name="religion" class="w-full border rounded px-3 py-2"><option value="">اختر</option><option>مسلم</option><option>غير مسلم</option></select>
          </div>
          <div><label class="block text-sm mb-1">الحالة الاجتماعية</label>
            <select name="marital_status" class="w-full border rounded px-3 py-2">
              <option value="">اختر</option><option>أعزب</option><option>متزوج</option><option>مطلق</option><option>أرمل</option>
            </select>
          </div>

          <div><label class="block text-sm mb-1">إيميل شخصي</label><input name="personal_email" type="email" class="w-full border rounded px-3 py-2"></div>
          <div><label class="block text-sm mb-1">جوال شخصي</label><input name="personal_phone" class="w-full border rounded px-3 py-2"></div>

          <div>
            <label class="block text-sm mb-1">الدور</label>
            <select name="role" class="w-full border rounded px-3 py-2">
              <option value="admin">مدير نظام</option><option value="manager">مدير</option><option value="employee" selected>موظف</option>
            </select>
          </div>
          <div>
            <label class="block text-sm mb-1">الإدارة</label>
            <select name="department_id" class="w-full border rounded px-3 py-2">
              <option value="">بدون</option>
              <?php foreach($departments as $d): ?>
                <option value="<?= $d['id'] ?>"><?= htmlspecialchars($d['name']) ?></option>
              <?php endforeach; ?>
            </select>
          </div>
          <div>
            <label class="block text-sm mb-1">وقت الدوام</label>
            <select name="work_shift_id" class="w-full border rounded px-3 py-2" required>
              <option value="">اختر</option>
              <?php foreach($work_shifts as $s): ?>
              <option value="<?= $s['id'] ?>">
                <?= htmlspecialchars($s['name']) ?> (<?= $s['type'] ?>: <?= substr($s['start_time'],0,5) ?> - <?= substr($s['end_time'],0,5) ?>)
              </option>
              <?php endforeach; ?>
            </select>
          </div>

          <div>
            <label class="block text-sm mb-1">الموقع الرئيسي</label>
            <select name="location_id" class="w-full border rounded px-3 py-2">
              <option value="">اختر</option>
              <?php foreach($locations as $l): ?>
                <option value="<?= $l['id'] ?>"><?= htmlspecialchars($l['name']) ?></option>
              <?php endforeach; ?>
            </select>
          </div>

          <div>
            <label class="block text-sm mb-1">يوم الراحة</label>
            <select name="day_off" class="w-full border rounded px-3 py-2">
              <option value="">بدون</option>
              <option value="6">السبت</option><option value="0">الأحد</option><option value="1">الاثنين</option>
              <option value="2">الثلاثاء</option><option value="3">الأربعاء</option><option value="4">الخميس</option><option value="5">الجمعة</option>
            </select>
          </div>

          <div>
            <label class="block text-sm mb-1">الرقم الوظيفي</label>
            <div class="flex items-center gap-2">
              <input id="employee_number" name="employee_number" class="w-full border rounded px-3 py-2" value="<?= htmlspecialchars($nextEmpNumber ?? '') ?>" placeholder="اتركه فاضي للتوليد التلقائي">
              <button type="button" id="emp_refresh" class="px-2 py-2 bg-gray-600 text-white rounded text-sm">تحديث</button>
            </div>
            <small id="emp_hint" class="block mt-1 text-xs opacity-80">
              <?= isset($nextEmpNumber) && $nextEmpNumber!=='' ? 'الرقم المقترح: '.$nextEmpNumber : '' ?>
            </small>
          </div>

          <div><label class="block text-sm mb-1">المسمّى الوظيفي</label><input name="job_title" class="w-full border rounded px-3 py-2"></div>
          <div><label class="block text-sm mb-1">تاريخ التوظيف</label><input type="date" name="hire_date" class="w-full border rounded px-3 py-2"></div>
          <div><label class="block text-sm mb-1">الحالة الوظيفية</label>
            <select name="employment_status" class="w-full border rounded px-3 py-2">
              <option value="نشط">نشط</option><option value="موقوف">موقوف</option><option value="مستقيل">مستقيل</option><option value="متعاقد">متعاقد</option>
            </select>
          </div>
          <div>
            <label for="nationality" class="block text-sm mb-1">الجنسية</label>
            <input type="text" name="nationality" id="nationality" class="w-full border rounded px-3 py-2" placeholder="مثال: سعودي">
          </div>

          <div class="md:col-span-3 border-t border-white/10 pt-3"></div>

          <div>
            <label class="block text-sm mb-1">صورة الموظف (اختياري)</label>
            <input type="file" name="profile_image" accept="image/*" class="w-full border rounded px-3 py-2 bg-white">
          </div>
          <div>
            <label class="inline-flex items-center gap-2 mt-6">
              <input type="checkbox" name="can_sign" value="1"> يستطيع التوقيع
            </label>
            <label class="inline-flex items-center gap-2 block mt-2">
              <input type="checkbox" name="can_transfer_tasks" value="1"> تحويل المهام بين الإدارات
            </label>
          </div>
          <div>
            <label class="block text-sm mb-1">درجة السرية</label>
            <select name="secret_level" class="w-full border rounded px-3 py-2">
              <option value="normal">عادية</option><option value="secret">سرية</option><option value="top_secret">سرية جدًا</option>
            </select>
          </div>
          <div class="md:col-span-2">
            <label class="block text-sm mb-1">ملف التوقيع (صورة) — اختياري</label>
            <input type="file" name="signature_file" accept="image/*" class="w-full border rounded px-3 py-2 bg-white">
          </div>
        </div>
      </section>

      <!-- 2) العنوان والاتصال -->
      <section data-step="2" class="card rounded shadow p-4 hidden">
        <h2 class="text-lg font-bold mb-3">📍 العنوان والاتصال</h2>
        <div class="grid md:grid-cols-3 gap-3">
          <div><label class="block text-sm mb-1">المبنى</label><input name="address_building" class="w-full border rounded px-3 py-2"></div>
          <div><label class="block text-sm mb-1">الشارع</label><input name="address_street" class="w-full border rounded px-3 py-2"></div>
          <div><label class="block text-sm mb-1">الحي</label><input name="address_district" class="w-full border rounded px-3 py-2"></div>
          <div><label class="block text-sm mb-1">المدينة</label><input name="address_city" class="w-full border rounded px-3 py-2"></div>
          <div><label class="block text-sm mb-1">الرمز البريدي</label><input name="address_zip" class="w-full border rounded px-3 py-2"></div>
          <div><label class="block text-sm mb-1">الرقم الإضافي</label><input name="address_additional_number" class="w-full border rounded px-3 py-2"></div>

          <!-- حقول عمل مميّزة عن الشخصية لتجنب التضارب -->
          <div><label class="block text-sm mb-1">إيميل العمل</label><input name="work_email" type="email" class="w-full border rounded px-3 py-2"></div>
          <div><label class="block text-sm mb-1">جوال العمل</label><input name="work_phone" class="w-full border rounded px-3 py-2"></div>

          <div>
            <label class="block text-sm mb-1">شهادة العنوان الوطني (PDF/صورة)</label>
            <input type="file" name="national_address_certificate" accept=".pdf,image/*" class="w-full border rounded px-3 py-2 bg-white text-black">
          </div>
        </div>
      </section>

      <!-- 3) الهوية والجواز -->
      <section data-step="3" class="card rounded shadow p-4 hidden">
        <h2 class="text-lg font-bold mb-3">🪪 الهوية والجواز</h2>
        <div class="grid md:grid-cols-3 gap-3">
          <div><label class="block text-sm mb-1">الهوية (10 أرقام)</label><input name="national_id" maxlength="10" class="w-full border rounded px-3 py-2"></div>
          <div><label class="block text-sm mb-1">تاريخ إصدار الهوية</label><input type="date" name="national_id_issue_date" class="w-full border rounded px-3 py-2"></div>
          <div><label class="block text-sm mb-1">مكان إصدار الهوية</label><input name="national_id_place" class="w-full border rounded px-3 py-2"></div>

          <div><label class="block text-sm mb-1">رقم الجواز</label><input name="passport_number" class="w-full border rounded px-3 py-2"></div>
          <div><label class="block text-sm mb-1">مكان إصدار الجواز</label><input name="passport_place" class="w-full border rounded px-3 py-2"></div>
          <div><label class="block text-sm mb-1">دولة الجواز</label><input name="passport_country" class="w-full border rounded px-3 py-2"></div>
          <div><label class="block text-sm mb-1">تاريخ إصدار الجواز</label><input type="date" name="passport_issue_date" class="w-full border rounded px-3 py-2"></div>
          <div><label class="block text-sm mb-1">تاريخ انتهاء الجواز</label><input type="date" name="passport_expire_date" class="w-full border rounded px-3 py-2"></div>
        </div>
      </section>

      <!-- 4) الحساب البنكي -->
      <section data-step="4" class="card rounded shadow p-4 hidden">
        <h2 class="text-lg font-bold mb-3">🏦 الحساب البنكي</h2>
        <div class="grid md:grid-cols-3 gap-3">
          <div><label class="block text-sm mb-1">رقم الحساب البنكي</label><input name="bank_account" class="w-full border rounded px-3 py-2"></div>
          <div>
            <label class="block text-sm mb-1">IBAN</label>
            <input id="iban" name="iban" class="w-full border rounded px-3 py-2 uppercase" placeholder="SA..">
            <small class="opacity-70">اكتب الآيبان واطلع من الحقل للتعرّف على البنك تلقائيًا</small>
            <small id="iban_msg" class="block mt-1 text-xs text-gray-200"></small>
          </div>
          <div><label class="block text-sm mb-1">اسم البنك</label><input id="bank_name" name="bank_name" class="w-full border rounded px-3 py-2 bg-gray-50" readonly></div>
          <div>
            <label class="block text-sm mb-1">شهادة الآيبان (PDF/صورة)</label>
            <input type="file" name="iban_certificate" accept=".pdf,image/*" class="w-full border rounded px-3 py-2 bg-white text-black">
          </div>
        </div>
      </section>

      <!-- شريط التحكّم السفلي (للخطوات 1–4 قبل الإنشاء) -->
      <div id="wizardBar" class="hidden rounded p-3 flex items-center justify-between">
        <div class="flex gap-2">
          <button type="button" class="px-4 py-2 bg-gray-600 text-white rounded" data-prev>السابق</button>
          <button type="button" class="px-4 py-2 bg-gray-600 text-white rounded" data-next>التالي</button>
        </div>
        <div>
          <button id="btnCreateBasic" type="button" class="px-4 py-2 bg-blue-600 text-white rounded">إنشاء المستخدم</button>
        </div>
      </div>

      <div class="text-sm text-green-200 hidden" id="basicDoneMsg">✅ تم إنشاء المستخدم — افتحت لك التبويبات تحت</div>
    </form>

    <!-- 5) المواقع -->
    <section data-step="5" class="card rounded shadow p-4 hidden" data-need-user>
      <h2 class="text-lg font-bold mb-3">📍 المواقع المصرّح بها</h2>
      <div id="loc_wrap" class="grid md:grid-cols-3 gap-2"></div>
      <button id="loc_save" class="px-3 py-2 rounded bg-blue-600 text-white mt-3">حفظ المواقع</button>
    </section>

    <!-- 6) الطوارئ -->
    <section data-step="6" class="card rounded shadow p-4 hidden" data-need-user>
      <h2 class="text-lg font-bold mb-3">🚑 جهات الطوارئ</h2>
      <div class="grid md:grid-cols-5 gap-2 mb-3">
        <input id="ec_name" class="border rounded px-3 py-2 bg-white text-black" placeholder="الاسم">
        <input id="ec_phone" class="border rounded px-3 py-2 bg-white text-black" placeholder="الجوال">
        <input id="ec_rel"   class="border rounded px-3 py-2 bg-white text-black" placeholder="صلة القرابة">
        <label class="inline-flex items-center gap-2"><input id="ec_primary" type="checkbox"> أولوي</label>
        <button id="ec_save" type="button" class="px-3 py-2 bg-green-600 text-white rounded">حفظ</button>
      </div>
      <div id="ec_list" class="divide-y border rounded bg-white"></div>
    </section>

    <!-- 7) العائلة -->
    <section data-step="7" class="card rounded shadow p-4 hidden" data-need-user>
      <h2 class="text-lg font-bold mb-3">👨‍👩‍👧‍👦 أفراد العائلة</h2>
      <div class="grid md:grid-cols-6 gap-2 mb-3">
        <input id="fm_id" type="hidden">
        <input id="fm_name" class="border rounded px-3 py-2 bg-white text-black" placeholder="الاسم">
        <select id="fm_rel" class="border rounded px-3 py-2 bg-white text-black">
          <option value="">صلة القرابة</option>
          <option>زوج</option><option>زوجة</option><option>ابن</option><option>ابنة</option>
          <option>أب</option><option>أم</option><option>أخ</option><option>أخت</option><option>أخرى</option>
        </select>
        <input id="fm_nid" class="border rounded px-3 py-2 bg-white text-black" placeholder="رقم الهوية">
        <input id="fm_birth" type="date" class="border rounded px-3 py-2 bg-white text-black">
        <label class="inline-flex items-center gap-2"><input id="fm_dep" type="checkbox" checked> معال</label>
        <button id="fm_save" type="button" class="px-3 py-2 bg-green-600 text-white rounded">حفظ</button>
      </div>
      <div id="fm_list" class="divide-y border rounded bg-white"></div>
    </section>

    <!-- 8) المؤهلات -->
    <section data-step="8" class="card rounded shadow p-4 hidden" data-need-user>
      <h2 class="text-lg font-bold mb-3">🎓 المؤهلات</h2>
      <form id="uq_form" class="grid md:grid-cols-6 gap-2 mb-3" enctype="multipart/form-data">
        <input id="uq_id" name="id" type="hidden">
        <input name="degree" id="uq_degree" class="border rounded px-3 py-2 bg-white text-black" placeholder="المؤهل">
        <input name="major" id="uq_major" class="border rounded px-3 py-2 bg-white text-black" placeholder="التخصص">
        <input name="institution" id="uq_inst" class="border rounded px-3 py-2 bg-white text-black" placeholder="الجهة">
        <input name="country" id="uq_country" class="border rounded px-3 py-2 bg-white text-black" placeholder="الدولة">
        <input name="graduation_date" id="uq_date" type="date" class="border rounded px-3 py-2 bg-white text-black">
        <input name="grade" id="uq_grade" class="border rounded px-3 py-2 bg-white text-black" placeholder="التقدير">
        <input name="file" id="uq_file" type="file" class="md:col-span-3 bg-white text-black">
        <button id="uq_save" type="button" class="px-3 py-2 bg-blue-600 text-white rounded md:col-span-1">حفظ</button>
      </form>
      <div id="uq_list" class="divide-y border rounded bg-white"></div>
    </section>

    <!-- 9) الراتب -->
    <section id="salary_section" data-step="9" class="card rounded shadow p-4 hidden" data-need-user>
      <h2 class="text-lg font-bold mb-3">💰 الراتب</h2>
      <div class="grid md:grid-cols-3 gap-3">
        <div class="md:col-span-3 flex flex-wrap items-end gap-4">
          <div>
            <label class="block text-sm mb-1">الراتب الأساسي</label>
            <input id="salary_input" name="salary" type="number" min="0" step="0.01" class="border rounded px-3 py-2 w-44 bg-white text-black" placeholder="مثلاً 5000">
          </div>
          <div class="flex items-center gap-2">
            <span class="text-sm opacity-80">النِّسب التلقائية:</span>
            <label class="inline-flex items-center gap-2"> سكن %
              <input id="housing_pct" type="number" min="0" step="0.01" value="25" class="border rounded px-2 py-1 w-20 bg-white text-black">
            </label>
            <label class="inline-flex items-center gap-2"> نقل %
              <input id="transport_pct" type="number" min="0" step="0.01" value="10" class="border rounded px-2 py-1 w-20 bg-white text-black">
            </label>
          </div>
        </div>
        <div class="md:col-span-3" id="salary_components_wrap">
          <label class="block mb-2 font-bold">بنود الرواتب</label>
          <select id="salary_components_select" class="w-full" multiple>
            <?php foreach ($salaryComponents as $comp): ?>
              <option value="<?= $comp['id'] ?>"><?= htmlspecialchars($comp['name']) ?></option>
            <?php endforeach; ?>
          </select>
          <div id="selected_components" class="mt-4 space-y-2"></div>
          <div id="components_hidden_fields"></div>
          <div class="mt-2">إجمالي البنود: <span id="salary_total">0</span> ريال</div>
        </div>
      </div>
      <div class="flex justify-end mt-3"><button id="salary_save" type="button" class="px-4 py-2 bg-blue-600 text-white rounded">حفظ الراتب</button></div>
    </section>

    <!-- 10) العقود -->
    <section data-step="10" class="card rounded shadow p-4 hidden" data-need-user>
      <h2 class="text-lg font-bold mb-3">📄 العقود</h2>
      <form id="ctr_form" class="space-y-4">
        <input type="hidden" id="ctr_id" name="id">
        <!-- تعريف العقد -->
        <div class="grid md:grid-cols-3 gap-3">
          <!-- نوع التوظيف (Segmented) -->
          <div>
            <label class="block text-sm mb-2">نوع التوظيف</label>
            <div class="flex flex-wrap gap-2 seg" data-bind="ctr_emp_type">
              <select name="employment_type" id="ctr_emp_type" class="hidden">
                <option>دوام كامل</option><option>دوام جزئي</option><option>عن بُعد</option>
                <option>أجير</option><option>تمهير</option><option>تدريب تعاوني</option><option>أخرى</option>
              </select>
              <button type="button" class="seg-btn" data-v="دوام كامل">دوام كامل</button>
              <button type="button" class="seg-btn" data-v="دوام جزئي">دوام جزئي</button>
              <button type="button" class="seg-btn" data-v="عن بُعد">عن بُعد</button>
              <button type="button" class="seg-btn" data-v="أجير">أجير</button>
              <button type="button" class="seg-btn" data-v="تمهير">تمهير</button>
              <button type="button" class="seg-btn" data-v="تدريب تعاوني">تدريب تعاوني</button>
              <button type="button" class="seg-btn" data-v="أخرى">أخرى</button>
            </div>
          </div>
          <!-- نوع العقد -->
          <div>
            <label class="block text-sm mb-2">نوع العقد</label>
            <select name="contract_type" id="ctr_type" class="w-full border rounded px-3 py-2 bg-white text-black">
              <option>محدد المدة</option>
              <option>غير محدد المدة</option>
              <option>مؤقت</option>
              <option>تدريب</option>
            </select>
          </div>
          <!-- الحالة -->
          <div>
            <label class="block text-sm mb-2">الحالة</label>
            <div class="seg flex gap-1" data-bind="ctr_status">
              <select name="status" id="ctr_status" class="hidden">
                <option>ساري</option><option>معلق</option><option>منتهي</option>
              </select>
              <button type="button" class="seg-btn" data-v="ساري">ساري</button>
              <button type="button" class="seg-btn" data-v="معلق">معلق</button>
              <button type="button" class="seg-btn" data-v="منتهي">منتهي</button>
            </div>
          </div>
        </div>
        <!-- الفترات والتنبيهات -->
        <div class="grid md:grid-cols-4 gap-3">
          <div>
            <label class="block text-sm mb-1">بداية</label>
            <input type="date" name="start_date" id="ctr_start" class="w-full border rounded px-3 py-2 bg-white text-black">
          </div>
          <div id="ctr_end_wrap">
            <label class="block text-sm mb-1">نهاية</label>
            <input type="date" name="end_date" id="ctr_end" class="w-full border rounded px-3 py-2 bg-white text-black">
            <small class="text-xs opacity-70">يُخفى تلقائيًا إذا كان العقد غير محدد المدة</small>
          </div>
          <div>
            <label class="block text-sm mb-1">أشهر التجربة</label>
            <input type="number" min="0" name="probation_months" id="ctr_prob" class="w-full border rounded px-3 py-2 bg-white text-black" placeholder="مثلاً 3">
          </div>
          <div class="grid grid-cols-2 gap-2">
            <div>
              <label class="block text-sm mb-1">تنبيه قبل نهاية العقد</label>
              <input type="number" min="0" name="notify_days_before_end" id="ctr_nend" value="30" class="w-full border rounded px-3 py-2 bg-white text-black">
            </div>
            <div>
              <label class="block text-sm mb-1">تنبيه قبل نهاية التجربة</label>
              <input type="number" min="0" name="notify_days_before_probation_end" id="ctr_nprob" value="7" class="w-full border rounded px-3 py-2 bg-white text-black">
            </div>
          </div>
        </div>
        <!-- مشروع وملاحظات -->
        <div class="grid md:grid-cols-3 gap-3">
          <div class="md:col-span-1">
            <label class="block text-sm mb-1">المشروع (اختياري)</label>
            <select name="project_id" id="ctr_project" class="w-full border rounded px-3 py-2 bg-white text-black">
              <option value="">بدون</option>
              <?php foreach($projects as $p): ?>
                <option value="<?= $p['id'] ?>"><?= htmlspecialchars($p['name']) ?></option>
              <?php endforeach; ?>
            </select>
          </div>
          <div class="md:col-span-2">
            <label class="block text-sm mb-1">ملاحظات</label>
            <textarea name="notes" id="ctr_notes" class="border rounded px-3 py-2 w-full bg-white text-black" rows="2"></textarea>
          </div>
        </div>
        <!-- أزرار -->
        <div class="flex justify-between items-center">
          <button id="ctr_reset" type="button" class="px-4 py-2 bg-gray-600 text-white rounded">تفريغ</button>
          <div class="flex gap-2">
            <button id="ctr_save" type="button" class="px-4 py-2 bg-blue-600 text-white rounded">حفظ</button>
          </div>
        </div>
      </form>
      <div id="ctr_list" class="divide-y border rounded bg-white"></div>
    </section>

  </div>

<!-- ===================== JS ===================== -->
<script>
let NEW_UID = null;
window.PROFILE_UID = null;

/* ===== Stepper/Sections ===== */
(function(){
  const sections = [...document.querySelectorAll('section[data-step]')];
  const btns = [...document.querySelectorAll('#stepper .step-btn')];
  let curr = 1;

  function enableAfterCreate(){
    btns.forEach(b=>{ if (+b.dataset.step >= 5) b.disabled = false; });
    document.querySelectorAll('[data-need-user]').forEach(el=>el.classList.remove('hidden'));
  }

  function show(step){
    curr = step;
    sections.forEach(s => s.classList.toggle('hidden', +s.dataset.step !== step));
    btns.forEach(b=> b.classList.toggle('active', +b.dataset.step === step));
    const wiz = document.getElementById('wizardBar');
    if (wiz) wiz.classList.toggle('hidden', !(step<=4 && !window.PROFILE_UID));
    if (step===10 && typeof toggleEnd==='function') toggleEnd();
    window.scrollTo({top:0, behavior:'smooth'});
  }

  btns.forEach(b => b.addEventListener('click', ()=>{ if (!b.disabled) show(+b.dataset.step); }));
  document.addEventListener('click', (e)=>{
    if (e.target.matches('[data-next]')) show(Math.min(curr+1, sections.length));
    if (e.target.matches('[data-prev]')) show(Math.max(curr-1, 1));
  });

  window.__enableAfterCreate = enableAfterCreate;
  show(1);
})();

/* منع submit الافتراضي */
$('#basicForm').on('submit', e => e.preventDefault());

/* Email check */
let emailTimer=null;
$('#basicForm [name="email"]').on('input blur', function(){
  clearTimeout(emailTimer);
  const val = this.value.trim();
  if(!val){ $('#email_hint').text(''); return; }
  emailTimer = setTimeout(async ()=>{
    const r = await fetch('/erp_task_system/api/check_email.php?email='+encodeURIComponent(val));
    const j = await r.json();
    if(j.ok){ $('#email_hint').text('الإيميل متاح').removeClass('text-red-600').addClass('text-green-600'); $('#btnCreateBasic').prop('disabled', false); }
    else    { $('#email_hint').text('الإيميل مستخدم').removeClass('text-green-600').addClass('text-red-600'); $('#btnCreateBasic').prop('disabled', true); }
  }, 350);
});

/* اقتراح رقم وظيفي */
async function loadNextEmpNumber(force = false){
  const $inp = $('#employee_number');
  if (!force && $inp.val().trim() !== '') return;
  try {
    const r = await fetch('/erp_task_system/api/next_employee_number.php');
    const j = await r.json();
    if (j.ok){ $('#emp_hint').text('الرقم المقترح: ' + j.number); if ($inp.val().trim() === '') $inp.val(j.number); }
    else { $('#emp_hint').text('تعذّر جلب الرقم المقترح'); }
  } catch(e){ console.error(e); $('#emp_hint').text('تعذّر جلب الرقم المقترح'); }
}
$(document).ready(()=> loadNextEmpNumber(false));
$('#emp_refresh').on('click', ()=> loadNextEmpNumber(true));
$('#employee_number').on('blur', function(){ if (this.value.trim()==='') loadNextEmpNumber(true); });

/* IBAN lookup */
function normalizeIBAN(raw){ return (raw||'').toString().replace(/\s+/g,'').toUpperCase(); }
let ibanTimer=null;
async function lookupIBAN(){
  const iban = normalizeIBAN($('#iban').val());
  $('#iban').val(iban);
  if (!iban || iban.length < 6){ $('#bank_name').val(''); $('#iban_msg').text(''); return; }
  const fd = new FormData(); fd.append('iban', iban);
  try{
    const r = await fetch('/erp_task_system/api/iban_lookup.php', { method:'POST', body:fd });
    const j = await r.json();
    if (j.ok){ $('#bank_name').val(j.bank_name || ''); $('#iban_msg').text(j.bank_code ? ('كود البنك: '+j.bank_code) : ''); }
    else    { $('#bank_name').val(''); $('#iban_msg').text(j.reason ? ('ما لقينا بنك مطابق ('+j.reason+')') : 'ما لقينا بنك مطابق'); }
  }catch(e){ console.error(e); $('#iban_msg').text('تعذر الاتصال بخدمة الآيبان'); }
}
$('#iban').on('input', function(){ clearTimeout(ibanTimer); ibanTimer = setTimeout(lookupIBAN, 400); });
$('#iban').on('blur', lookupIBAN);

/* ===== إنشاء المستخدم ===== */
$('#btnCreateBasic').on('click', async ()=>{
  const f = document.getElementById('basicForm');
  const fd = new FormData(f);
  // تأكدنا أن الحقول ما تتصادم (work_* مستقلة عن personal_*)
  const r = await fetch('/erp_task_system/api/users_basic_create.php',{method:'POST',body:fd});
  const j = await r.json();
  if(!j.ok){ alert((j.msg || 'خطأ في الحفظ') + (j.err ? ("\n"+j.err) : '')); return; }
  NEW_UID = j.id; window.PROFILE_UID = NEW_UID;
  document.getElementById('basicDoneMsg').classList.remove('hidden');
  if (typeof window.__enableAfterCreate==='function') window.__enableAfterCreate();
  buildLocSelect(); ecList(); fmList(); uqList(); ctrList();
  const btn5 = document.querySelector('#stepper .step-btn[data-step="5"]'); if (btn5) btn5.click();
});

/* ===== المواقع ===== */
async function buildLocSelect(){
  if(!window.PROFILE_UID) return;
  const fd=new FormData(); fd.append('action','list'); fd.append('user_id',window.PROFILE_UID);
  const r=await fetch('/erp_task_system/api/user_locations.php',{method:'POST',body:fd});
  const j=await r.json(); const w=document.getElementById('loc_wrap'); w.innerHTML='';
  if(j.ok){ j.data.forEach(x=>{ w.insertAdjacentHTML('beforeend',`<label class="flex items-center gap-2 p-2 border rounded bg-white text-black"><input type="checkbox" value="${x.id}" ${+x.assigned?'checked':''}><span>${x.name}</span></label>`); }); }
}
$('#loc_save').on('click', async ()=>{
  const ids=[...document.querySelectorAll('#loc_wrap input[type=checkbox]:checked')].map(i=>i.value);
  const fd=new FormData(); fd.append('action','save'); fd.append('user_id',window.PROFILE_UID);
  ids.forEach(i=>fd.append('locations[]',i));
  const r=await fetch('/erp_task_system/api/user_locations.php',{method:'POST',body:fd});
  const j=await r.json(); if(!j.ok){ alert('فشل الحفظ'); return; } alert('تم الحفظ');
});

/* ===== الطوارئ ===== */
async function ecList(){
  if(!window.PROFILE_UID) return; const fd=new FormData(); fd.append('action','list'); fd.append('user_id',window.PROFILE_UID);
  const r=await fetch('/erp_task_system/api/emergency_contacts.php',{method:'POST',body:fd});
  const j=await r.json(); const box=document.getElementById('ec_list'); box.innerHTML='';
  if(j.ok){ j.data.forEach(row=>{ const div=document.createElement('div'); div.className='p-3 flex items-center justify-between'; div.innerHTML=`<div><div class="font-medium text-black">${row.contact_name} ${row.is_primary?'⭐':''}</div><div class="text-sm text-gray-600">${row.relationship||''} • ${row.phone}</div></div><button class="text-red-600" data-id="${row.id}">حذف</button>`; div.querySelector('button').onclick=()=>ecDelete(row.id); box.appendChild(div); }); }
}
$('#ec_save').on('click', async ()=>{
  const fd=new FormData(); fd.append('action','save'); fd.append('user_id',window.PROFILE_UID);
  fd.append('contact_name',$('#ec_name').val()); fd.append('phone',$('#ec_phone').val()); fd.append('relationship',$('#ec_rel').val()); fd.append('is_primary',$('#ec_primary').is(':checked')?1:0);
  const r=await fetch('/erp_task_system/api/emergency_contacts.php',{method:'POST',body:fd}); const j=await r.json(); if(!j.ok){ alert('خطأ بالحفظ'); return; } $('#ec_name,#ec_phone,#ec_rel').val(''); $('#ec_primary').prop('checked',false); ecList();
});
async function ecDelete(id){ const fd=new FormData(); fd.append('action','delete'); fd.append('user_id',window.PROFILE_UID); fd.append('id',id); const r=await fetch('/erp_task_system/api/emergency_contacts.php',{method:'POST',body:fd}); const j=await r.json(); if(j.ok) ecList(); }

/* ===== العائلة ===== */
async function fmList(){ if(!window.PROFILE_UID) return; const fd=new FormData(); fd.append('action','list'); fd.append('user_id',window.PROFILE_UID); const r=await fetch('/erp_task_system/api/family_members.php',{method:'POST',body:fd}); const j=await r.json(); const box=document.getElementById('fm_list'); box.innerHTML=''; if(j.ok){ j.data.forEach(row=>{ const div=document.createElement('div'); div.className='p-3 flex items-center justify-between'; div.innerHTML=`<div><div class="font-medium text-black">${row.name} ${row.is_dependent?'(معال)':''}</div><div class="text-sm text-gray-600">${row.relationship||''} • ${row.national_id||''} • ${row.birth_date||''}</div></div><div class="flex gap-3"><button class="text-blue-600">تعديل</button><button class="text-red-600">حذف</button></div>`; div.querySelector('.text-blue-600').onclick=()=>fmFill(row); div.querySelector('.text-red-600').onclick=()=>fmDelete(row.id); box.appendChild(div); }); } }
function fmFill(r){ $('#fm_id').val(r.id); $('#fm_name').val(r.name||''); $('#fm_rel').val(r.relationship||''); $('#fm_nid').val(r.national_id||''); $('#fm_birth').val(r.birth_date||''); $('#fm_dep').prop('checked', +r.is_dependent===1); }
$('#fm_save').on('click', async ()=>{ const fd=new FormData(); fd.append('action','save'); fd.append('user_id',window.PROFILE_UID); fd.append('id',$('#fm_id').val()); fd.append('name',$('#fm_name').val()); fd.append('relationship',$('#fm_rel').val()); fd.append('national_id',$('#fm_nid').val()); fd.append('birth_date',$('#fm_birth').val()); fd.append('is_dependent',$('#fm_dep').is(':checked')?1:0); const r=await fetch('/erp_task_system/api/family_members.php',{method:'POST',body:fd}); const j=await r.json(); if(!j.ok){ alert('خطأ بالحفظ'); return; } $('#fm_id,#fm_name,#fm_rel,#fm_nid,#fm_birth').val(''); $('#fm_dep').prop('checked',true); fmList(); });
async function fmDelete(id){ if(!confirm('تأكيد الحذف؟')) return; const fd=new FormData(); fd.append('action','delete'); fd.append('user_id',window.PROFILE_UID); fd.append('id',id); const r=await fetch('/erp_task_system/api/family_members.php',{method:'POST',body:fd}); const j=await r.json(); if(j.ok) fmList(); }

/* ===== المؤهلات ===== */
async function uqList(){ if(!window.PROFILE_UID) return; const fd=new FormData(); fd.append('action','list'); fd.append('user_id',window.PROFILE_UID); const r=await fetch('/erp_task_system/api/user_qualifications.php',{method:'POST',body:fd}); const j=await r.json(); const box=document.getElementById('uq_list'); box.innerHTML=''; if(j.ok){ j.data.forEach(row=>{ const div=document.createElement('div'); div.className='p-3 flex items-center justify-between'; div.innerHTML=`<div><div class="font-medium text-black">${row.degree||''} - ${row.major||''}</div><div class="text-sm text-gray-600">${row.institution||''} • ${row.country||''} • ${row.graduation_date||''} • ${row.grade||''}</div>${row.file_url? `<a class=\"text-blue-600 underline text-sm\" href=\"${row.file_url}\" target=\"_blank\">عرض المرفق</a>`:''}</div><div class="flex gap-3"><button class="text-blue-600">تعديل</button><button class="text-red-600">حذف</button></div>`; div.querySelector('.text-blue-600').onclick=()=>uqFill(row); div.querySelector('.text-red-600').onclick=()=>uqDelete(row.id); box.appendChild(div); }); } }
function uqFill(r){ $('#uq_id').val(r.id); $('#uq_degree').val(r.degree||''); $('#uq_major').val(r.major||''); $('#uq_inst').val(r.institution||''); $('#uq_country').val(r.country||''); $('#uq_date').val(r.graduation_date||''); $('#uq_grade').val(r.grade||''); $('#uq_file').val(''); }
$('#uq_save').on('click', async (e)=>{ const form = document.getElementById('uq_form'); const fd=new FormData(form); fd.append('action','save'); fd.append('user_id',window.PROFILE_UID); const r=await fetch('/erp_task_system/api/user_qualifications.php',{method:'POST',body:fd}); const j=await r.json(); if(!j.ok){ alert('خطأ بالحفظ'); return; } form.reset(); $('#uq_id').val(''); uqList(); });
async function uqDelete(id){ if(!confirm('تأكيد الحذف؟')) return; const fd=new FormData(); fd.append('action','delete'); fd.append('user_id',window.PROFILE_UID); fd.append('id',id); const r=await fetch('/erp_task_system/api/user_qualifications.php',{method:'POST',body:fd}); const j=await r.json(); if(j.ok) uqList(); }

/* ===== الراتب ===== */
let salaryComponents = <?= json_encode($salaryComponents, JSON_UNESCAPED_UNICODE) ?>;
const compTypes = Object.fromEntries(salaryComponents.map(c => [String(c.id), c.type||'']));
function getComponentName(id){ const c = salaryComponents.find(x => String(x.id) === String(id)); return c ? c.name : ''; }

$(document).ready(function(){
  $('#salary_components_select').select2({ placeholder: "اختر البنود المالية..." });
  $('#salary_components_select').on('change', function(){
    const selected = $(this).val() || [];
    const container = $('#selected_components');
    const hidden    = $('#components_hidden_fields');
    container.html(''); hidden.html('');
    selected.forEach(function(id){
      container.append(`
        <div class="flex gap-2 items-center component-row mb-2" data-id="${id}">
          <span class="w-40 inline-block font-bold">${getComponentName(id)}</span>
          <input type="number" min="0" step="0.01" name="component_value[${id}]" placeholder="القيمة" class="border rounded px-2 py-1 w-28 bg-white text-black" required>
          <input type="text" name="component_notes[${id}]" placeholder="ملاحظات" class="border rounded px-2 py-1 w-40 bg-white text-black">
          <button type="button" class="text-red-500 hover:text-red-700 font-bold" onclick="removeComponent(${id})">×</button>
        </div>
      `);
      hidden.append(`<input type=\"hidden\" name=\"component_ids[]\" value=\"${id}\">`);
    });
    calcSalaryTotal();
    setTimeout(()=>{ if (HOUSING_COMP) decorateDefaultRow(HOUSING_COMP.id); if (TRANSPORT_COMP) decorateDefaultRow(TRANSPORT_COMP.id); recomputeAuto(); }, 0);
  });
  $(document).on('input', '.component-row input[type=number]', calcSalaryTotal);
});
function removeComponent(id){ const $sel = $('#salary_components_select'); const vals = ($sel.val() || []).filter(v => v !== String(id)); $sel.val(vals).trigger('change'); calcSalaryTotal(); }
function calcSalaryTotal(){ let total = 0; document.querySelectorAll('.component-row').forEach(row=>{ const id = row.dataset.id; const v = parseFloat(row.querySelector('input[type=number]')?.value||''); if(!isNaN(v)) total += (compTypes[id]==='deduction' ? -v : v); }); $('#salary_total').text(total.toFixed(2)); }

// الأكواد الافتراضية
const DEFAULTS = { basicCode: 'BASIC', housingCode: 'HOUSING', transportCode: 'TRANSPORT' };
function findCompByCodeOrName(code, nameKeyword){ code = (code||'').toLowerCase(); const byCode = salaryComponents.find(c => (c.code||'').toLowerCase() === code); if (byCode) return byCode; return salaryComponents.find(c => (c.name||'').includes(nameKeyword)); }
const BASIC_COMP     = findCompByCodeOrName(DEFAULTS.basicCode,     'الراتب الأساسي') || null;
const HOUSING_COMP   = findCompByCodeOrName(DEFAULTS.housingCode,   'بدل السكن')       || null;
const TRANSPORT_COMP = findCompByCodeOrName(DEFAULTS.transportCode, 'بدل النقل')       || null;

function selectIfExists(ids){ const $sel = $('#salary_components_select'); const set = new Set($sel.val() || []); ids.filter(Boolean).forEach(id => set.add(String(id))); $sel.val([...set]).trigger('change'); }
function decorateDefaultRow(id){ const row = document.querySelector(`.component-row[data-id="${id}"]`); if (!row || row.querySelector('[data-reauto]')) return; const btn = document.createElement('button'); btn.type = 'button'; btn.textContent = '↺ تلقائي'; btn.className = 'text-xs underline text-blue-700 ml-1'; btn.dataset.reauto = id; row.appendChild(btn); }
function recomputeAuto(){ const base = parseFloat(document.getElementById('salary_input')?.value || ''); const hp = parseFloat(document.getElementById('housing_pct')?.value || '25'); const tp = parseFloat(document.getElementById('transport_pct')?.value || '10'); if (isNaN(base)) return; function setAmountFor(id, amount, respectManual=true){ const row = document.querySelector(`.component-row[data-id="${id}"]`); const input = row?.querySelector('input[type=number]'); if (!row || !input) return; if (respectManual && row.dataset.manual === '1') return; input.value = (parseFloat(amount||0) || 0).toFixed(2); }
  if (BASIC_COMP)     setAmountFor(BASIC_COMP.id, base, false);
  if (HOUSING_COMP)   setAmountFor(HOUSING_COMP.id,   base * (isNaN(hp)?25:hp) / 100, true);
  if (TRANSPORT_COMP) setAmountFor(TRANSPORT_COMP.id, base * (isNaN(tp)?10:tp) / 100, true);
  if (typeof calcSalaryTotal === 'function') calcSalaryTotal(); }
$(document).on('input', '.component-row input[type=number]', function(){ const row = this.closest('.component-row'); const id  = row?.dataset.id; if (id && (HOUSING_COMP && String(id)===String(HOUSING_COMP.id) || TRANSPORT_COMP && String(id)===String(TRANSPORT_COMP.id))) { row.dataset.manual = '1'; } if (typeof calcSalaryTotal === 'function') calcSalaryTotal(); });
$(document).on('click', '[data-reauto]', function(){ const row = this.closest('.component-row'); if (row) delete row.dataset.manual; recomputeAuto(); });
(function initDefaultsOnce(){ const ids = [BASIC_COMP?.id, HOUSING_COMP?.id, TRANSPORT_COMP?.id].filter(Boolean); if (ids.length) selectIfExists(ids); setTimeout(()=>{ if (HOUSING_COMP) decorateDefaultRow(HOUSING_COMP.id); if (TRANSPORT_COMP) decorateDefaultRow(TRANSPORT_COMP.id); recomputeAuto(); }, 150); })();
$('#salary_input, #housing_pct, #transport_pct').on('input', recomputeAuto);

// حفظ الراتب
$('#salary_save').on('click', async ()=>{
  if (!window.PROFILE_UID) { alert('إنشئ المستخدم أولاً'); return; }
  const fd = new FormData(); fd.append('action','save'); fd.append('user_id', window.PROFILE_UID);
  const base = parseFloat(document.getElementById('salary_input')?.value || ''); if (!isNaN(base)) fd.append('salary', base);
  document.querySelectorAll('#components_hidden_fields input[name="component_ids[]"]').forEach(inp=> fd.append('component_ids[]', inp.value));
  document.querySelectorAll('.component-row').forEach(row=>{ const id = row.dataset.id; const amount = row.querySelector('input[type=number]')?.value || ''; const note = row.querySelector('input[type=text]')?.value || ''; fd.append('component_value['+id+']', amount); fd.append('component_notes['+id+']', note); });
  try{ const r = await fetch('/erp_task_system/api/user_salary.php', { method:'POST', body: fd }); const j = await r.json(); if (j.ok) alert('تم حفظ الراتب وبنود الرواتب ✅'); else alert('فشل الحفظ: ' + (j.msg || '')); }catch(e){ console.error(e); alert('خطأ في الاتصال'); }
});

/* ===== العقود ===== */
async function ctrList(){ if(!window.PROFILE_UID) return; const fd=new FormData(); fd.append('action','list'); fd.append('user_id',window.PROFILE_UID); const r=await fetch('/erp_task_system/api/contracts.php',{method:'POST',body:fd}); const j=await r.json(); const box=document.getElementById('ctr_list'); box.innerHTML=''; if(!j.ok){ box.innerHTML='<div class="p-3 text-red-600">تعذّر التحميل</div>'; return; } if(!j.data.length){ box.innerHTML='<div class="p-3 text-gray-500">لا توجد عقود</div>'; return; } j.data.forEach(c=>{ const div=document.createElement('div'); div.className='p-3 rounded border bg-white flex flex-wrap justify-between items-center gap-3'; div.innerHTML=`
      <div>
        <div class="font-bold text-black">${c.employment_type} • ${c.contract_type}
          <span class="inline-block ml-2 px-2 py-0.5 text-xs rounded ${c.status==='ساري'?'bg-green-100 text-green-700':(c.status==='معلق'?'bg-amber-100 text-amber-700':'bg-red-100 text-red-700')}">${c.status}</span>
        </div>
        <div class="text-sm text-gray-600">بداية: ${c.start_date||'—'} • نهاية: ${c.end_date||'—'} • تجربة: ${c.probation_months||'0'} شهر ${c.project_name ? `• مشروع: ${c.project_name}` : ''}</div>
      </div>
      <div class="flex gap-2">
        <button class="px-2 py-1 text-blue-600"  data-act="edit">تعديل</button>
        <button class="px-2 py-1 text-amber-600" data-act="extend">تمديد</button>
        <button class="px-2 py-1 text-green-700" data-act="renew">تجديد</button>
        <button class="px-2 py-1 text-red-600"   data-act="close">إنهاء</button>
        <button class="px-2 py-1 text-gray-700"  data-act="files">مرفقات (${c.files_count||0})</button>
      </div>`; const bind=(sel,fn)=>div.querySelector(`[data-act="${sel}"]`).onclick=()=>fn(c); bind('edit',ctrFill); bind('extend',()=>ctrExtendPrompt(c.id)); bind('renew',()=>ctrRenewPrompt(c.id)); bind('close',()=>ctrClosePrompt(c.id)); bind('files',()=>ctrFiles(c.id)); box.appendChild(div); }); }

function ctrFill(c){
  $('#ctr_id').val(c.id);
  $('#ctr_emp_type').val(c.employment_type);
  $('#ctr_type').val(c.contract_type);
  $('#ctr_status').val(c.status);
  $('#ctr_start').val(c.start_date||'');
  $('#ctr_end').val(c.end_date||'');
  $('#ctr_prob').val(c.probation_months||'');
  $('#ctr_nend').val(c.notify_days_before_end||30);
  $('#ctr_nprob').val(c.notify_days_before_probation_end||7);
  $('#ctr_project').val(c.project_id||'');
  $('#ctr_notes').val(c.notes||'');
  syncSegFromSelect('ctr_emp_type');
  syncSegFromSelect('ctr_status');
  toggleEnd();
  window.scrollTo({top:0, behavior:'smooth'});
}
$('#ctr_reset').on('click', (e)=>{ e.preventDefault(); $('#ctr_form')[0].reset(); $('#ctr_id').val(''); syncSegFromSelect('ctr_emp_type'); syncSegFromSelect('ctr_status'); toggleEnd(); });
$('#ctr_save').on('click', async (e)=>{ const fd=new FormData($('#ctr_form')[0]); fd.append('action','save'); fd.append('user_id',window.PROFILE_UID); const r=await fetch('/erp_task_system/api/contracts.php',{method:'POST',body:fd}); const j=await r.json(); if(!j.ok){ alert(j.msg==='overlap'?'يوجد عقد ساري متداخل':'خطأ بالحفظ'); return; } $('#ctr_id').val(''); $('#ctr_form')[0].reset(); syncSegFromSelect('ctr_emp_type'); syncSegFromSelect('ctr_status'); ctrList(); });
async function ctrExtendPrompt(id){ const d=prompt('أدخل تاريخ نهاية جديد (YYYY-MM-DD)'); if(!d) return; const fd=new FormData(); fd.append('action','extend'); fd.append('user_id',window.PROFILE_UID); fd.append('id',id); fd.append('new_end_date',d); const r=await fetch('/erp_task_system/api/contracts.php',{method:'POST',body:fd}); const j=await r.json(); if(j.ok) ctrList(); else alert('خطأ'); }
async function ctrRenewPrompt(id){ const s=prompt('تاريخ بداية للعقد المجدَّد (YYYY-MM-DD)'); if(!s) return; const e=prompt('تاريخ نهاية (اختياري) (YYYY-MM-DD)'); const fd=new FormData(); fd.append('action','renew'); fd.append('user_id',window.PROFILE_UID); fd.append('id',id); fd.append('new_start_date',s); if(e) fd.append('new_end_date',e); fd.append('close_old','1'); const r=await fetch('/erp_task_system/api/contracts.php',{method:'POST',body:fd}); const j=await r.json(); if(j.ok) ctrList(); else alert('خطأ'); }
async function ctrClosePrompt(id){ if(!confirm('تأكيد إنهاء العقد؟')) return; const fd=new FormData(); fd.append('action','close'); fd.append('user_id',window.PROFILE_UID); fd.append('id',id); const r=await fetch('/erp_task_system/api/contracts.php',{method:'POST',body:fd}); const j=await r.json(); if(j.ok) ctrList(); else alert('خطأ'); }
async function ctrFiles(contract_id){ const box=document.createElement('div'); box.className='fixed inset-0 bg-black/40 flex items-center justify-center z-50'; box.innerHTML=`<div class=\"bg-white rounded shadow p-4 w-[90%] max-w-3xl\"><div class=\"flex justify-between items-center mb-3\"><h4 class=\"font-bold text-black\">مرفقات العقد #${contract_id}</h4><button class=\"px-2 py-1 bg-gray-600 text-white rounded\" id=\"cf_close\">إغلاق</button></div><form id=\"cf_form\" class=\"flex items-center gap-2 mb-3\" enctype=\"multipart/form-data\"><input type=\"file\" name=\"file\" class=\"border rounded px-3 py-2\"><button class=\"px-3 py-2 bg-blue-600 text-white rounded\">رفع</button></form><div id=\"cf_list\" class=\"divide-y border rounded\"></div></div>`; document.body.appendChild(box); box.querySelector('#cf_close').onclick=()=>box.remove(); async function list(){ const fd=new FormData(); fd.append('action','files'); fd.append('user_id',window.PROFILE_UID); fd.append('contract_id',contract_id); const r=await fetch('/erp_task_system/api/contracts.php',{method:'POST',body:fd}); const j=await r.json(); const l=box.querySelector('#cf_list'); l.innerHTML=''; if(j.ok && j.data.length){ j.data.forEach(f=>{ const row=document.createElement('div'); row.className='p-2 flex justify-between items-center'; row.innerHTML=`<a class=\"text-blue-600 underline\" href=\"${f.url}\" target=\"_blank\">${f.file_name}</a><button class=\"text-red-600\" data-id=\"${f.id}\">حذف</button>`; row.querySelector('button').onclick=async ()=>{ const fd=new FormData(); fd.append('action','delete_file'); fd.append('user_id',window.PROFILE_UID); fd.append('id',f.id); const r=await fetch('/erp_task_system/api/contracts.php',{method:'POST',body:fd}); const j=await r.json(); if(j.ok) list(); }; l.appendChild(row); }); } else { l.innerHTML='<div class="p-3 text-gray-500">لا توجد ملفات</div>'; } } box.querySelector('#cf_form').onsubmit=async (e)=>{ e.preventDefault(); const fd=new FormData(e.target); fd.append('action','upload_file'); fd.append('user_id',window.PROFILE_UID); fd.append('contract_id',contract_id); const r=await fetch('/erp_task_system/api/contracts.php',{method:'POST',body:fd}); const j=await r.json(); if(!j.ok){ alert('فشل الرفع'); return; } e.target.reset(); list(); }; list(); }

/* ربط أزرار الـ Segmented بالـ select المخفي */
document.addEventListener('click', (e)=>{
  const btn = e.target.closest('.seg-btn'); if (!btn) return;
  const seg  = btn.closest('.seg'); if (!seg) return;
  const bind = seg.getAttribute('data-bind'); if (!bind) return;
  const sel  = document.getElementById(bind); if (!sel) return;
  sel.value = btn.dataset.v || '';
  seg.querySelectorAll('.seg-btn').forEach(b=> b.classList.toggle('active', b === btn));
});
function syncSegFromSelect(id){ const seg = document.querySelector(`.seg[data-bind="${id}"]`); const sel = document.getElementById(id); if (!seg || !sel) return; const v = sel.value; seg.querySelectorAll('.seg-btn').forEach(b=> b.classList.toggle('active', b.dataset.v === v)); }
syncSegFromSelect('ctr_emp_type');
syncSegFromSelect('ctr_status');

/* إخفاء نهاية العقد إذا كان غير محدد المدة */
const ctrTypeSel = document.getElementById('ctr_type');
function toggleEnd(){ const show = (ctrTypeSel && ctrTypeSel.value !== 'غير محدد المدة'); const wrap = document.getElementById('ctr_end_wrap'); if (wrap) wrap.style.display = show ? '' : 'none'; if (!show) document.getElementById('ctr_end').value = ''; }
if (ctrTypeSel){ ctrTypeSel.addEventListener('change', toggleEnd); toggleEnd(); }
</script>
</body>
</html>
