<?php
require_once '../config/db.php';
require_once '../authorize.php';
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// ✅ تحقق الجلسة والصلاحيات
if (
  !isset($_SESSION['user']) ||
  !isset($_SESSION['user']['name']) ||
  !isset($_SESSION['user']['email']) ||
  !isset($_SESSION['user']['role']) ||
  !in_array($_SESSION['user']['role'], ['admin', 'manager', 'user'])
) {
  session_unset();
  session_destroy();
  header('Location: ../unauthorized.php');
  exit;
}

$user = $_SESSION['user'];
$role = $user['role'];
$department_id = $user['department_id'];
$user_id = $user['id'];
$user_secret_level = $user['secret_level'] ?? null;

// ✅ ثيم المستخدم
$color = $user['color'] ?? [
  'hex_code' => '#0b1023',
  'second_text_color' => '#e9eefb',
  'text_color' => '#e9eefb',
  'nav_color' => '#121b35',
  'chart_color' => 'rgba(111,125,251,.8)',
];
$userThemeColor = $color['hex_code'];
$userThemeSecondTextColor = $color['second_text_color'];
$userThemeTextColor = $color['text_color'];
$userThemeNavColor = $color['nav_color'];
$allColors = $user['colors'] ?? [];
$chartColor = $color['chart_color'];
$currentUserColorId = $user['color_id'] ?? null;

/* ===== حصر الوصول على الأدمن ومدير الموارد البشرية فقط ===== */
$deptStmt = $conn->prepare("SELECT name FROM departments WHERE id = ?");
$deptStmt->execute([$department_id]);
$myDeptName = $deptStmt->fetchColumn() ?: ($user['department_name'] ?? '');

$isAdmin     = ($role === 'admin');
$isHRManager = ($role === 'manager' && $myDeptName === 'إدارة الموارد البشرية');

if (!($isAdmin || $isHRManager)) {
  $_SESSION['error'] = 'غير مصرح بالدخول لهذه الصفحة. هذه الصفحة مخصصة للأدمن ومدير الموارد البشرية فقط.';
  header('Location: ../unauthorized.php');
  exit;
}

// ✅ CSRF
$_SESSION['csrf'] = $_SESSION['csrf'] ?? bin2hex(random_bytes(32));
$CSRF = $_SESSION['csrf'];

// ✅ المواقع
$locations = $conn->query("SELECT id, name FROM locations ORDER BY name ASC")->fetchAll(PDO::FETCH_ASSOC);
$locationsMap = array_column($locations, 'name', 'id');

// ✅ جلب المستخدمين (حسب الصلاحية)
if ($isAdmin || $isHRManager) {
  $stmt = $conn->prepare("
    SELECT users.*, departments.name AS department_name 
    FROM users 
    LEFT JOIN departments ON users.department_id = departments.id
    ORDER BY users.id DESC
  ");
  $stmt->execute();
  $users = $stmt->fetchAll(PDO::FETCH_ASSOC);
} else {
  $stmt_deps = $conn->prepare("SELECT id FROM departments WHERE id = ? OR parent_id = ?");
  $stmt_deps->execute([$department_id, $department_id]);
  $allowed_dept_ids = $stmt_deps->fetchAll(PDO::FETCH_COLUMN) ?: [$department_id];

  $placeholders = implode(',', array_fill(0, count($allowed_dept_ids), '?'));
  $stmt = $conn->prepare("
    SELECT users.*, departments.name AS department_name 
    FROM users 
    LEFT JOIN departments ON users.department_id = departments.id
    WHERE users.department_id IN ($placeholders)
    ORDER BY users.id DESC
  ");
  $stmt->execute($allowed_dept_ids);
  $users = $stmt->fetchAll(PDO::FETCH_ASSOC);
}

// ✅ تجميعات حسب الموقع
$grouped_by_location = [];
$location_counts = [];
foreach ($users as $u) {
  $loc_id = $u['location_id'] ?? 0;
  if ($loc_id) {
    $grouped_by_location[$loc_id][] = $u;
    $location_counts[$loc_id] = ($location_counts[$loc_id] ?? 0) + 1;
  }
}

// ✅ الإدارات وكل أسمائها
$departments_all = $conn->query("SELECT id, name, parent_id FROM departments ORDER BY name ASC")->fetchAll(PDO::FETCH_ASSOC);
$departmentNamesMap = array_column($departments_all, 'name', 'id');

// ✅ تجميعات حسب الإدارة
$grouped_by_department = [];
$department_counts = [];
foreach ($users as $u) {
  $dep_id = $u['department_id'] ?? 0;
  if ($dep_id) {
    $grouped_by_department[$dep_id][] = $u;
    $department_counts[$dep_id] = ($department_counts[$dep_id] ?? 0) + 1;
  }
}

// ✅ المدراء
$managers_stmt = $conn->prepare("
  SELECT u.id, u.name, d.name AS department_name
  FROM users u
  LEFT JOIN departments d ON u.department_id = d.id
  WHERE u.role = 'manager'
  ORDER BY u.name ASC
");
$managers_stmt->execute();
$managers = $managers_stmt->fetchAll(PDO::FETCH_ASSOC);

// ✅ إجازات
$pending_stmt = $conn->prepare("
  SELECT lr.*, u.name AS employee_name, d.name AS department_name
  FROM leave_requests lr
  JOIN users u ON lr.user_id = u.id
  LEFT JOIN departments d ON u.department_id = d.id
  WHERE lr.status LIKE 'بانتظار%'
  ORDER BY lr.created_at DESC
");
$pending_stmt->execute();
$pending_leaves = $pending_stmt->fetchAll(PDO::FETCH_ASSOC);

$today = date('Y-m-d');

$on_leave_stmt = $conn->prepare("
  SELECT lr.*, u.name AS employee_name, d.name AS department_name
  FROM leave_requests lr
  JOIN users u ON lr.user_id = u.id
  LEFT JOIN departments d ON u.department_id = d.id
  WHERE lr.start_date <= :today
    AND lr.end_date   >= :today
    AND lr.status IN ('مقبولة','في إجازة')
  ORDER BY lr.start_date ASC
");
$on_leave_stmt->execute(['today' => $today]);
$on_leave = $on_leave_stmt->fetchAll(PDO::FETCH_ASSOC);

// تحديث الإجازات المنتهية (يفضّل نقلها لكرون)
$conn->prepare("
  UPDATE leave_requests
  SET status = 'منتهية'
  WHERE status IN ('مقبولة','في إجازة')
    AND end_date < CURDATE()
")->execute();

// ✅ العهد بانتظار الموارد البشرية
$pending_assets_stmt = $conn->prepare("
  SELECT ar.*, u.name AS employee_name, d.name AS department_name
  FROM asset_requests ar
  JOIN users u ON ar.user_id = u.id
  LEFT JOIN departments d ON u.department_id = d.id
  WHERE ar.status = 'بانتظار الموارد البشرية'
  ORDER BY ar.created_at DESC
");
$pending_assets_stmt->execute();
$pending_assets = $pending_assets_stmt->fetchAll(PDO::FETCH_ASSOC);

// ✅ شارات الأرقام
$cnt_pending_leaves = (int)$conn->query("SELECT COUNT(*) FROM leave_requests WHERE status LIKE 'بانتظار%'")->fetchColumn();
$cnt_pending_assets = (int)$conn->query("SELECT COUNT(*) FROM asset_requests WHERE status='بانتظار الموارد البشرية'")->fetchColumn();
$cnt_payroll_drafts = (int)$conn->query("SELECT COUNT(*) FROM payrolls WHERE status IN ('draft','pending')")->fetchColumn();
$cnt_settlements_pending = (int)$conn->query("SELECT COUNT(*) FROM settlements WHERE status IN ('draft','pending','بانتظار الاعتماد')")->fetchColumn();

?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
  <meta charset="UTF-8" />
  <title>إدارة المستخدمين</title>
  <script src="https://cdn.tailwindcss.com"></script>
  <!-- Select2 CSS -->
  <link href="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/css/select2.min.css" rel="stylesheet" />
  <!-- jQuery -->
  <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
  <!-- Select2 JS -->
  <script src="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.min.js"></script>

  <style>
    /* ============ Select2 ============ */
    .select2-container .select2-selection--single {
      height: 48px; line-height: 48px; font-size: 13px;
    }
    .select2-container--default .select2-selection--single {
      background-color: <?= $userThemeNavColor ?> !important; border: none;
    }
    .select2-container--default .select2-selection--single .select2-selection__rendered {
      line-height: 48px; color: <?= $userThemeTextColor ?> !important;
    }
    .select2-container--default .select2-selection--single .select2-selection__arrow { height: 48px; }
    .select2-dropdown { font-size: 12px; }
    .select2-results__option { padding: 12px; line-height: 24px; }
    .select2-container { width: 100% !important; }

    /* ============ ثيم الصفحة ============ */
    body, .main-card { background-color: <?= $userThemeColor ?> !important; }
    /* لا نلوّن الروابط/الصفوف بالهوفر بلون الخلفية */
    a:hover { opacity: .9; }

    nav,
    .select2-results__option,
    .select2-search,
    .select2-selection--multiple,
    li,
    .card {
      background-color: <?= $userThemeNavColor ?> !important;
    }

    .text, label, h3, option, h2, th, td, li, select, input, a {
      color: <?= $userThemeTextColor ?> !important;
    }

    option { text-align: center; }

    .text-button { color: #ffffff !important; }
    .move-button { background-color: #4d70f4; border-radius: 15px !important; }
    .card-blue-list { background-color: #4d70f4; color:#fff !important; }
    .card-green-list { background-color: #8BFB75 !important; color: #0e162a !important; }
    .move-button:hover, .view-tasks:hover { background-color: rgb(19, 11, 57) !important; }

    .view-button {
      background-color: #4d70f4; color: #FFFFFF !important; width: 70px !important;
      border-radius: 15px !important; display: inline-block; text-align: center;
      padding: 5px 5px; text-decoration: none; box-sizing: border-box;
    }

    .edit-button {
      background-color: #8BFB75 !important; color: #0e162a !important; width: 70px !important;
      border-radius: 15px !important; display: inline-block; text-align: center;
      padding: 5px 5px; text-decoration: none; box-sizing: border-box;
    }
    .delete-button {
      background-color: #FD7474 !important; color: #FFFFFF !important; width: 70px !important;
      border-radius: 15px !important; display: inline-block; text-align: center;
      padding: 5px 5px; text-decoration: none; box-sizing: border-box;
    }
    .edit-button:hover, .view-button:hover, .download-button:hover, .delete-button:hover {
      color: <?= ($currentUserColorId == 3 ? '#FFFFFF' : '#000000'); ?> !important;
    }

    .hr-btn:hover { background-color: inherit !important; }

    /* ============ أزرار HR (مصغّرة، صفّين، تبدأ يمين) ============ */
    :root{
      --table-max: 64rem;   /* تقريب عرض الجدول */
      --actions-max: 70rem; /* نخلي الأزرار أعرض بشوي */
      --actions-gutter: 1rem;
    }
    .hr-actions-wrap{
      max-width: var(--actions-max);
      margin-inline: auto;
      padding-inline: var(--actions-gutter);
      margin-block: .5rem 1rem;
    }
    .hr-actions-grid{
      direction: rtl;
      display: grid;
      grid-template-rows: repeat(2, auto);
      grid-auto-flow: column dense;
      grid-auto-columns: minmax(220px, 240px); /* زر أعرض لكن مصغّر */
      gap: .7rem 1rem;
      align-items: start;
      justify-content: end; /* تلزق يمين */
    }
    .hr-btn{
      color:#fff; font-weight:700; font-size:.9rem;
      padding:.65rem .95rem; border-radius:.65rem;
      display:inline-flex; align-items:center; gap:.55rem;
      white-space:nowrap; box-shadow:0 1px 0 rgb(0 0 0/.12);
      transition: transform .12s ease, box-shadow .12s ease;
    }
    .hr-btn:hover{ transform: translateY(-1px); box-shadow:0 4px 10px rgb(0 0 0/.15) }
    .hr-btn:focus-visible{ outline:2px solid #ffffff55; outline-offset:2px }
    .hr-badge{
      min-width:1.4rem; height:1.2rem; line-height:1.2rem;
      padding:0 .45rem; border-radius:9999px; font-size:.75rem;
      background:#00000022; text-align:center; transform: translateY(-1px);
    }
    @media (max-width: 640px){
      .hr-actions-wrap{ max-width: 100%; padding-inline: .75rem; }
      .hr-actions-grid{
        grid-auto-flow: row;
        grid-template-rows:none;
        grid-template-columns: repeat(2, minmax(0, 1fr));
        justify-content: center;
      }
      .hr-btn{ justify-content:center; }
    }
  </style>
<style>
/* ===== Drawer Overlay & Panel (ثابت يمين + z-index عالي) ===== */
#hrOverlay{
  position:fixed; inset:0; background:rgba(0,0,0,.45);
  z-index:10000; opacity:0; pointer-events:none; transition:opacity .25s ease;
}
#hrPanel{
  position:fixed; top:0; right:0; height:100vh; width:min(560px, 92vw);
  background-color: <?= $userThemeNavColor ?> !important;
  color: <?= $userThemeTextColor ?> !important;
  z-index:10001; transform: translateX(100%); transition: transform .25s ease;
  box-shadow:-16px 0 32px rgba(0,0,0,.35); display:flex; flex-direction:column;
}

/* حالة الفتح: نضيفها على <body> */
body.hr-drawer-open #hrOverlay{ opacity:1; pointer-events:auto; }
body.hr-drawer-open #hrPanel{ transform: translateX(0); }

/* رأس الدرج */
.hr-drawer-head{ display:flex; align-items:center; justify-content:space-between;
  padding:.75rem 1rem; border-bottom:1px solid rgba(255,255,255,.08); }
.hr-drawer-title{ font-weight:700; font-size:1rem; }
.hr-drawer-close{ font-size:1.35rem; line-height:1; padding:.25rem .5rem; border-radius:.5rem; }
.hr-drawer-close:hover{ background:rgba(255,255,255,.08); }

/* محتوى الدرج */
.hr-drawer-body{ padding:1rem; overflow:auto; height:100%; direction: rtl; }

/* شبكة الأزرار داخل الدرج (اسم جديد عشان ما يتعارض) */
.hr-drawer-grid{
  display:grid; gap:.75rem; direction: rtl;
  grid-template-columns: repeat(2, minmax(0, 1fr));
}
@media (min-width: 768px){ .hr-drawer-grid{ grid-template-columns: repeat(3, minmax(0, 1fr)); } }

/* شكل زر HR */
.hr-btn{
  color:#fff; font-weight:700; font-size:.9rem;
  padding:.65rem .95rem; border-radius:.65rem;
  display:inline-flex; align-items:center; gap:.55rem; justify-content:center;
  white-space:nowrap; box-shadow:0 1px 0 rgb(0 0 0/.12);
  transition: transform .12s ease, box-shadow .12s ease;
}
.hr-btn:hover{ transform: translateY(-1px); box-shadow:0 4px 10px rgb(0 0 0/.15) }
.hr-badge{
  min-width:1.4rem; height:1.2rem; line-height:1.2rem;
  padding:0 .45rem; border-radius:9999px; font-size:.75rem;
  background:#00000022; text-align:center; transform: translateY(-1px);
}

/* زر فتح الدرج */
.hr-toggle{
  background:#4f46e5; color:#fff; font-weight:600; border-radius:.6rem;
  padding:.55rem .85rem; display:inline-flex; align-items:center; gap:.5rem;
}
.hr-toggle:hover{ filter:brightness(.95); }
</style>
<style>
  .detail-row .inline-detail{
    background-color: <?= $userThemeNavColor ?> !important;
    color: <?= $userThemeTextColor ?> !important;
    border: 1px solid rgba(255,255,255,.08);
  }
  .detail-row .detail-cell{ padding: .5rem !important; }
</style>


</head>

<body class="text-right">
  <?php include $_SERVER['DOCUMENT_ROOT'] . '/erp_task_system/navbar/navbar.php'; ?>

  <script>
    // ✅ متغير CSRF للاستخدام في fetch
    const CSRF_TOKEN = '<?= $CSRF ?>';
    $(document).ready(function() {
      $('#theme_color_select').select2({
        templateResult: formatColorOption,
        templateSelection: formatColorOption
      });
      function formatColorOption(option) {
        if (!option.id) return option.text;
        var hexCode = $(option.element).data('hex');
        return $('<span><span style="display:inline-block;width:20px;height:20px;background-color:'+hexCode+';margin-right:12px;vertical-align:middle;border:1px solid #ccc;"></span>'+option.text+'</span>');
      }
      $('#theme_color_select').on('select2:select', function(){ $(this).closest('form').submit(); });
    });
  </script>

  <!-- تعويض ارتفاع الناف بار -->
  <div class="h-20"></div>

  <?php if (isset($_SESSION['error'])): ?>
    <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded relative mb-4">
      <?= $_SESSION['error']; unset($_SESSION['error']); ?>
    </div>
  <?php endif; ?>

  <?php if (isset($_SESSION['success'])): ?>
    <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded relative mb-4">
      <?= $_SESSION['success']; unset($_SESSION['success']); ?>
    </div>
  <?php endif; ?>

<?php
$btnBase = 'hr-btn transition';
$groups = [
  'الأساسيات' => [
    ['label'=>'➕ إضافة مستخدم','href'=>'add.php','class'=>'bg-indigo-600 hover:bg-indigo-700'],
  ],
  'الحضور والورديات' => [
    ['label'=>'🗓️ الورديات','href'=>'work_shifts.php','class'=>'bg-cyan-800 hover:bg-cyan-900'],
  ],
  'الإجازات' => [
    ['label'=>'🌴 طلبات الإجازة','href'=>'leave_requests.php','class'=>'bg-emerald-600 hover:bg-emerald-700','badge'=>$cnt_pending_leaves],
  ],
  'التدريب والتقييم' => [
    ['label'=>'🎓 الدورات','href'=>'trainings_list.php','class'=>'bg-amber-600 hover:bg-amber-700'],
    ['label'=>'⭐ التقييمات','href'=>'evaluations_list.php','class'=>'bg-amber-700 hover:bg-amber-800'],
  ],
  'العهدة والجزاءات' => [
    ['label'=>'📦 إدارة العهد','href'=>'asset_requests_list.php','class'=>'bg-indigo-600 hover:bg-indigo-700','badge'=>$cnt_pending_assets],
    ['label'=>'⏱️ التأخير/الغياب','href'=>'late_absence_penalties/index.php','class'=>'bg-rose-600 hover:bg-rose-700'],
  ],
];
// أقسام مالية تظهر بالصلاحيات فقط
$hrCanSeeMoney = ($isAdmin || $isHRManager);

if ($hrCanSeeMoney) {
  $groups['الرواتب والتسويات'] = [
    ['label'=>'💰 إدارة الرواتب','href'=>'salaries/index.php','class'=>'bg-pink-600 hover:bg-pink-700','badge'=>$cnt_payroll_drafts],
    ['label'=>'📑 المخالصات','href'=>'settlements/index.php','class'=>'bg-fuchsia-600 hover:bg-fuchsia-700','badge'=>$cnt_settlements_pending],
  ];
}
$groups['التأمينات والمواقع'] = [
  ['label'=>'🧾 التأمينات الاجتماعية','href'=>'social_insurance/index.php','class'=>'bg-emerald-600 hover:bg-emerald-700'],
  ['label'=>'📍 إدارة المواقع','href'=>'add_location.php','class'=>'bg-purple-600 hover:bg-purple-700'],
];

// نحول المجموعات إلى قائمة أزرار مسطّحة
$buttons = [];
foreach ($groups as $title => $links) {
  foreach ($links as $ln) {
    $ln['title'] = $title;
    $buttons[] = $ln;
  }
}
?>
<?php
  // تأكيد وجود $buttons (احتياط)
  if (!isset($buttons)) {
    $buttons = [];
    foreach ($groups as $title => $links) {
      foreach ($links as $ln) { $ln['title']=$title; $buttons[]=$ln; }
    }
  }
?>

<!-- زر فتح الدرج -->
<div class="max-w-5xl mx-auto px-4 mt-2 mb-4 flex gap-2 justify-end">
  <button id="hrToggle" class="hr-toggle" type="button" aria-haspopup="dialog" aria-controls="hrPanel">
    <span>☰</span><span>قائمة الإجراءات</span>
  </button>
  <a href="help.php" target="_blank" rel="noopener" class="hr-toggle" aria-label="تعليمات الصفحة">❓ تعليمات</a>
</div>

<!-- Overlay + Panel (مخفية افتراضياً بـ hidden) -->
<div id="hrOverlay" hidden></div>

<aside id="hrPanel" class="shadow-xl" hidden role="dialog" aria-modal="true" aria-labelledby="hrTitle">
  <div class="hr-drawer-head">
    <h3 id="hrTitle" class="hr-drawer-title">قائمة الإجراءات</h3>
    <button id="hrClose" class="hr-drawer-close" type="button" aria-label="إغلاق">×</button>
  </div>

  <div class="hr-drawer-body">
    <div class="hr-drawer-grid">
      <?php foreach ($buttons as $btn): ?>
        <a href="/erp_task_system/users/<?= htmlspecialchars($btn['href']) ?>"
           class="hr-btn <?= htmlspecialchars($btn['class']) ?>"
           title="<?= htmlspecialchars($btn['title'].' — '.preg_replace('/^[^ ]+ /','',$btn['label'])) ?>"
           aria-label="<?= htmlspecialchars($btn['label']) ?>">
          <span><?= htmlspecialchars($btn['label']) ?></span>
          <?php if (!empty($btn['badge'])): ?>
            <span class="hr-badge"><?= (int)$btn['badge'] ?></span>
          <?php endif; ?>
        </a>
      <?php endforeach; ?>
    </div>
  </div>
</aside>


<div class="max-w-5xl mx-auto">
  <div class="flex justify-between items-center mb-6 flex-wrap gap-2">
    <h1 class="text text-xl font-bold">👥 قائمة المستخدمين</h1>
  </div>

  <table class="card w-full text-sm border">
    <thead>
      <tr>
        <th class="p-2 border">الاسم</th>
        <th class="p-2 border">البريد</th>
        <th class="p-2 border">الدور</th>
        <th class="p-2 border">درجة السرية</th>
        <th class="p-2 border">الإدارة</th>
        <th class="p-2 border">تحويل المهام</th>
        <th class="p-2 border">التوقيع</th>
        <th class="p-2 border">عرض التوقيع</th>
        <th class="p-2 border">الإجراءات</th>
      </tr>
    </thead>
    
    <!-- قالب صف تفاصيل الموظف (Inline) -->
<template id="empDetailsRowTemplate">
  <tr id="emp-detail-row" class="detail-row">
    <td class="detail-cell" colspan="9">
      <div class="inline-detail p-4 rounded shadow">
        <div class="flex justify-between items-center mb-3">
          <h2 class="text text-lg font-bold">📄 تفاصيل الموظف</h2>
          <button data-action="close" class="text-red-600 hover:underline text-sm" aria-label="إغلاق">❌ إغلاق</button>
        </div>

        <div class="flex gap-6 items-center">
          <img data-field="image" src="" class="w-24 h-24 object-cover rounded-full border" alt="">
          <div class="grid grid-cols-2 gap-4 flex-1 text-sm">
            <div class="text"><strong>الاسم:</strong> <span data-field="name"></span></div>
            <div class="text"><strong>الرقم الوظيفي:</strong> <span data-field="number"></span></div>
            <div class="text"><strong>البريد:</strong> <span data-field="email"></span></div>
            <div class="text"><strong>المسمى الوظيفي:</strong> <span data-field="job"></span></div>
            <div class="text"><strong>الإدارة:</strong> <span data-field="dept"></span></div>
            <div class="text"><strong>الموقع:</strong> <span data-field="location"></span></div>
            <div class="text"><strong>الحالة:</strong> <span data-field="status"></span></div>
          </div>
        </div>

        <div class="mt-4 flex flex-wrap gap-2 text-left">
          <a data-action="edit"  href="#" class="move-button text-button px-4 py-2 rounded text-sm">✏️ تعديل</a>
          <a data-action="files" href="#" class="bg-gray-700 text-white px-4 py-2 rounded hover:bg-gray-800 text-sm">📂 ملفات الموظف</a>
          <a data-action="eval"  href="#" target="_blank" class="bg-yellow-500 text-white px-4 py-2 rounded hover:bg-yellow-600 text-sm">⭐ تقييم الموظف</a>
          <a data-action="train" href="#" class="bg-green-500 text-white px-4 py-2 rounded hover:bg-green-600 text-sm">🎓 تدريبات الموظف</a>
          <a data-action="late"  href="#" class="bg-red-600 text-white px-4 py-2 rounded hover:bg-red-700 text-sm">⏱️ التأخير/الغياب</a>
        </div>
      </div>
    </td>
  </tr>
</template>

    
    <tbody>
      <?php foreach ($users as $usr): ?>
<tr class="cursor-pointer"
    onclick='toggleEmpRow(this, <?= json_encode([
      "id"               => $usr["id"],
      "name"             => $usr["name"],
      "employee_number"  => $usr["employee_number"],
      "email"            => $usr["email"],
      "job_title"        => $usr["job_title"],
      "department_name"  => $usr["department_name"],
      "location_name"    => isset($locationsMap[$usr["location_id"]]) ? $locationsMap[$usr["location_id"]] : "",
      "employment_status"=> $usr["employment_status"],
      "profile_image"    => $usr["profile_image"] ?? ""
    ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES) ?>)'>
          <td class="p-2 border"><?= htmlspecialchars($usr['name']) ?></td>
          <td class="p-2 border"><?= htmlspecialchars($usr['email']) ?></td>
          <td class="p-2 border"><?= htmlspecialchars($usr['role']) ?></td>
          <td class="p-2 border"><?= htmlspecialchars($usr['secret_level'] ?? '—') ?></td>
          <td class="p-2 border"><?= htmlspecialchars($usr['department_name'] ?? '—') ?></td>
          <td class="p-2 border text-center"><?= !empty($usr['can_transfer_tasks']) ? '✅' : '—' ?></td>
          <td class="p-2 border text-center"><?= !empty($usr['can_sign']) ? '✅' : '—' ?></td>
          <td class="p-2 border text-center">
            <?php if (!empty($usr['signature_path'])): ?>
<a href="<?= htmlspecialchars($usr['signature_path']) ?>"
   target="_blank"
   class="text-blue-600 view-button"
   onclick="event.stopPropagation()">عرض</a>
            <?php else: ?> — <?php endif; ?>
          </td>
<td class="p-2 border no-toggle" onclick="event.stopPropagation()">
  <!-- باقي الأكشنز مثل ما هي -->
            <div class="flex items-center gap-2">
              <a href="edit.php?id=<?= (int)$usr['id'] ?>" class="edit-button">تعديل</a>
              <span class="text-gray-400 text-lg">|</span>
              <!-- ✅ حذف عبر POST + CSRF -->
              <form method="POST" action="delete.php" onsubmit="return confirm('هل أنت متأكد؟')" class="inline">
                <input type="hidden" name="id" value="<?= (int)$usr['id'] ?>">
                <input type="hidden" name="csrf" value="<?= $CSRF ?>">
                <button type="submit" class="delete-button">حذف</button>
              </form>
            </div>
          </td>
        </tr>
      <?php endforeach; ?>
    </tbody>
  </table>
</div>

<?php if (count($pending_leaves) > 0): ?>
  <div class="max-w-4xl mx-auto mt-10">
    <h2 class="text text-lg font-bold mb-4 text-red-600">📌 طلبات إجازة بانتظار الموافقة</h2>
    <table class="card w-full text-sm border rounded shadow">
      <thead>
        <tr>
          <th class="p-2 border">الموظف</th>
          <th class="p-2 border">الإدارة</th>
          <th class="p-2 border">النوع</th>
          <th class="p-2 border">من</th>
          <th class="p-2 border">إلى</th>
          <th class="p-2 border">الحالة</th>
        </tr>
      </thead>
      <tbody>
        <?php foreach ($pending_leaves as $leave): ?>
          <tr>
            <td class="p-2 border"><?= htmlspecialchars($leave['employee_name']) ?></td>
            <td class="p-2 border"><?= htmlspecialchars($leave['department_name']) ?></td>
            <td class="p-2 border"><?= htmlspecialchars($leave['type']) ?></td>
            <td class="p-2 border"><?= htmlspecialchars($leave['start_date']) ?></td>
            <td class="p-2 border"><?= htmlspecialchars($leave['end_date']) ?></td>
            <td class="p-2 border text-orange-600 font-semibold">
              <?= htmlspecialchars($leave['status']) ?>
              <?php
                $isManager = ($role === 'manager' && $leave['department_name'] === $myDeptName);
                $isHR      = ($role === 'manager' && $myDeptName === 'إدارة الموارد البشرية');
                $isAdmin   = ($role === 'admin');

                $canManagerApprove = $isManager && is_null($leave['manager_status']);
                $canHRApprove      = $isHR && is_null($leave['hr_status']);
              ?>
              <?php if ($canManagerApprove || $canHRApprove || $isAdmin): ?>
                <form method="POST" action="update_leave_status.php" class="mt-2 space-y-1">
                  <input type="hidden" name="id" value="<?= (int)$leave['id'] ?>">
                  <input type="hidden" name="csrf" value="<?= $CSRF ?>">
                  <textarea name="note" class="w-full text-xs border rounded px-2 py-1" placeholder="ملاحظة ..."></textarea>
                  <input type="hidden" name="actor" value="<?= $canHRApprove ? 'hr' : ($canManagerApprove ? 'manager' : 'admin') ?>">
                  <div class="flex gap-1 mt-1">
                    <button name="action" value="accept" class="bg-green-600 text-white text-xs px-2 py-1 rounded">موافقة</button>
                    <button name="action" value="reject" class="bg-red-600 text-white text-xs px-2 py-1 rounded">رفض</button>
                  </div>
                </form>
              <?php endif; ?>
            </td>
          </tr>
        <?php endforeach; ?>
      </tbody>
    </table>
  </div>
<?php endif; ?>

<?php if (count($on_leave) > 0): ?>
  <div class="max-w-4xl mx-auto mt-10">
    <h2 class="text-lg font-bold mb-4 text-green-700">🌴 الموظفون في إجازة حالياً</h2>
    <table class="w-full card text-sm border rounded shadow">
      <thead>
        <tr>
          <th class="p-2 border">الموظف</th>
          <th class="p-2 border">الإدارة</th>
          <th class="p-2 border">نوع الإجازة</th>
          <th class="p-2 border">تاريخ البداية</th>
          <th class="p-2 border">تاريخ النهاية</th>
          <th class="p-2 border">ملاحظة</th>
        </tr>
      </thead>
      <tbody>
        <?php foreach ($on_leave as $row): ?>
          <tr>
            <td class="p-2 border"><?= htmlspecialchars($row['employee_name']) ?></td>
            <td class="p-2 border"><?= htmlspecialchars($row['department_name']) ?></td>
            <td class="p-2 border"><?= htmlspecialchars($row['type']) ?></td>
            <td class="p-2 border"><?= htmlspecialchars($row['start_date']) ?></td>
            <td class="p-2 border"><?= htmlspecialchars($row['end_date']) ?></td>
            <td class="p-2 border"><?= nl2br(htmlspecialchars($row['reason'])) ?></td>
          </tr>
        <?php endforeach; ?>
      </tbody>
    </table>
  </div>
<?php endif; ?>

<?php if (count($pending_assets) > 0): ?>
  <div class="max-w-4xl mx-auto mt-10">
    <h2 class="text-lg font-bold mb-4 text-red-600">📌 طلبات عهدة بانتظار موافقة الموارد البشرية</h2>
    <table class="card w-full text-sm border rounded shadow">
      <thead>
        <tr>
          <th class="p-2 border">الموظف</th>
          <th class="p-2 border">الإدارة</th>
          <th class="p-2 border">نوع العهدة</th>
          <th class="p-2 border">تفاصيل</th>
          <th class="p-2 border">التاريخ</th>
          <th class="p-2 border">الإجراءات</th>
        </tr>
      </thead>
      <tbody>
        <?php foreach ($pending_assets as $asset): ?>
          <tr>
            <td class="p-2 border"><?= htmlspecialchars($asset['employee_name']) ?></td>
            <td class="p-2 border"><?= htmlspecialchars($asset['department_name']) ?></td>
            <td class="p-2 border"><?= htmlspecialchars($asset['asset_type_name']) ?></td>
            <td class="p-2 border"><?= nl2br(htmlspecialchars($asset['details'])) ?></td>
            <td class="p-2 border"><?= htmlspecialchars($asset['created_at']) ?></td>
            <td class="p-2 border text-orange-600 font-semibold">
              <form method="POST" action="update_asset_status.php" class="mt-2 space-y-1 asset-approval-form">
                <input type="hidden" name="id" value="<?= (int)$asset['id'] ?>">
                <input type="hidden" name="csrf" value="<?= $CSRF ?>">

                <textarea name="note" class="w-full text-xs border rounded px-2 py-1" placeholder="ملاحظة ..."></textarea>

                <!-- قائمة المدراء (تظهر بعد الضغط على موافقة) -->
                <div class="manager-select-area hidden mt-2">
                  <select name="assigned_manager_id" class="w-full rounded border p-2 mb-2" disabled>
                    <option value="">اختر المدير المختص</option>
                    <?php foreach ($managers as $manager): ?>
                      <option value="<?= (int)$manager['id'] ?>">
                        <?= htmlspecialchars($manager['name']) ?> (<?= htmlspecialchars($manager['department_name']) ?>)
                      </option>
                    <?php endforeach; ?>
                  </select>
                  <select name="approved_by_manager" class="w-full rounded border p-2 mb-2" disabled>
                    <option value="">اختر المدير المعتمد</option>
                    <?php foreach ($managers as $manager): ?>
                      <option value="<?= (int)$manager['id'] ?>">
                        <?= htmlspecialchars($manager['name']) ?> (<?= htmlspecialchars($manager['department_name']) ?>)
                      </option>
                    <?php endforeach; ?>
                  </select>
                </div>

                <div class="flex gap-1 mt-1">
                  <button type="button" class="bg-green-600 text-white text-xs px-2 py-1 rounded show-managers-btn">موافقة</button>
                  <button name="action" value="reject" class="bg-red-600 text-white text-xs px-2 py-1 rounded">رفض</button>
                </div>

                <!-- زر إرسال بعد اختيار المدير -->
                <button name="action" value="accept" class="submit-approval-btn bg-green-700 text-white text-xs px-2 py-1 rounded mt-2 hidden w-full">تحويل للمدير المختص</button>
              </form>
            </td>
          </tr>
        <?php endforeach; ?>
      </tbody>
    </table>
  </div>
<?php endif; ?>

<!-- ✅ كانبان الموظفين حسب الإدارة -->
<h2 class="text-xl font-bold text-center my-6 text-green-800">🏢 الموظفون حسب الإدارات</h2>
<div class="px-4 sm:px-6 lg:px-8">
  <div class="flex gap-4 overflow-x-auto mb-16 mt-10 px-4 sm:px-6 lg:px-8">
    <?php foreach ($departments_all as $dept): ?>
      <div class="card bg-white rounded shadow w-72 flex-shrink-0">
        <h2 class="card-green-list text-white p-2 rounded-t text-center "><?= htmlspecialchars($dept['name']) ?></h2>

        <?php if (!empty($dept['parent_id'])): ?>
          <div class="text text-xs text-center text-gray-500 p-1 border-b border-gray-200">
            ↪ تابعة لـ <?= htmlspecialchars($departmentNamesMap[$dept['parent_id']] ?? '—') ?>
          </div>
        <?php endif; ?>

        <div class="p-2 space-y-2 min-h-[200px]"
             ondragover="event.preventDefault()"
             ondrop="onDeptDrop(event, <?= (int)$dept['id'] ?>)">
          <?php foreach ($grouped_by_department[$dept['id']] ?? [] as $emp): ?>
            <div class="main-card task-card p-2 rounded shadow hover:bg-green-50"
                 draggable="true"
                 ondragstart="onDeptDragStart(event)"
                 data-user-id="<?= (int)$emp['id'] ?>">
              <div class="text font-bold"><?= htmlspecialchars($emp['name']) ?></div>
              <div class="text text-sm text-gray-600"><?= htmlspecialchars($emp['job_title'] ?? '') ?></div>
              <div class="text text-xs text-gray-500"><?= htmlspecialchars($emp['email']) ?></div>
            </div>
          <?php endforeach; ?>
        </div>
      </div>
    <?php endforeach; ?>
  </div>
</div>

<!-- ✅ رسم بياني عدد الموظفين حسب الإدارة -->
<div class="max-w-5xl mx-auto main-card task p-4 rounded shadow mb-10">
  <canvas id="departmentChart" height="90"></canvas>
</div>

<div class="px-4 sm:px-6 lg:px-8">
  <a href="add_location.php" class="bg-purple-600 text-white px-4 py-2 rounded hover:bg-purple-700 inline-block mb-4">📍 إدارة المواقع</a>
</div>

<!-- ✅ الموظفون حسب المواقع -->
<h2 class="text-xl font-bold text-center my-6 text-blue-800">📍 الموظفون حسب المواقع</h2>
<div class="px-4 sm:px-6 lg:px-8">
  <div class="flex gap-4 overflow-x-auto mb-16 mt-10 px-4 sm:px-6 lg:px-8">
    <?php foreach ($locations as $loc): ?>
      <div class="card bg-white rounded shadow w-72 flex-shrink-0">
        <h2 class="card-blue-list text-white p-2 rounded-t text-center "><?= htmlspecialchars($loc['name']) ?></h2>
        <div class="p-2 space-y-2 min-h-[200px]"
             ondragover="event.preventDefault()"
             ondrop="onDrop(event, <?= (int)$loc['id'] ?>)">
          <?php foreach ($grouped_by_location[$loc['id']] ?? [] as $emp): ?>
            <div class="main-card task-card p-2 rounded shadow hover:bg-blue-50"
                 draggable="true"
                 ondragstart="onDragStart(event)"
                 data-user-id="<?= (int)$emp['id'] ?>">
              <div class="text font-bold"><?= htmlspecialchars($emp['name']) ?></div>
              <div class="text text-sm text-gray-600"><?= htmlspecialchars($emp['job_title'] ?? '') ?></div>
              <div class="text text-xs text-gray-500"><?= htmlspecialchars($emp['email']) ?></div>
            </div>
          <?php endforeach; ?>
        </div>
      </div>
    <?php endforeach; ?>
  </div>
</div>

<!-- ✅ الرسم البياني للمواقع -->
<div class="max-w-5xl mx-auto main-card task p-4 rounded shadow mb-10 mt-10">
  <canvas id="locationChart" height="90"></canvas>
</div>

<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script>
  // مواقع
  const locCtx = document.getElementById('locationChart').getContext('2d');
  const locationLabels = <?= json_encode(array_column($locations, 'name'), JSON_UNESCAPED_UNICODE) ?>;
  const locationCounts = <?= json_encode(array_map(fn($loc) => $location_counts[$loc['id']] ?? 0, $locations)) ?>;

  new Chart(locCtx, {
    type: 'bar',
    data: {
      labels: locationLabels,
      datasets: [{
        label: 'عدد الموظفين',
        data: locationCounts,
        backgroundColor: '<?= $chartColor ?>',
        borderColor: 'rgb(255, 255, 255)',
        borderWidth: 1
      }]
    },
    options: {
      responsive: true,
      plugins: {
        legend: { display: false },
        tooltip: {
          callbacks: { label: (c) => c.dataset.label + ': ' + c.parsed.y },
          titleColor: '<?= htmlspecialchars($userThemeSecondTextColor, ENT_QUOTES, 'UTF-8') ?>',
          bodyColor:  '<?= htmlspecialchars($userThemeSecondTextColor, ENT_QUOTES, 'UTF-8') ?>'
        }
      },
      scales: {
        x: { ticks: { color: '<?= htmlspecialchars($userThemeSecondTextColor, ENT_QUOTES, 'UTF-8') ?>' } },
        y: { beginAtZero: true, ticks: { color: '<?= htmlspecialchars($userThemeSecondTextColor, ENT_QUOTES, 'UTF-8') ?>' } }
      }
    }
  });

  // إدارات
  const depLabels = <?= json_encode(array_column($departments_all, 'name'), JSON_UNESCAPED_UNICODE) ?>;
  const depCounts = <?= json_encode(array_map(fn($d) => $department_counts[$d['id']] ?? 0, $departments_all)) ?>;

  new Chart(document.getElementById('departmentChart').getContext('2d'), {
    type: 'bar',
    data: {
      labels: depLabels,
      datasets: [{
        label: 'عدد الموظفين بالإدارة',
        data: depCounts,
        backgroundColor: '<?= $chartColor ?>',
        borderColor: 'rgb(255, 255, 255)',
        borderWidth: 1
      }]
    },
    options: {
      responsive: true,
      plugins: {
        legend: { display: false },
        tooltip: {
          callbacks: { label: (c) => c.dataset.label + ': ' + c.parsed.y },
          titleColor: '<?= htmlspecialchars($userThemeSecondTextColor, ENT_QUOTES, 'UTF-8') ?>',
          bodyColor:  '<?= htmlspecialchars($userThemeSecondTextColor, ENT_QUOTES, 'UTF-8') ?>'
        }
      },
      scales: {
        x: { ticks: { color: '<?= htmlspecialchars($userThemeSecondTextColor, ENT_QUOTES, 'UTF-8') ?>' } },
        y: { beginAtZero: true, ticks: { color: '<?= htmlspecialchars($userThemeSecondTextColor, ENT_QUOTES, 'UTF-8') ?>' } }
      }
    }
  });
</script>

<script>
  let currentEmployeeNumber = null;
  let currentEmployeeId     = null;

  function toggleEmpRow(tr, user){
    // ✔ عرّف rowId أول شي (أفضل نخليه ثابت على id الموظف)
    if (!tr.dataset.rowId) tr.dataset.rowId = 'r' + (user.id ?? Date.now());

    const tbody = tr.parentNode;

    // سكّر أي صف تفاصيل مفتوح
    const existing = document.getElementById('emp-detail-row');
    if (existing) {
      const openFor = existing.dataset.forRowId;
      if (openFor === tr.dataset.rowId) { existing.remove(); return; } // نفس الصف → اقفل وبس
      existing.remove(); // صف آخر → اقفل القديم وكمل
    }

    // جهّز القالب
    const tpl = document.getElementById('empDetailsRowTemplate');
    const row = tpl.content.firstElementChild.cloneNode(true);
    row.id = 'emp-detail-row';
    row.dataset.forRowId = tr.dataset.rowId;

    // خله يساوي عدد الأعمدة تلقائي
    const td = row.querySelector('td');
    td.colSpan = tr.children.length;

    // عبّي البيانات
    row.querySelector('[data-field="name"]').textContent     = user.name || '—';
    row.querySelector('[data-field="number"]').textContent   = user.employee_number || '—';
    row.querySelector('[data-field="email"]').textContent    = user.email || '—';
    row.querySelector('[data-field="job"]').textContent      = user.job_title || '—';
    row.querySelector('[data-field="dept"]').textContent     = user.department_name || '—';
    row.querySelector('[data-field="location"]').textContent = user.location_name || '—';
    row.querySelector('[data-field="status"]').textContent   = user.employment_status || '—';
    const img = row.querySelector('[data-field="image"]');
    img.src = (user.profile_image && user.profile_image !== '') ? ('img/' + user.profile_image) : '';
    img.alt = user.name || '';

    // متغيرات المودالات
    currentEmployeeNumber = user.employee_number || user.id || null;
    currentEmployeeId     = user.id || null;

    // روابط الأكشنز
    row.querySelector('[data-action="edit"]').href = 'edit.php?id=' + user.id;
    row.querySelector('[data-action="eval"]').href = 'add_evaluation.php?id=' + (user.employee_number || user.id);

    row.querySelector('[data-action="files"]').addEventListener('click', (e)=>{ e.preventDefault(); openEmployeeFilesModal(currentEmployeeNumber); });
    row.querySelector('[data-action="train"]').addEventListener('click', (e)=>{ e.preventDefault(); openTrainingsModal(currentEmployeeNumber); });
    row.querySelector('[data-action="late"]').addEventListener('click',  (e)=>{ e.preventDefault(); openLateModal(currentEmployeeId, user.name); });
    row.querySelector('[data-action="close"]').addEventListener('click', ()=> row.remove());

    // حطّه فوق الصف اللي انضغط
    tbody.insertBefore(row, tr);
  }
</script>

<script>
  // ✅ سحب/إفلات — المواقع
  let draggedUserId = null;
  function onDragStart(event) {
    draggedUserId = event.target.dataset.userId;
  }
  function onDrop(event, newLocationId) {
    event.preventDefault();
    if (!draggedUserId) return;
    fetch('update_location_api.php', {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ user_id: draggedUserId, location_id: newLocationId, csrf: CSRF_TOKEN })
    })
    .then(res => res.json())
    .then(data => {
      if (data.success) {
        const draggedElement = document.querySelector(`[data-user-id='${draggedUserId}']`);
        const targetColumn = event.currentTarget;
        if (draggedElement && targetColumn) targetColumn.appendChild(draggedElement);
      } else {
        alert('حدث خطأ أثناء التحديث');
      }
    });
  }
</script>

<script>
  // ✅ سحب/إفلات — الإدارات
  let draggedUserIdDept = null;
  function onDeptDragStart(event) {
    draggedUserIdDept = event.target.dataset.userId;
  }
  function onDeptDrop(event, newDepartmentId) {
    event.preventDefault();
    if (!draggedUserIdDept) return;
    fetch('update_department_api.php', {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ user_id: draggedUserIdDept, department_id: newDepartmentId, csrf: CSRF_TOKEN })
    })
    .then(res => res.json())
    .then(data => {
      if (data.success) {
        const draggedElement = document.querySelector(`[data-user-id='${draggedUserIdDept}']`);
        const targetColumn = event.currentTarget;
        if (draggedElement && targetColumn) targetColumn.appendChild(draggedElement);
      } else {
        alert('حدث خطأ أثناء التحديث');
      }
    });
  }
</script>

<!-- ✅ مودال ملفات الموظف -->
<div id="employeeFilesModal" class="fixed inset-0 z-50 flex items-center justify-center bg-black bg-opacity-50 hidden">
  <div class="bg-white rounded-lg shadow-lg p-6 w-full max-w-lg relative">
    <button onclick="closeEmployeeFilesModal()" class="absolute top-2 left-2 text-gray-600 text-2xl" aria-label="إغلاق">&times;</button>
    <h2 class="font-bold mb-4 text-lg">📂 ملفات الموظف <span id="modalEmpNumber" class="text-blue-600"></span></h2>

    <div id="employeeFilesList" class="mb-6">
      <div class="text-gray-400 text-center">جاري تحميل الملفات ...</div>
    </div>

    <hr class="my-4">

    <form id="employeeFileUploadForm" method="post" enctype="multipart/form-data" action="upload_employee_file.php" target="uploadFileFrame">
      <input type="hidden" name="employee_number" id="modalEmployeeNumberInput">
      <input type="hidden" name="csrf" value="<?= $CSRF ?>">
      <input type="file" name="file" required class="border p-2 w-full mb-3">
      <button type="submit" class="bg-blue-600 text-white px-4 py-2 rounded hover:bg-blue-700">رفع الملف</button>
      <span id="employeeFileUploadResult" class="block mt-3 text-sm"></span>
    </form>
    <iframe name="uploadFileFrame" class="hidden" title="upload"></iframe>
  </div>
</div>

<script>
  // ملفات الموظف
  function openEmployeeFilesModal(empNumber = null) {
    if (!empNumber && !currentEmployeeNumber) { alert('لا يوجد موظف محدد!'); return; }
    currentEmployeeNumber = empNumber || currentEmployeeNumber;
    document.getElementById('employeeFilesModal').classList.remove('hidden');
    document.getElementById('modalEmpNumber').textContent = currentEmployeeNumber;
    document.getElementById('modalEmployeeNumberInput').value = currentEmployeeNumber;
    fetchEmployeeFiles(currentEmployeeNumber);
    document.getElementById('employeeFileUploadResult').textContent = '';
  }
  function closeEmployeeFilesModal() {
    document.getElementById('employeeFilesModal').classList.add('hidden');
  }
  function fetchEmployeeFiles(empNumber) {
    document.getElementById('employeeFilesList').innerHTML = `<div class="text-gray-400 text-center">جاري تحميل الملفات ...</div>`;
    fetch('fetch_employee_files.php?id=' + encodeURIComponent(empNumber))
      .then(res => res.text())
      .then(html => { document.getElementById('employeeFilesList').innerHTML = html; });
  }
  window.uploadFileFrame.onload = function() {
    var content = window.uploadFileFrame.document.body.innerText;
    document.getElementById('employeeFileUploadResult').textContent = content;
    if (content.includes('نجاح') || content.includes('✔')) {
      fetchEmployeeFiles(currentEmployeeNumber);
      setTimeout(() => { document.getElementById('employeeFileUploadResult').textContent = ''; }, 1200);
      document.querySelector('#employeeFileUploadForm input[type="file"]').value = '';
    }
  };
  document.getElementById('employeeFileUploadForm').onsubmit = function() {
    document.getElementById('employeeFileUploadResult').textContent = "جاري رفع الملف...";
    return true;
  }
</script>

<!-- ✅ مودال الدورات التدريبية -->
<div id="trainingsModal" class="fixed inset-0 z-50 flex items-center justify-center bg-black bg-opacity-50 hidden">
  <div class="bg-white rounded-lg shadow-lg p-6 w-full max-w-2xl relative" style="direction: rtl;">
    <button onclick="closeTrainingsModal()" class="absolute top-2 left-2 text-gray-600 text-2xl" aria-label="إغلاق">&times;</button>
    <h2 class="font-bold mb-4 text-lg">🎓 دورات الموظف <span id="modalTrainingsEmpNumber" class="text-blue-600"></span></h2>

    <div id="employeeTrainingsList" class="mb-6">
      <div class="text-gray-400 text-center">جاري تحميل الدورات ...</div>
    </div>
    <hr class="my-4">

    <form id="addTrainingForm" class="space-y-2" onsubmit="return submitAddTraining(event)">
      <input type="hidden" name="employee_id" id="addTrainingEmpNumber">
      <input type="hidden" name="csrf" value="<?= $CSRF ?>">
      <div class="grid grid-cols-1 md:grid-cols-3 gap-2">
        <input type="text" name="course_name" required placeholder="اسم الدورة" class="border p-2 rounded">
        <input type="text" name="provider" required placeholder="الجهة" class="border p-2 rounded">
        <input type="date" name="course_date" required class="border p-2 rounded">
      </div>
      <div class="grid grid-cols-1 md:grid-cols-3 gap-2">
        <input type="number" name="hours" required placeholder="عدد الساعات" class="border p-2 rounded">
        <input type="text" name="employee_feedback" placeholder="ملاحظات" class="border p-2 rounded">
        <input type="file" name="certificate" accept="application/pdf,image/*" class="border p-2 rounded">
      </div>
      <button type="submit" class="bg-green-600 text-white px-4 py-2 rounded hover:bg-green-700">➕ إضافة دورة</button>
      <span id="addTrainingResult" class="block mt-2 text-sm"></span>
    </form>
  </div>
</div>

<script>
  // الدورات
  function openTrainingsModal(empNumber = null) {
    if (!empNumber && !currentEmployeeNumber) { alert('لا يوجد موظف محدد!'); return; }
    currentEmployeeNumber = empNumber || currentEmployeeNumber;
    $('#trainingsModal').removeClass('hidden');
    $('#modalTrainingsEmpNumber').text(currentEmployeeNumber);
    $('#addTrainingEmpNumber').val(currentEmployeeNumber);
    fetchEmployeeTrainings(currentEmployeeNumber);
    $('#addTrainingResult').text('');
  }
  function closeTrainingsModal() {
    $('#trainingsModal').addClass('hidden');
  }
  function fetchEmployeeTrainings(empNumber) {
    $('#employeeTrainingsList').html(`<div class="text-gray-400 text-center">جاري تحميل الدورات ...</div>`);
    fetch('fetch_employee_trainings.php?id=' + encodeURIComponent(empNumber))
      .then(res => res.text())
      .then(html => { $('#employeeTrainingsList').html(html); });
  }
  function submitAddTraining(e) {
    e.preventDefault();
    var form = document.getElementById('addTrainingForm');
    var formData = new FormData(form);
    $('#addTrainingResult').text("جاري الحفظ ...");
    fetch('add_training.php', { method: 'POST', body: formData })
      .then(res => res.text())
      .then(result => {
        $('#addTrainingResult').text(result);
        if (result.includes('✔') || result.includes('نجاح')) {
          fetchEmployeeTrainings(currentEmployeeNumber);
          setTimeout(() => { $('#addTrainingResult').text(''); }, 1200);
          form.reset();
        }
      });
    return false;
  }
</script>

<script>
  // ✅ إظهار قوائم المدراء ديناميكيًا + تمكين required فقط عند الموافقة
  document.querySelectorAll('.show-managers-btn').forEach(function(btn) {
    btn.addEventListener('click', function(e) {
      e.preventDefault();
      var form = btn.closest('.asset-approval-form');
      var area = form.querySelector('.manager-select-area');
      area.classList.remove('hidden');
      // فعّل الحقول وخليها required
      area.querySelectorAll('select').forEach(function(s){
        s.disabled = false;
        s.required = true;
      });
      form.querySelector('.submit-approval-btn').classList.remove('hidden');
      btn.style.display = 'none';
    });
  });
</script>

<!-- ✅ مودال التأخير/الغياب -->
<div id="lateAbsenceModal" class="fixed inset-0 z-50 hidden items-center justify-center bg-black/50">
  <div class="bg-white rounded-lg shadow-lg w-full max-w-5xl relative" style="max-height:85vh; overflow:auto; direction: rtl;">
    <button type="button" onclick="closeLateModal()" class="absolute top-2 left-3 text-2xl" aria-label="إغلاق">&times;</button>

    <div class="p-4 border-b flex flex-wrap items-center gap-3">
      <div class="font-bold">تفاصيل التأخير والغياب: <span id="lateEmpName" class="text-blue-700"></span></div>

      <div class="ml-auto flex items-center gap-2">
        <label>الشهر:</label>
        <select id="lateMonth" class="border rounded px-2 py-1">
          <?php for($m=1;$m<=12;$m++): ?>
            <option value="<?=$m?>"><?=$m?></option>
          <?php endfor; ?>
        </select>

        <label>السنة:</label>
        <input type="number" id="lateYear" class="border rounded px-2 py-1 w-24" min="2000" max="2100" value="<?= date('Y') ?>">

        <button type="button" class="bg-indigo-600 text-white px-3 py-1 rounded" onclick="reloadLateDetails()">عرض</button>
      </div>
    </div>

    <div id="lateAbsenceBody" class="p-4">جاري التحميل...</div>
  </div>
</div>

<script>
  // التأخير/الغياب
  function openLateModal(empId, empName) {
    currentEmployeeId = empId;
    $('#lateEmpName').text(empName || '');
    const now = new Date();
    if (!$('#lateMonth').data('set')) { $('#lateMonth').val(now.getMonth() + 1); $('#lateMonth').data('set', true); }
    if (!$('#lateYear').data('set'))  { $('#lateYear').val(now.getFullYear());    $('#lateYear').data('set',  true); }
    $('#lateAbsenceBody').html('جاري التحميل...');
    $('#lateAbsenceModal').removeClass('hidden').addClass('flex');
    reloadLateDetails();
  }
  function reloadLateDetails() {
    if (!currentEmployeeId) return;
    const m = $('#lateMonth').val();
    const y = $('#lateYear').val();
    $.get('late_absence_penalties/employee_late_details.php', { id: currentEmployeeId, month: m, year: y })
      .done(function (html) { $('#lateAbsenceBody').html(html); })
      .fail(function () { $('#lateAbsenceBody').html("<div class='text-red-600 p-4'>تعذر تحميل البيانات.</div>"); });
  }
  function closeLateModal() {
    $('#lateAbsenceModal').addClass('hidden').removeClass('flex');
  }
  $(document).on('click', '.open-late-modal', function (e) {
    e.preventDefault();
    const empId   = $(this).data('emp-id');
    const empName = $(this).data('emp-name') || '';
    openLateModal(empId, empName);
  });
  $('#lateAbsenceModal').on('click', function (e) {
    if (e.target === this) closeLateModal();
  });
  $(document).on('click', '#lateAbsenceModal .move-button', function () {
    closeLateModal();
  });
</script>

<script>
(function(){
  const body    = document.body;
  const overlay = document.getElementById('hrOverlay');
  const panel   = document.getElementById('hrPanel');
  const openBtn = document.getElementById('hrToggle');
  const closeBtn= document.getElementById('hrClose');

  function openDrawer(){
    if (!overlay || !panel) return;
    overlay.hidden = false; panel.hidden = false;
    // نعطي المتصفح فريم عشان يفعّل الـtransition
    requestAnimationFrame(()=>{
      body.classList.add('hr-drawer-open','overflow-hidden');
    });
  }
  function closeDrawer(){
    body.classList.remove('hr-drawer-open','overflow-hidden');
    // ننتظر نهاية الانتقال ثم نخفي العناصر
    setTimeout(()=>{ overlay.hidden = true; panel.hidden = true; }, 260);
  }

  openBtn && openBtn.addEventListener('click', openDrawer);
  closeBtn && closeBtn.addEventListener('click', closeDrawer);
  overlay && overlay.addEventListener('click', closeDrawer);
  document.addEventListener('keydown', (e)=>{ if(e.key === 'Escape') closeDrawer(); });
})();
</script>

</body>
</html>