<?php
/* /erp_task_system/users/social_insurance/insurance_import.php
   استيراد التأمينات وربط بالموظف عبر national_id (CSV/Excel) */

session_start();

/* ===== Debug قوي عشان نكسر صفحة 500 ونشوف السبب ===== */
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
ini_set('log_errors', 1);
@mkdir(__DIR__ . '/../../logs', 0775, true);
ini_set('error_log', __DIR__ . '/../../logs/insurance_import.log');
error_reporting(E_ALL);
register_shutdown_function(function () {
  $e = error_get_last();
  if ($e && in_array($e['type'], [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR])) {
    http_response_code(500);
    echo "<pre style='direction:ltr;padding:16px;background:#111;color:#eee;white-space:pre-wrap'>";
    echo "FATAL: {$e['message']} in {$e['file']}:{$e['line']}";
    echo "</pre>";
  }
});

/* ===== تحديد المسارات بثبات ===== */
$BASE = rtrim($_SERVER['DOCUMENT_ROOT'], '/') . '/erp_task_system';
$dbPaths = [
  __DIR__ . '/../../config/db.php',
  $BASE . '/config/db.php'
];
$authPaths = [
  __DIR__ . '/../../unauthorized.php',
  $BASE . '/unauthorized.php'
];

$dbLoaded = false;
foreach ($dbPaths as $p) {
  if (file_exists($p)) { require_once $p; $dbLoaded = true; break; }
}
if (!$dbLoaded) {
  die("<pre>DB file not found. Tried:\n- " . implode("\n- ", $dbPaths) . "</pre>");
}

/* صلاحيات */
if (!isset($_SESSION['user']) || !in_array($_SESSION['user']['role'] ?? '', ['admin','manager'])) {
  foreach ($authPaths as $p) {
    if (file_exists($p)) { header("Location: " . str_replace($BASE, '/erp_task_system', $p)); exit; }
  }
  die('unauthorized.php not found.');
}

/* أسماء الأعمدة المتوقعة من ملف التأمينات */
$COLUMNS_MAP = [
  'national_id'           => ['رقم الهوية','national_id','iqama','id'],
  'gosi_number'           => ['رقم المشترك','gosi_number','subscriber_no','رقم التأمينات'],
  'salary'                => ['الأجر المسجل','الراتب','salary','wage'],
  'subscription_date'     => ['بداية الاشتراك','subscription_start','subscription_date','start_date'],
  'end_date'              => ['نهاية الاشتراك','end_date','stop_date'],
  'contribution_employee' => ['حصة الموظف','employee_contribution','emp_share'],
  'contribution_employer' => ['حصة المنشأة','employer_contribution','org_share'],
  'status'                => ['الحالة','status','subscription_status'],
];

/* أدوات مساعدة */
function findCol($headers, $aliases) {
  foreach ($headers as $i => $h) {
    $hNorm = mb_strtolower(trim((string)$h), 'UTF-8');
    foreach ($aliases as $alias) {
      if ($hNorm === mb_strtolower(trim((string)$alias), 'UTF-8')) return $i;
    }
  }
  return null;
}
function toMoney($v){
  $v = trim((string)$v);
  $v = str_replace([","," "], ["",""], $v);
  return ($v !== '' && is_numeric($v)) ? number_format((float)$v, 2, '.', '') : null;
}
function toDate($v){
  $v = trim((string)$v);
  if ($v === '' || $v === '0000-00-00') return null;
  $try = [$v, str_replace('/','-',$v)];
  foreach ($try as $t) {
    $ts = strtotime($t);
    if ($ts && $ts > 0) return date('Y-m-d', $ts);
  }
  return null;
}
function toStatus($v){
  $v = mb_strtolower(trim((string)$v),'UTF-8');
  if (in_array($v, ['نشط','active'])) return 'active';
  if (in_array($v, ['موقوف','متوقف','stopped'])) return 'stopped';
  if ($v === 'معلق' || $v === 'suspended') return 'suspended';
  return 'active';
}

/* نجيب employee_id عبر national_id فقط */
function employeeIdByNational(PDO $conn, $national_id){
  if (!$national_id) return null;
  $q = $conn->prepare("SELECT id FROM users WHERE national_id = ? LIMIT 1");
  $q->execute([$national_id]);
  $id = $q->fetchColumn();
  return $id ? (int)$id : null;
}

/* نتيجة التنفيذ */
$done = ['inserted'=>0,'updated'=>0,'skipped'=>0,'errors'=>0,'notes'=>[]];

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_FILES['file'])) {
  $file = $_FILES['file'];
  if ($file['error'] !== UPLOAD_ERR_OK) {
    $done['notes'][] = 'فشل رفع الملف.';
  } else {
    $tmp = $file['tmp_name'];
    $name = $file['name'];
    $ext  = strtolower(pathinfo($name, PATHINFO_EXTENSION));
    $rows = [];

    try {
      if ($ext === 'csv') {
        // ===== CSV =====
        $fh = fopen($tmp, 'r');
        if (!$fh) throw new Exception('تعذر فتح CSV');
        // إزالة BOM لو موجود
        $first = fgets($fh);
        if ($first === false) throw new Exception('تعذر قراءة رؤوس CSV');
        $first = preg_replace('/^\xEF\xBB\xBF/', '', $first);
        $headers = str_getcsv($first);
        while (($r = fgetcsv($fh)) !== false) { $rows[] = $r; }
        fclose($fh);

      } elseif (in_array($ext, ['xls','xlsx'])) {
        // ===== Excel (xls/xlsx) =====
        $vendorPath = $BASE . '/vendor/autoload.php';
        if (!file_exists($vendorPath)) {
          throw new Exception('مكتبة PhpSpreadsheet غير مثبّتة. ثبّت عبر: composer require phpoffice/phpspreadsheet');
        }
        require_once $vendorPath;

        $spreadsheet = \PhpOffice\PhpSpreadsheet\IOFactory::load($tmp);
        $sheet = $spreadsheet->getActiveSheet();
        $headers = [];
        $highestCol = $sheet->getHighestColumn();
        $highestRow = $sheet->getHighestRow();

        // الصف 1 رؤوس
        foreach ($sheet->rangeToArray("A1:{$highestCol}1", null, true, true, true) as $row) {
          $headers = array_values($row);
        }
        // من الصف 2 بيانات
        for ($r = 2; $r <= $highestRow; $r++) {
          $row = $sheet->rangeToArray("A{$r}:{$highestCol}{$r}", null, true, true, true);
          $rows[] = array_values($row[$r]);
        }

      } else {
        throw new Exception('الامتداد غير مدعوم. ارفع CSV أو XLS/XLSX.');
      }

      // خرائط الفهارس
      $idx = [];
      foreach ($COLUMNS_MAP as $key => $aliases) {
        $idx[$key] = findCol($headers, $aliases);
      }
      if ($idx['national_id'] === null) {
        throw new Exception('لازم عمود national_id (رقم الهوية). عدّل $COLUMNS_MAP لو اسم العمود مختلف.');
      }

      /* UPSERT مبني على UNIQUE(national_id) */
      $sqlUpsert = "
        INSERT INTO social_insurance
          (employee_id, national_id, gosi_number, salary, subscription_date, end_date, 
           contribution_employee, contribution_employer, status, last_updated)
        VALUES
          (:employee_id, :national_id, :gosi_number, :salary, :subscription_date, :end_date, 
           :contribution_employee, :contribution_employer, :status, NOW())
        ON DUPLICATE KEY UPDATE
          employee_id = VALUES(employee_id),
          gosi_number = VALUES(gosi_number),
          salary = VALUES(salary),
          subscription_date = VALUES(subscription_date),
          end_date = VALUES(end_date),
          contribution_employee = VALUES(contribution_employee),
          contribution_employer = VALUES(contribution_employer),
          status = VALUES(status),
          last_updated = NOW()
      ";
      $stmtUpsert = $conn->prepare($sqlUpsert);

      foreach ($rows as $i => $r) {
        try {
          $national_id   = trim((string)$r[$idx['national_id']]);
          if ($national_id === '') {
            $done['skipped']++;
            $done['notes'][] = "سطر ".($i+2).": national_id فاضي.";
            continue;
          }

          $gosi_number   = $idx['gosi_number']   !== null ? trim((string)$r[$idx['gosi_number']])   : null;
          $salary        = $idx['salary']        !== null ? toMoney($r[$idx['salary']])             : null;
          $sub_date      = $idx['subscription_date']!== null ? toDate($r[$idx['subscription_date']]) : null;
          $end_date      = $idx['end_date']      !== null ? toDate($r[$idx['end_date']])            : null;
          $c_emp         = $idx['contribution_employee']!== null ? toMoney($r[$idx['contribution_employee']]) : null;
          $c_org         = $idx['contribution_employer']!== null ? toMoney($r[$idx['contribution_employer']]) : null;
          $status        = $idx['status']        !== null ? toStatus($r[$idx['status']])            : 'active';

          $employee_id = employeeIdByNational($conn, $national_id);
          if (!$employee_id) {
            $done['skipped']++;
            $done['notes'][] = "سطر ".($i+2).": ما لقيت موظف بهوية ($national_id).";
            continue;
          }

          $params = [
            ':employee_id'           => $employee_id,
            ':national_id'           => $national_id,
            ':gosi_number'           => $gosi_number,
            ':salary'                => $salary,
            ':subscription_date'     => $sub_date,
            ':end_date'              => $end_date,
            ':contribution_employee' => $c_emp,
            ':contribution_employer' => $c_org,
            ':status'                => $status,
          ];

          $stmtUpsert->execute($params);
          $rc = $stmtUpsert->rowCount();
          $done['inserted'] += ($rc === 1) ? 1 : 0;
          $done['updated']  += ($rc === 2) ? 1 : 0;

        } catch (Exception $eRow) {
          $done['errors']++;
          $done['notes'][] = "سطر ".($i+2).": خطأ ".$eRow->getMessage();
        }
      }

    } catch (Exception $e) {
      $done['notes'][] = "خطأ أثناء القراءة: ".$e->getMessage();
    }
  }
}
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
  <meta charset="UTF-8">
  <title>استيراد التأمينات الاجتماعية (CSV/Excel)</title>
  <script src="https://cdn.tailwindcss.com"></script>
</head>
<body class="bg-gray-50 p-6">
  <div class="max-w-2xl mx-auto bg-white rounded-xl shadow p-6">
    <h1 class="text-xl font-bold mb-4">🧾 استيراد التأمينات الاجتماعية</h1>

    <form method="post" enctype="multipart/form-data" class="space-y-4">
      <input type="file" name="file" accept=".csv,.xls,.xlsx" required class="block w-full border rounded p-2">
      <button class="px-4 py-2 bg-blue-600 text-white rounded hover:bg-blue-700">استيراد</button>
    </form>

    <?php if (!empty($done['inserted']) || !empty($done['updated']) || !empty($done['errors']) || !empty($done['skipped'])): ?>
      <div class="mt-6">
        <h2 class="font-semibold mb-2">النتيجة:</h2>
        <ul class="list-disc pr-5 space-y-1">
          <li>المضاف: <?= (int)$done['inserted'] ?></li>
          <li>المحدَّث: <?= (int)$done['updated'] ?></li>
          <li>المتجاوز: <?= (int)$done['skipped'] ?></li>
          <li>الأخطاء: <?= (int)$done['errors'] ?></li>
        </ul>
        <?php if (!empty($done['notes'])): ?>
          <details class="mt-3">
            <summary class="cursor-pointer text-sm text-gray-700">ملاحظات</summary>
            <ul class="list-disc pr-5 text-sm mt-2 space-y-1">
              <?php foreach ($done['notes'] as $n): ?>
                <li><?= htmlspecialchars($n, ENT_QUOTES, 'UTF-8') ?></li>
              <?php endforeach; ?>
            </ul>
          </details>
        <?php endif; ?>
      </div>
    <?php endif; ?>

    <hr class="my-6">
    <div class="text-sm text-gray-600">
      <p>💡 لازم ملفك يحتوي عمود <code>national_id</code> ويطابق الأسماء في <code>$COLUMNS_MAP</code>.</p>
      <p>🔒 التحديث مبني على <code>UNIQUE(national_id)</code> بجدول <code>social_insurance</code>.</p>
      <p>📦 لملفات Excel تأكد وجود <code>/erp_task_system/vendor/autoload.php</code> (PhpSpreadsheet).</p>
      <p>📝 لو بقت المشكلة، راجع اللوق: <code>/erp_task_system/logs/insurance_import.log</code>.</p>
    </div>
  </div>
</body>
</html>