<?php
session_start();
require_once '../../config/db.php';
require_once '../../auth.php';
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Define the month and year for which penalties should be calculated
$month = isset($_GET['month']) ? intval($_GET['month']) : intval(date('n'));
$year  = isset($_GET['year'])  ? intval($_GET['year'])  : intval(date('Y'));

// ----------------- Helper Functions -----------------

/**
 * Get the basic salary for an employee for a given month and year.
 */
function get_employee_basic_salary(PDO $conn, int $emp_id, int $month, int $year): float {
    $stmt = $conn->prepare(
        "SELECT COALESCE(SUM(esi.value), 0)
         FROM employee_salary_items AS esi
         JOIN salary_components AS sc ON sc.id = esi.component_id
         WHERE esi.employee_id = ?
           AND sc.type = 'basic'
           AND MONTH(esi.created_at) = ?
           AND YEAR(esi.created_at) = ?"
    );
    $stmt->execute([$emp_id, $month, $year]);
    return (float)$stmt->fetchColumn();
}

/**
 * Get total late hours for an employee in a specific month and year.
 */
function get_employee_late_hours(PDO $conn, int $emp_id, int $month, int $year): float {
    $stmt = $conn->prepare("SELECT ws.start_time
                            FROM users u
                            LEFT JOIN work_shifts ws ON ws.id = u.work_shift_id
                            WHERE u.id = ?");
    $stmt->execute([$emp_id]);
    $shift = $stmt->fetch(PDO::FETCH_ASSOC);
    $shift_start = $shift['start_time'] ?? '08:00:00';

    $att = $conn->prepare("SELECT DATE(`date`) AS day, MIN(TIME(`date`)) AS in_time
                           FROM attendance
                           WHERE user_id = ? AND MONTH(`date`) = ? AND YEAR(`date`) = ? AND status = 1
                           GROUP BY DATE(`date`)");
    $att->execute([$emp_id, $month, $year]);
    $days = $att->fetchAll(PDO::FETCH_ASSOC);

    $total_late_minutes = 0;
    foreach ($days as $d) {
        if ($d['in_time'] > $shift_start) {
            $diff = (strtotime($d['in_time']) - strtotime($shift_start)) / 60;
            $total_late_minutes += $diff;
        }
    }

    return $total_late_minutes / 60;
}

/**
 * Get absent days for an employee by comparing department schedule and approved leave.
 */
function get_employee_absent_days(PDO $conn, int $user_id, int $month, int $year): int {
    // جلب بيانات الموظف
    $stmt = $conn->prepare("SELECT department_id, day_off FROM users WHERE id = ?");
    $stmt->execute([$user_id]);
    $empData = $stmt->fetch(PDO::FETCH_ASSOC);
    $dept_id = (int)$empData['department_id'];
    $day_off = $empData['day_off'];

    // أيام العمل في الإدارة
// جلب أيام العمل الفعلية للإدارة من جدول department_work_days
$stmt2 = $conn->prepare("
    SELECT day_of_week 
    FROM department_work_days 
    WHERE department_id = ? AND is_working = 1
");
$stmt2->execute([$dept_id]);
$dayNames = $stmt2->fetchAll(PDO::FETCH_COLUMN);

// تحويل أسماء الأيام إلى أرقام (PHP: الاثنين = 1 .. الأحد = 7)
$dayMap = [
    'monday'    => 1,
    'tuesday'   => 2,
    'wednesday' => 3,
    'thursday'  => 4,
    'friday'    => 5,
    'saturday'  => 6,
    'sunday'    => 7,
];

$departmentDays = [];
foreach ($dayNames as $dayName) {
    $lower = strtolower(trim($dayName));
    if (isset($dayMap[$lower])) {
        $departmentDays[] = $dayMap[$lower];
    }
}

    // حذف يوم الإجازة الأسبوعية
    if ($day_off !== null && $day_off !== '') {
        $departmentDays = array_diff($departmentDays, [(int)$day_off]);
    }

    // حساب جميع أيام العمل الفعلية
    $num_days = cal_days_in_month(CAL_GREGORIAN, $month, $year);
    $workingDates = [];
    for ($d = 1; $d <= $num_days; $d++) {
        $date = "$year-" . str_pad($month, 2, '0', STR_PAD_LEFT) . "-" . str_pad($d, 2, '0', STR_PAD_LEFT);
        $weekday = (int)date('N', strtotime($date));
        if (in_array($weekday, $departmentDays)) {
            $workingDates[] = $date;
        }
    }

    // جلب أيام الحضور
    $stmt3 = $conn->prepare("SELECT DATE(`date`) AS day
                             FROM attendance
                             WHERE user_id = ? AND MONTH(`date`) = ? AND YEAR(`date`) = ? AND status = 1
                             GROUP BY DATE(`date`)");
    $stmt3->execute([$user_id, $month, $year]);
    $attendanceDays = $stmt3->fetchAll(PDO::FETCH_COLUMN);

    // جلب فترات الإجازات المقبولة (من - إلى)
    $stmt4 = $conn->prepare("SELECT start_date, end_date
                             FROM leave_requests
                             WHERE user_id = ? AND status = 'مقبولة' AND (
                                   (MONTH(start_date) = ? AND YEAR(start_date) = ?) OR
                                   (MONTH(end_date) = ? AND YEAR(end_date) = ?)
                             )");
    $stmt4->execute([$user_id, $month, $year, $month, $year]);
    $leaves = $stmt4->fetchAll(PDO::FETCH_ASSOC);

    // تجميع كل أيام الإجازة داخل هذا الشهر
    $leaveDates = [];
    foreach ($leaves as $leave) {
        $start = new DateTime($leave['start_date']);
        $end   = new DateTime($leave['end_date']);

        while ($start <= $end) {
            $date = $start->format('Y-m-d');
            if ((int)$start->format('n') == $month && (int)$start->format('Y') == $year) {
                $leaveDates[] = $date;
            }
            $start->modify('+1 day');
        }
    }

    // حساب أيام الغياب: أيام العمل - الحضور - الإجازة
    $absentDays = 0;
    foreach ($workingDates as $date) {
        if (!in_array($date, $attendanceDays) && !in_array($date, $leaveDates)) {
            $absentDays++;
        }
    }

    return $absentDays;
}

/**
 * Calculate late deduction based on base salary and late hours.
 */
function calculate_late_deduction(float $base_salary, float $late_hours, int $hours_per_day = 8): float {
    $daily_wage = $base_salary / 30;
    return $late_hours * 1.5 * ($daily_wage / $hours_per_day);
}

/**
 * Calculate absent deduction based on base salary and absent days.
 */
function calculate_absent_deduction(float $base_salary, int $absent_days): float {
    $daily_wage = $base_salary / 30;
    return $absent_days * $daily_wage;
}

// ----------------- Main Logic -----------------

$stmt = $conn->prepare("SELECT id FROM users WHERE employment_status = 'نشط'");
$stmt->execute();
$employees = $stmt->fetchAll(PDO::FETCH_ASSOC);

foreach ($employees as $emp) {
    $empId = (int)$emp['id'];

    $baseSalary       = get_employee_basic_salary($conn, $empId, $month, $year);
    $lateHours        = get_employee_late_hours($conn, $empId, $month, $year);
    $absentDays       = get_employee_absent_days($conn, $empId, $month, $year);
    $lateDeduction    = calculate_late_deduction($baseSalary, $lateHours);
    $absentDeduction  = calculate_absent_deduction($baseSalary, $absentDays);

    $stmtCheck = $conn->prepare("SELECT id FROM late_absence_penalties WHERE user_id = ? AND month = ? AND year = ?");
    $stmtCheck->execute([$empId, $month, $year]);
    $existingId = $stmtCheck->fetchColumn();

    if ($existingId) {
        $stmtUpdate = $conn->prepare("UPDATE late_absence_penalties
                                      SET late_hours = ?, absent_days = ?, late_deduction = ?, absent_deduction = ?, updated_at = NOW()
                                      WHERE user_id = ? AND month = ? AND year = ?");
        $stmtUpdate->execute([$lateHours, $absentDays, $lateDeduction, $absentDeduction, $empId, $month, $year]);
    } else {
        $stmtInsert = $conn->prepare("INSERT INTO late_absence_penalties
            (user_id, month, year, late_hours, absent_days, late_deduction, absent_deduction)
            VALUES (?, ?, ?, ?, ?, ?, ?)");
        $stmtInsert->execute([$empId, $month, $year, $lateHours, $absentDays, $lateDeduction, $absentDeduction]);
    }
}

echo "تم حساب وتحديث خصومات التأخير والغياب بنجاح للشهر $month من سنة $year.";