<?php
session_start();
require_once '../config/db.php';

function verify_and_migrate_password(PDO $conn, array $user, string $password): bool {
    $stored = (string)($user['password'] ?? '');
    $uid    = (int)$user['id'];
    $ok     = false;

    // bcrypt ($2y$...)
    if ($stored && preg_match('/^\$2y\$/', $stored)) {
        $ok = password_verify($password, $stored);
        if ($ok && password_needs_rehash($stored, PASSWORD_DEFAULT)) {
            $conn->prepare("UPDATE users SET password = ? WHERE id = ?")->execute([password_hash($password, PASSWORD_DEFAULT), $uid]);
        }
        return $ok;
    }

    // crypt قديم ($1$ MD5, $5$ SHA-256, $6$ SHA-512)
    if (!$ok && $stored && preg_match('/^\$(1|5|6)\$/', $stored)) {
        $ok = hash_equals(crypt($password, $stored), $stored);
    }

    // md5 hex (32)
    if (!$ok && $stored && preg_match('/^[a-f0-9]{32}$/i', $stored)) {
        $ok = hash_equals(strtolower($stored), md5($password));
    }

    // sha1 hex (40)
    if (!$ok && $stored && preg_match('/^[a-f0-9]{40}$/i', $stored)) {
        $ok = hash_equals(strtolower($stored), sha1($password));
    }

    // نص صريح قديم
    if (!$ok && $stored !== '') {
        $ok = hash_equals($stored, $password);
    }

    // لو نجح بأي طريقة، رحّل لبكربت
    if ($ok) {
        $new = password_hash($password, PASSWORD_DEFAULT);
        $conn->prepare("UPDATE users SET password = ? WHERE id = ?")->execute([$new, $uid]);
    }

    return $ok;
}

$error = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $identifier = trim($_POST['email'] ?? $_POST['login'] ?? '');
    $password   = (string)($_POST['password'] ?? '');

    if ($identifier === '' || $password === '') {
        $error = "❌ فضلاً أدخل المعرّف وكلمة المرور.";
    } else {
        // جلب المستخدم حسب النوع
        if (filter_var($identifier, FILTER_VALIDATE_EMAIL)) {
            $stmt = $conn->prepare("SELECT * FROM users WHERE email = ? LIMIT 1");
            $stmt->execute([$identifier]);
        } elseif (ctype_digit($identifier)) {
            $stmt = $conn->prepare("
                SELECT * FROM users
                WHERE national_id = ? OR employee_number = ?
                LIMIT 1
            ");
            $stmt->execute([$identifier, $identifier]);
        } else {
            $stmt = $conn->prepare("SELECT * FROM users WHERE employee_number = ? LIMIT 1");
            $stmt->execute([$identifier]);
        }

        $user = $stmt->fetch(PDO::FETCH_ASSOC);

        if ($user && verify_and_migrate_password($conn, $user, $password)) {
            // ألوان المستخدم
            $color = null;
            if (!empty($user['color_id'])) {
                $stmtcolor = $conn->prepare("SELECT * FROM colors WHERE id = ? LIMIT 1");
                $stmtcolor->execute([$user['color_id']]);
                $color = $stmtcolor->fetch(PDO::FETCH_ASSOC) ?: null;
            }
            $allColors = $conn->query("SELECT * FROM colors ORDER BY color_name")->fetchAll(PDO::FETCH_ASSOC);

            $_SESSION['user'] = [
                'id'                 => (int)$user['id'],
                'name'               => $user['name'],
                'email'              => $user['email'],
                'role'               => $user['role'],
                'department_id'      => (int)($user['department_id'] ?? 0),
                'color_id'           => (int)($user['color_id'] ?? 0),
                'color'              => $color ?: ['hex_code'=>'#f8fafc','text_color'=>'#111827','nav_color'=>'#233a58','second_text_color'=>'#94a3b8'],
                'colors'             => $allColors,
                'secret_level'       => $user['secret_level'] ?? 'normal',
                'can_transfer_tasks' => (int)($user['can_transfer_tasks'] ?? 0),
                'can_sign'           => (int)($user['can_sign'] ?? 0),
                'signature_path'     => $user['signature_path'] ?? null
            ];

            if (in_array($user['role'], ['admin','manager'])) {
                header("Location: ../tasks/dashboard/index.php");
            } else {
                header("Location: ../tasks/kanban_new.php");
            }
            exit;
        } else {
            $error = $user ? "❌ كلمة المرور غير صحيحة." : "❌ ما لقينا مستخدم بهالمعرّف.";
        }
    }
}
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <title>تسجيل الدخول</title>
    <script src="https://cdn.tailwindcss.com"></script>
</head>
<body class="flex items-center justify-center min-h-screen bg-gray-100">
    <div class="bg-white p-8 rounded shadow-md w-96">
        <h2 class="text-2xl mb-6 text-center font-bold">تسجيل الدخول</h2>
        <?php if (!empty($error)): ?>
            <div class="text-red-500 text-center mb-4"><?= htmlspecialchars($error, ENT_QUOTES, 'UTF-8') ?></div>
        <?php endif; ?>
        <form method="POST" novalidate>
            <div class="mb-4">
                <label class="block mb-1">الإيميل / رقم الهوية / الرقم الوظيفي</label>
                <input type="text" name="email" required class="w-full border rounded px-3 py-2" placeholder="user@domain.com أو 1XXXXXXXXX أو 12345">
            </div>
            <div class="mb-4">
                <label class="block mb-1">كلمة المرور</label>
                <input type="password" name="password" required class="w-full border rounded px-3 py-2">
            </div>
            <button type="submit" class="w-full bg-blue-600 text-white py-2 rounded hover:bg-blue-700">دخول</button>
        </form>
    </div>
</body>
</html>