<?php
// request_leave.php
declare(strict_types=1);
date_default_timezone_set('Asia/Riyadh');

require_once '../config/db.php';
require_once '../auth.php';
if (session_status() === PHP_SESSION_NONE) session_start();

$user = $_SESSION['user'] ?? null;
if (!$user || empty($user['id'])) { header('Location: ../unauthorized.php'); exit; }
$user_id   = (int)$user['id'];
$user_name = $user['name'] ?? '';

/* ===== Helpers ===== */
function hasColumn(PDO $conn, string $table, string $column): bool {
  $stmt = $conn->prepare("
    SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS
    WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ? AND COLUMN_NAME = ?
  ");
  $stmt->execute([$table, $column]);
  return (bool)$stmt->fetchColumn();
}
function parseYmd(string $s): ?DateTime {
  $dt = DateTime::createFromFormat('Y-m-d', $s);
  return ($dt && $dt->format('Y-m-d') === $s) ? $dt : null;
}
function ymd(DateTime $d): string { return $d->format('Y-m-d'); }

/** ترجع بداية "السنة المرضية" بناءً على أقدم طلب مرضي خلال 365 يوم قبل تاريخ معيّن */
function getSickYearStart(PDO $conn, int $user_id, string $refYmd): string {
  // إذا عندنا طلبات قديمة فيها medical_year_start خذها، وإلا خذ أقدم start_date
  $q = $conn->prepare("
    SELECT COALESCE(MIN(CASE WHEN medical_year_start IS NOT NULL THEN medical_year_start END),
                    MIN(start_date)) AS year_start
    FROM leave_requests
    WHERE user_id=? AND (leave_type_code='sick' OR type IN ('إجازة مرضية','مرضية'))
      AND status IN ('pending','approved')
      AND start_date BETWEEN DATE_SUB(?, INTERVAL 365 DAY) AND ?
  ");
  $q->execute([$user_id, $refYmd, $refYmd]);
  $ys = $q->fetchColumn();
  return $ys ?: $refYmd; // لا يوجد؟ نبدأ من تاريخ الطلب الحالي
}

/** يحسب التداخل بين مدتين Y-m-d (شامل الطرفين) */
function overlapDays(string $A1, string $A2, string $B1, string $B2): int {
  if ($A2 < $A1 || $B2 < $B1) return 0;
  $start = max($A1, $B1);
  $end   = min($A2, $B2);
  if ($end < $start) return 0;
  $sd = new DateTime($start);
  $ed = new DateTime($end);
  return (int)$sd->diff($ed)->format('%a') + 1;
}

/**
 * توزيع المرضية داخل السنة المرضية:
 * - أول 30 يوم: أجر كامل
 * - ثم 60 يوم: 75%
 * - ثم 30 يوم: بدون أجر
 * أي أيام فوق 120 تتحسب بدون أجر (unpaid) وتعلّق برسالة تحذير.
 */
function allocateSick(PDO $conn, int $user_id, string $startYmd, string $endYmd): array {
  $yearStart = getSickYearStart($conn, $user_id, $startYmd);

  // كم يوم مرضي مستهلك قبل بداية الطلب الحالي ضمن السنة المرضية؟
  $usedSoFar = 0;
  $q = $conn->prepare("
    SELECT start_date, end_date FROM leave_requests
    WHERE user_id=? AND (leave_type_code='sick' OR type IN ('إجازة مرضية','مرضية'))
      AND status IN ('pending','approved')
      AND end_date < ?  -- قبل بداية الطلب الحالي
      AND end_date >= ? -- داخل السنة المرضية
  ");
  $q->execute([$user_id, $startYmd, $yearStart]);
  foreach ($q->fetchAll(PDO::FETCH_ASSOC) as $r) {
    $usedSoFar += overlapDays($r['start_date'], $r['end_date'], $yearStart, date('Y-m-d', strtotime($startYmd.' -1 day')));
  }

  // أيام الطلب الحالي
  $reqDays = overlapDays($startYmd, $endYmd, $startYmd, $endYmd);

  // حدود الشرائح
  $FULL_CAP  = 30;  // حد نهاية الشريحة الأولى
  $P75_CAP   = 90;  // حد نهاية الشريحة الثانية (30+60)
  $UNPAID_CAP= 120; // حد نهاية الشريحة الثالثة (30+60+30)

  $full_rem  = max(0, $FULL_CAP  - $usedSoFar);
  $full_take = min($reqDays, $full_rem);
  $usedSoFar += $full_take;  $reqDays -= $full_take;

  $p75_rem   = max(0, $P75_CAP   - $usedSoFar);
  $p75_take  = min($reqDays, $p75_rem);
  $usedSoFar += $p75_take;   $reqDays -= $p75_take;

  $unpd_rem  = max(0, $UNPAID_CAP- $usedSoFar);
  $unpd_take = min($reqDays, $unpd_rem);
  $usedSoFar += $unpd_take;  $reqDays -= $unpd_take;

  $excess = $reqDays; // أي بواقي بعد 120
  return [
    'year_start' => $yearStart,
    'days_count' => ($full_take + $p75_take + $unpd_take + $excess),
    'full'       => $full_take,
    'p75'        => $p75_take,
    'unpaid'     => $unpd_take + $excess, // الزائد يتحسب بدون أجر
    'excess'     => $excess
  ];
}

/* ==== تحميل أنواع الإجازات ==== */
$allTypes = $conn->query("
  SELECT code,name_ar,paid_mode,max_days_per_request,once_per_service,min_service_months,needs_proof,auto_from_calendar
  FROM leave_types ORDER BY id
")->fetchAll(PDO::FETCH_ASSOC);
$visibleTypes = array_values(array_filter($allTypes, fn($t)=> (int)$t['auto_from_calendar'] === 0));
$byCode = [];
foreach ($allTypes as $t) $byCode[$t['code']] = $t;
$allowed_codes = array_column($visibleTypes, 'code');

/* ==== CSRF ==== */
if (empty($_SESSION['csrf'])) $_SESSION['csrf'] = bin2hex(random_bytes(16));

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  $type_code  = $_POST['type']        ?? '';
  $start_date = $_POST['start_date']  ?? '';
  $end_date   = $_POST['end_date']    ?? '';
  $reason     = trim($_POST['reason'] ?? '');
  $csrf       = $_POST['csrf']        ?? '';

  if (!$csrf || !hash_equals($_SESSION['csrf'], $csrf)) {
    $_SESSION['error'] = "⚠️ انتهت صلاحية الجلسة. حدّث الصفحة وحاول مرة ثانية.";
    header("Location: request_leave.php"); exit;
  }
  if (!in_array($type_code, $allowed_codes, true)) {
    $_SESSION['error'] = "❌ نوع الإجازة غير مسموح.";
    header("Location: request_leave.php"); exit;
  }
  $rule = $byCode[$type_code] ?? null;
  if (!$rule) { $_SESSION['error'] = "❌ نوع الإجازة غير معروف."; header("Location: request_leave.php"); exit; }

  $start = parseYmd($start_date);
  $end   = parseYmd($end_date);
  if (!$start || !$end) { $_SESSION['error'] = "❌ صيغة التاريخ غير صحيحة."; header("Location: request_leave.php"); exit; }
  if ($end < $start)   { $_SESSION['error'] = "❌ تاريخ النهاية قبل البداية."; header("Location: request_leave.php"); exit; }

  $daysCount = (int)$start->diff($end)->format('%a') + 1;
  if (!empty($rule['max_days_per_request']) && $daysCount > (int)$rule['max_days_per_request']) {
    $_SESSION['error'] = "⚠️ الحد الأقصى لهذا النوع هو {$rule['max_days_per_request']} يوم.";
    header("Location: request_leave.php"); exit;
  }

  // شروط خاصة للحج
  if ($type_code === 'hajj') {
    if (!empty($rule['min_service_months'])) {
      $hireStmt = $conn->prepare("SELECT hire_date FROM users WHERE id=?");
      $hireStmt->execute([$user_id]); $hire_date = $hireStmt->fetchColumn();
      if ($hire_date) {
        $months = (new DateTime($hire_date))->diff(new DateTime('today')); 
        $months = $months->y * 12 + $months->m;
        if ($months < (int)$rule['min_service_months']) {
          $_SESSION['error'] = "⚠️ يشترط لإجازة الحج خدمة لا تقل عن ".(int)$rule['min_service_months']." شهرًا.";
          header("Location: request_leave.php"); exit;
        }
      }
    }
    if (!empty($rule['once_per_service'])) {
      $hasCodeCol = hasColumn($conn, 'leave_requests', 'leave_type_code');
      if ($hasCodeCol) $st = $conn->prepare("SELECT COUNT(*) FROM leave_requests WHERE user_id=? AND leave_type_code='hajj'");
      else             $st = $conn->prepare("SELECT COUNT(*) FROM leave_requests WHERE user_id=? AND type IN ('إجازة حج','حج')");
      $st->execute([$user_id]);
      if ((int)$st->fetchColumn() > 0) { $_SESSION['error']="⚠️ إجازة الحج تُمنح مرة واحدة خلال الخدمة."; header("Location: request_leave.php"); exit; }
    }
  }

  // منع التداخل (pending/approved لو موجود status)
  $hasStatus = hasColumn($conn, 'leave_requests', 'status');
  $ovSql = "SELECT COUNT(*) FROM leave_requests
            WHERE user_id=? AND NOT (end_date < ? OR start_date > ?)";
  if ($hasStatus) $ovSql .= " AND status IN ('pending','approved')";
  $ov = $conn->prepare($ovSql);
  $ov->execute([$user_id, ymd($start), ymd($end)]);
  if ((int)$ov->fetchColumn() > 0) {
    $_SESSION['error'] = "❌ يوجد طلب إجازة متداخل مع نفس المدة.";
    header("Location: request_leave.php"); exit;
  }

  // تجهيز الإدخال
  $cols = ['user_id','type','start_date','end_date','reason'];
  $ph   = ['?','?','?','?','?'];
  $vals = [$user_id, $rule['name_ar'], ymd($start), ymd($end), $reason];

  $hasTypeCode = hasColumn($conn, 'leave_requests', 'leave_type_code');
  $hasCreated  = hasColumn($conn, 'leave_requests', 'created_at');
  $hasDays     = hasColumn($conn, 'leave_requests', 'days_count');
  $hasFull     = hasColumn($conn, 'leave_requests', 'paid_days_full');
  $hasP75      = hasColumn($conn, 'leave_requests', 'paid_days_75');
  $hasUnpaid   = hasColumn($conn, 'leave_requests', 'unpaid_days');
  $hasMYStart  = hasColumn($conn, 'leave_requests', 'medical_year_start');

  if ($hasTypeCode) { $cols[]='leave_type_code'; $ph[]='?'; $vals[]=$type_code; }
  if ($hasDays)     { $cols[]='days_count';      $ph[]='?'; $vals[]=$daysCount; }
  if ($hasStatus)   { $cols[]='status';          $ph[]='?'; $vals[]='pending'; }
  if ($hasCreated)  { $cols[]='created_at';      $ph[]='?'; $vals[] = date('Y-m-d H:i:s'); }

  $extraMsg = '';

  // توزيع المرضية (فقط لو النوع sick)
  if ($type_code === 'sick') {
    $alloc = allocateSick($conn, $user_id, ymd($start), ymd($end));
    if ($hasDays)    { $vals[array_search('days_count', $cols)] = $alloc['days_count']; }
    if ($hasFull)    { $cols[]='paid_days_full'; $ph[]='?'; $vals[]=$alloc['full']; }
    if ($hasP75)     { $cols[]='paid_days_75';   $ph[]='?'; $vals[]=$alloc['p75']; }
    if ($hasUnpaid)  { $cols[]='unpaid_days';    $ph[]='?'; $vals[]=$alloc['unpaid']; }
    if ($hasMYStart) { $cols[]='medical_year_start'; $ph[]='?'; $vals[]=$alloc['year_start']; }
    if ($alloc['excess'] > 0) $extraMsg = " (تنبيه: تجاوزت 120 يوم في السنة المرضية، الزائد محسوب بدون أجر)";
  }

  $sql = "INSERT INTO leave_requests (".implode(',', $cols).") VALUES (".implode(',', $ph).")";
  try {
    $stmt = $conn->prepare($sql);
    $stmt->execute($vals);
    $_SESSION['success'] = "✅ تم إرسال طلب {$rule['name_ar']} بنجاح ({$daysCount} يوم).{$extraMsg}";
  } catch (Throwable $e) {
    $_SESSION['error'] = "❌ فشل حفظ الطلب: " . $e->getMessage();
  }
  header("Location: request_leave.php"); exit;
}
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
  <meta charset="UTF-8">
  <title>طلب إجازة</title>
  <meta name="viewport" content="width=device-width, initial-scale=1" />
  <script src="https://cdn.tailwindcss.com"></script>
</head>
<body class="bg-slate-50 p-6">
  <div class="max-w-xl mx-auto">
    <h1 class="text-2xl font-bold mb-4">📄 طلب إجازة</h1>

    <?php if (!empty($_SESSION['error'])): ?>
      <div class="bg-red-100 text-red-700 px-4 py-2 rounded mb-4">
        <?= htmlspecialchars($_SESSION['error'], ENT_QUOTES, 'UTF-8'); unset($_SESSION['error']); ?>
      </div>
    <?php endif; ?>

    <?php if (!empty($_SESSION['success'])): ?>
      <div class="bg-green-100 text-green-700 px-4 py-2 rounded mb-4">
        <?= htmlspecialchars($_SESSION['success'], ENT_QUOTES, 'UTF-8'); unset($_SESSION['success']); ?>
      </div>
    <?php endif; ?>

    <form method="POST" class="space-y-4 bg-white p-4 rounded shadow">
      <input type="hidden" name="csrf" value="<?= htmlspecialchars($_SESSION['csrf'], ENT_QUOTES, 'UTF-8') ?>">

      <div>
        <label class="block mb-1 text-sm">الموظف</label>
        <input type="text" value="<?= htmlspecialchars($user_name, ENT_QUOTES, 'UTF-8') ?>" disabled class="w-full border rounded px-3 py-2 bg-gray-100">
      </div>

      <div>
        <label class="block mb-1 text-sm">نوع الإجازة</label>
        <select name="type" required class="w-full border rounded px-3 py-2">
          <option value="">اختر النوع</option>
          <?php foreach ($visibleTypes as $t): ?>
            <option value="<?= htmlspecialchars($t['code'], ENT_QUOTES, 'UTF-8') ?>">
              <?= htmlspecialchars($t['name_ar'], ENT_QUOTES, 'UTF-8') ?>
            </option>
          <?php endforeach; ?>
        </select>
      </div>

      <div class="grid grid-cols-1 sm:grid-cols-2 gap-4">
        <div>
          <label class="block mb-1 text-sm">تاريخ البداية</label>
          <input type="date" name="start_date" id="start" required class="w-full border rounded px-3 py-2">
        </div>
        <div>
          <label class="block mb-1 text-sm">تاريخ النهاية</label>
          <input type="date" name="end_date" id="end" required class="w-full border rounded px-3 py-2">
        </div>
      </div>

      <p id="daysInfo" class="text-sm text-slate-600"></p>

      <div>
        <label class="block mb-1 text-sm">سبب الإجازة</label>
        <textarea name="reason" class="w-full border rounded px-3 py-2" rows="3" placeholder="(اختياري)"></textarea>
      </div>

      <button type="submit" class="bg-blue-600 text-white px-4 py-2 rounded hover:bg-blue-700">إرسال الطلب</button>
    </form>
  </div>

  <script>
    const start = document.getElementById('start');
    const end   = document.getElementById('end');
    const info  = document.getElementById('daysInfo');
    function updateEndMin(){
      if (start.value) {
        end.min = start.value;
        if (end.value && end.value < start.value) end.value = start.value;
      } else end.removeAttribute('min');
    }
    function calcDays(){
      if (!start.value || !end.value) { info.textContent=''; return; }
      const s = new Date(start.value), e = new Date(end.value);
      if (e < s) { info.textContent = '⚠️ تاريخ النهاية قبل البداية.'; return; }
      const diff = Math.round((e - s) / 86400000) + 1;
      info.textContent = `عدد الأيام: ${diff}`;
    }
    start.addEventListener('change', ()=>{ updateEndMin(); calcDays(); });
    end.addEventListener('change', calcDays);
    
    
  </script>
</body>
</html>