<?php
session_start();
require_once '../../config/db.php';
require_once '../../auth.php';
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

/* ================== ثيم المستخدم ================== */
$user  = $_SESSION['user'] ?? [];
$role  = $user['role']      ?? 'employee';
$color = $user['color']     ?? ['hex_code'=>'#f8fafc','text_color'=>'#111827','nav_color'=>'#233a58','second_text_color'=>'#94a3b8'];
$userThemeColor            = $color['hex_code']         ?? '#f8fafc';
$userThemeTextColor        = $color['text_color']       ?? '#111827';
$userThemeNavColor         = $color['nav_color']        ?? '#233a58';
$userThemeSecondTextColor  = $color['second_text_color']?? '#94a3b8';
$allColors                 = $user['colors']            ?? [];
$currentUserColorId        = $user['color_id']          ?? null;

/* ================== إعدادات الرواتب/الحساب ================== */
$WORK_DAYS_MONTH   = 30;
$WORK_HOURS_DAY    = 8;
$LATE_MULTIPLIER   = 1.5;   // معامل خصم التأخير
$ABS_MULTIPLIER    = 1.5;   // معامل خصم الغياب
$OT_RATE_WORKING   = 1.5;   // أوفرتايم يوم عمل
$OT_RATE_OFFDAY    = 2.0;   // أوفرتايم يوم راحة
$HALF_DAY_MINUTES  = 240;   // أقل من 4 ساعات = نصف يوم

/* =============== دوال مساعدة =============== */
function month_bounds(int $month, int $year): array {
    $num_days = cal_days_in_month(CAL_GREGORIAN, $month, $year);
    $Y = (int)date('Y'); $M = (int)date('n'); $D = (int)date('j');
    if ($year > $Y || ($year == $Y && $month > $M)) return [0,0,true];  // مستقبل
    $last = $num_days;
    if ($year == $Y && $month == $M) $last = max(0, $D-1);              // إلى أمس
    return [$num_days, $last, false];
}
function get_shift_times(PDO $conn, int $emp_id): array {
    $stmt = $conn->prepare("
        SELECT ws.start_time, ws.end_time
        FROM users u
        LEFT JOIN work_shifts ws ON ws.id = u.work_shift_id
        WHERE u.id = ?
    ");
    $stmt->execute([$emp_id]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC) ?: [];
    $start = ($row['start_time'] ?? '08:00:00') ?: '08:00:00';
    $end   = ($row['end_time']   ?? '16:00:00') ?: '16:00:00';
    return [$start, $end];
}
function get_work_weekdays(PDO $conn, int $emp_id): array {
    $stmt = $conn->prepare("SELECT department_id, day_off FROM users WHERE id=?");
    $stmt->execute([$emp_id]);
    $u = $stmt->fetch(PDO::FETCH_ASSOC) ?: [];
    $dept_id = (int)($u['department_id'] ?? 0);
    $day_off = $u['day_off'] ?? null;

    $stmt = $conn->prepare("SELECT day_of_week FROM department_work_days WHERE department_id=? AND is_working=1");
    $stmt->execute([$dept_id]);
    $names = $stmt->fetchAll(PDO::FETCH_COLUMN);

    $map = ['monday'=>1,'tuesday'=>2,'wednesday'=>3,'thursday'=>4,'friday'=>5,'saturday'=>6,'sunday'=>7];
    $wd = [];
    foreach ($names as $n) { $l=strtolower(trim($n)); if(isset($map[$l])) $wd[]=$map[$l]; }
    if ($day_off !== null && $day_off !== '') $wd = array_values(array_diff($wd, [(int)$day_off]));
    return $wd ?: [1,2,3,4,5];
}
function get_leave_dates(PDO $conn, int $emp_id, int $month, int $year, int $last): array {
    if ($last<=0) return [];
    $stmt=$conn->prepare("
        SELECT start_date,end_date FROM leave_requests
        WHERE user_id=? AND (status='مقبولة' OR status='approved') AND (
          (MONTH(start_date)=? AND YEAR(start_date)=?) OR
          (MONTH(end_date)=? AND YEAR(end_date)=?) OR
          (start_date<=LAST_DAY(CONCAT(?, '-', ?, '-01')) AND end_date>=CONCAT(?, '-', ?, '-01'))
        )
    ");
    $stmt->execute([$emp_id,$month,$year,$month,$year,$year,$month,$year,$month]);
    $rows=$stmt->fetchAll(PDO::FETCH_ASSOC);

    $from=new DateTime(sprintf('%04d-%02d-01',$year,$month));
    $to=new DateTime(sprintf('%04d-%02d-%02d',$year,$month,$last));
    $dates=[];
    foreach($rows as $lv){
        try{$s=new DateTime($lv['start_date']);$e=new DateTime($lv['end_date']);}catch(Exception $e){continue;}
        if($s<$from)$s=clone $from; if($e>$to)$e=clone $to;
        while($s<=$e){ $dates[]=$s->format('Y-m-d'); $s->modify('+1 day'); }
    }
    return $dates;
}
function get_month_attendance(PDO $conn, int $emp_id, int $month, int $year): array {
    $stmt = $conn->prepare("
        SELECT DATE(`date`) AS day,
               MIN(TIME(`check_in`))  AS in_time,
               MAX(TIME(`check_out`)) AS out_time,
               MAX(status)            AS status
        FROM attendance
        WHERE user_id=? AND MONTH(`date`)=? AND YEAR(`date`)=?
        GROUP BY DATE(`date`)
        ORDER BY day
    ");
    $stmt->execute([$emp_id, $month, $year]);
    $out = [];
    while ($r = $stmt->fetch(PDO::FETCH_ASSOC)) {
        $in = $r['in_time']; $outt = $r['out_time'];
        $worked = 0;
        if ($in && $outt) $worked = max(0, (int)round((strtotime($outt)-strtotime($in))/60));
        $r['worked_minutes'] = $worked;
        $out[$r['day']] = $r;
    }
    return $out;
}
function get_employee_absent_days(PDO $conn, int $emp_id, int $month, int $year, int $halfDayMinutes=240): float {
    [$num,$last,$future]=month_bounds($month,$year);
    if($future || $last<=0) return 0.0;

    $wd    = get_work_weekdays($conn,$emp_id);
    $att   = get_month_attendance($conn,$emp_id,$month,$year);
    $leave = get_leave_dates($conn,$emp_id,$month,$year,$last);

    $abs=0.0;
    for($d=1;$d<=$last;$d++){
        $date=sprintf('%04d-%02d-%02d',$year,$month,$d);
        $dow=(int)date('N',strtotime($date));
        if(!in_array($dow,$wd,true)) continue;
        if(in_array($date,$leave,true)) continue;

        $row=$att[$date]??null;
        if(!$row){ $abs += 1.0; continue; }
        $worked=(int)($row['worked_minutes']??0);
        if($worked < $halfDayMinutes) $abs += 0.5;
    }
    return $abs;
}

/* ============== بنود الرواتب (آخر قيمة لكل بند حتى نهاية الشهر) ============== */
function get_employee_basic_salary(PDO $conn, int $emp_id, int $month, int $year): float {
    $period = sprintf('%04d-%02d-01', $year, $month); // نستخدمه مع LAST_DAY()
    $sql = "
        SELECT esi.value
        FROM employee_salary_items AS esi
        JOIN salary_components AS sc ON sc.id = esi.component_id
        WHERE esi.employee_id = ?
          AND sc.type = 'basic'
          AND DATE(esi.created_at) <= LAST_DAY(?)
        ORDER BY esi.created_at DESC, esi.id DESC
        LIMIT 1
    ";
    $stmt = $conn->prepare($sql);
    $stmt->execute([$emp_id, $period]);
    $val = $stmt->fetchColumn();
    return ($val !== false && $val !== null) ? (float)$val : 0.0;
}
function get_employee_component_sum(PDO $conn, int $emp_id, string $type, int $month, int $year): float {
    // يجمع آخر قيمة لكل بند من نوع $type (allowance/ deduction) حتى نهاية الشهر
    $period = sprintf('%04d-%02d-01', $year, $month);
    $sql = "
        SELECT COALESCE(SUM(esi.value), 0) AS total
        FROM salary_components sc
        LEFT JOIN employee_salary_items esi
          ON esi.id = (
            SELECT esi2.id
            FROM employee_salary_items esi2
            WHERE esi2.employee_id = ?
              AND esi2.component_id = sc.id
              AND DATE(esi2.created_at) <= LAST_DAY(?)
            ORDER BY esi2.created_at DESC, esi2.id DESC
            LIMIT 1
          )
        WHERE sc.type = ?
    ";
    $stmt = $conn->prepare($sql);
    $stmt->execute([$emp_id, $period, $type]);
    return (float)$stmt->fetchColumn();
}

/* ============== تأخير وإضافي ============== */
function get_employee_late_hours(PDO $conn, int $emp_id, int $month, int $year, int $grace_minutes = 0): float {
    [$shift_start,] = get_shift_times($conn, $emp_id);
    $days = get_month_attendance($conn, $emp_id, $month, $year);
    $total_late_minutes = 0;
    foreach ($days as $r) {
        $in = $r['in_time'] ?? null;
        $status = $r['status'] ?? '';
        $isPresent = in_array($status, ['حاضر','متأخر'], true) || (string)$status === '1';
        if ($in && $isPresent && $in > $shift_start) {
            $diff = (strtotime($in) - strtotime($shift_start)) / 60;
            $diff = max(0, (int)round($diff - $grace_minutes));
            $total_late_minutes += $diff;
        }
    }
    return $total_late_minutes / 60.0;
}
function calculate_overtime_pay(
    PDO $conn, int $emp_id, int $month, int $year, float $base_salary,
    int $workHoursDay=8, float $rate_working=1.5, float $rate_offday=2.0, int $workDaysMonth=30
): float {
    if ($workDaysMonth<=0 || $workHoursDay<=0 || $base_salary<=0) return 0.0;
    [$shift_start, $shift_end] = get_shift_times($conn, $emp_id);
    $weekdays = get_work_weekdays($conn, $emp_id);
    $att = get_month_attendance($conn, $emp_id, $month, $year);
    $hourly = $base_salary / ($workDaysMonth * $workHoursDay);
    $total = 0.0;
    foreach ($att as $day => $row) {
        $dow = (int)date('N', strtotime($day));
        $in  = $row['in_time'] ?? null;
        $out = $row['out_time'] ?? null;
        if (!$in || !$out) continue;
        $worked = max(0, (int)round((strtotime($out)-strtotime($in))/60));
        if (!in_array($dow, $weekdays, true)) {
            $total += ($hourly * $rate_offday) * ($worked / 60);
        } else {
            if (strtotime($out) > strtotime($shift_end)) {
                $after_end = (int)round((strtotime($out)-strtotime($shift_end))/60);
                $cap       = max(0, $worked - ($workHoursDay*60));
                $ot_mins   = max(0, min($after_end, $cap));
                if ($ot_mins > 0) $total += ($hourly * $rate_working) * ($ot_mins/60);
            }
        }
    }
    return $total;
}

/* ============== التأمينات الاجتماعية (آخر سجل ساري داخل الشهر) ============== */
function get_employee_gosi_deduction(PDO $conn, int $emp_id, int $month, int $year): float {
    $firstDay = sprintf('%04d-%02d-01', $year, $month);
    $sql = "
        SELECT salary, contribution_employee
        FROM social_insurance
        WHERE employee_id = ?
          AND (subscription_date IS NULL OR DATE(subscription_date) <= LAST_DAY(?))
          AND (end_date IS NULL OR DATE(end_date) >= ?)
        ORDER BY COALESCE(last_updated, subscription_date, '1970-01-01') DESC, id DESC
        LIMIT 1
    ";
    $stmt = $conn->prepare($sql);
    $stmt->execute([$emp_id, $firstDay, $firstDay]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$row) return 0.0;

    $base = (float)($row['salary'] ?? 0);
    $pct  = (float)($row['contribution_employee'] ?? 0); // نسبة الموظف
    if ($base <= 0 || $pct <= 0) return 0.0;

    return round($base * ($pct / 100), 2);
}

/* ============== الإجازة المرضية: أثرها الشهري ============== */
function get_sick_impact_monthly(PDO $conn, int $emp_id, int $month, int $year): array {
    // نهاية الشهر (لو هو الحالي فإلى أمس)
    $num_days = cal_days_in_month(CAL_GREGORIAN, $month, $year);
    $Y=(int)date('Y'); $M=(int)date('n'); $D=(int)date('j');
    $monthStart = sprintf('%04d-%02d-01', $year, $month);
    $monthEnd   = ($year==$Y && $month==$M)
        ? sprintf('%04d-%02d-%02d', $year, $month, max(1,$D-1))
        : sprintf('%04d-%02d-%02d', $year, $month, $num_days);

    // بداية سنة مرضية (أقدم طلب مرضي خلال 365 يوم قبل نهاية الشهر)
    $st = $conn->prepare("
        SELECT MIN(start_date) FROM leave_requests
        WHERE user_id=? AND (leave_type_code='sick' OR type IN ('إجازة مرضية','مرضية'))
          AND (status='approved' OR status='مقبولة')
          AND start_date <= ?
          AND start_date >= DATE_SUB(?, INTERVAL 365 DAY)
    ");
    $st->execute([$emp_id, $monthEnd, $monthEnd]);
    $yearStart = $st->fetchColumn();

    if (!$yearStart) {
        $ck = $conn->prepare("
          SELECT MIN(start_date) FROM leave_requests
          WHERE user_id=? AND (leave_type_code='sick' OR type IN ('إجازة مرضية','مرضية'))
            AND (status='approved' OR status='مقبولة')
            AND NOT (end_date < ? OR start_date > ?)
        ");
        $ck->execute([$emp_id, $monthStart, $monthEnd]);
        $yearStart = $ck->fetchColumn() ?: null;
        if (!$yearStart) return ['full'=>0,'p75'=>0,'unpaid'=>0];
    }

    // كل مدد المرضية المتقاطعة مع [yearStart..monthEnd]
    $q = $conn->prepare("
        SELECT start_date, end_date
        FROM leave_requests
        WHERE user_id=? AND (leave_type_code='sick' OR type IN ('إجازة مرضية','مرضية'))
          AND (status='approved' OR status='مقبولة')
          AND NOT (end_date < ? OR start_date > ?)
        ORDER BY start_date
    ");
    $q->execute([$emp_id, $yearStart, $monthEnd]);
    $ranges = $q->fetchAll(PDO::FETCH_ASSOC);
    if (!$ranges) return ['full'=>0,'p75'=>0,'unpaid'=>0];

    // نبني أيام مرضية يوم-بيوم
    $sick = [];
    foreach ($ranges as $r) {
        $from = max($yearStart, $r['start_date']);
        $to   = min($monthEnd, $r['end_date']);
        $d = new DateTime($from); $toD = new DateTime($to);
        while ($d <= $toD) { $sick[$d->format('Y-m-d')] = true; $d->modify('+1 day'); }
    }

    // نمشي من بداية السنة المرضية ونحدد الشريحة
    $full=$p75=$unpaid=0; $ordinal=0;
    $d = new DateTime($yearStart); $end = new DateTime($monthEnd);
    while ($d <= $end) {
        $k = $d->format('Y-m-d');
        if (isset($sick[$k])) {
            $ordinal++;
            $inMonth = ($k >= $monthStart && $k <= $monthEnd);
            if     ($ordinal <= 30) { if ($inMonth) $full++; }
            elseif ($ordinal <= 90) { if ($inMonth) $p75++; }
            else                    { if ($inMonth) $unpaid++; } // بعد 90 = بدون أجر (وبعد 120 يظل بدون أجر)
        }
        $d->modify('+1 day');
    }
    return ['full'=>$full,'p75'=>$p75,'unpaid'=>$unpaid];
}

/* ================== مدخلات الصفحة ================== */
$selectedMonth = isset($_GET['month']) ? (int)$_GET['month'] : (isset($_POST['month']) ? (int)$_POST['month'] : (int)date('n'));
$selectedYear  = isset($_GET['year'])  ? (int)$_GET['year']  : (isset($_POST['year'])  ? (int)$_POST['year']  : (int)date('Y'));
$calcLate      = isset($_GET['calc_late']) ? (int)$_GET['calc_late'] : 0; // 0/1
$calcOT        = isset($_GET['calc_ot'])   ? (int)$_GET['calc_ot']   : 0; // 0/1
$calcAbs       = isset($_GET['calc_abs'])  ? (int)$_GET['calc_abs']  : 1; // 0/1
$calcGosi      = isset($_GET['calc_gosi']) ? (int)$_GET['calc_gosi'] : 1; // 0/1

/* ================== جلب الموظفين ================== */
$stmt = $conn->prepare("SELECT u.id, u.name, u.job_title, u.work_shift_id FROM users u WHERE u.employment_status = 'نشط'");
$stmt->execute();
$employees = $stmt->fetchAll(PDO::FETCH_ASSOC);

/* ================== حفظ المسير ================== */
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $month = (int)$_POST['month'];
    $year  = (int)$_POST['year'];
    $created_by = $user['id'] ?? null;
    $status = 'draft';
    $payroll_employees = $_POST['payroll'] ?? [];

    $stmt = $conn->prepare("INSERT INTO payrolls (month, year, created_by, status, created_at) VALUES (?, ?, ?, ?, NOW())");
    $stmt->execute([$month, $year, $created_by, $status]);
    $payroll_id = $conn->lastInsertId();

    foreach ($payroll_employees as $emp_id => $row) {
        $base_salary     = (float)($row['base_salary'] ?? 0);
        $allowance       = (float)($row['allowance']   ?? 0);
        $other_deduction = (float)($row['deduction']   ?? 0);
        $gosi_deduction  = (float)($row['gosi_deduction'] ?? 0); // هيدن
        $sick_deduction  = (float)($row['sick_deduction'] ?? 0); // جديد
        $absent_days     = (float)($row['absent_days'] ?? 0);

        // نجمع الخصومات المحاسبية + التأمينات + المرضية
        $total_deductions = $other_deduction + $gosi_deduction + $sick_deduction;

        $stmt2 = $conn->prepare("
            INSERT INTO payroll_items (payroll_id, user_id, base_salary, total_allowances, total_deductions, absent_days)
            VALUES (?, ?, ?, ?, ?, ?)
        ");
        $stmt2->execute([$payroll_id, $emp_id, $base_salary, $allowance, $total_deductions, $absent_days]);
    }

    $_SESSION['success'] = "تم إنشاء المسير بنجاح.";
    header("Location: index.php");
    exit;
}

/* ================== AJAX: إعادة صفوف الجدول ================== */
if (isset($_GET['ajax']) && (int)$_GET['ajax'] === 1) {
    $month    = (int)($_GET['month'] ?? date('n'));
    $year     = (int)($_GET['year']  ?? date('Y'));
    $calcLate = (int)($_GET['calc_late'] ?? 0);
    $calcOT   = (int)($_GET['calc_ot']   ?? 0);
    $calcAbs  = (int)($_GET['calc_abs']  ?? 1);
    $calcGosi = (int)($_GET['calc_gosi'] ?? 1);

    foreach ($employees as $emp) {
        $empId       = (int)$emp['id'];
        $base_salary = get_employee_basic_salary($conn, $empId, $month, $year);
        $allowance   = get_employee_component_sum($conn, $empId, 'allowance', $month, $year);
        $deduction   = get_employee_component_sum($conn, $empId, 'deduction', $month, $year);
        $absent_days = (float)(get_employee_absent_days($conn, $empId, $month, $year, $HALF_DAY_MINUTES) ?? 0);

        // خصم التأخير
        $late_hours = $calcLate ? get_employee_late_hours($conn, $empId, $month, $year) : 0;
        $late_ded   = $calcLate ? ($late_hours * $LATE_MULTIPLIER * (($base_salary / $WORK_DAYS_MONTH) / $WORK_HOURS_DAY)) : 0;

        // الإضافي
        $overtime_pay = $calcOT ? calculate_overtime_pay($conn, $empId, $month, $year, $base_salary, $WORK_HOURS_DAY, $OT_RATE_WORKING, $OT_RATE_OFFDAY, $WORK_DAYS_MONTH) : 0;

        // خصم الغياب
        $absence_ded = 0.0;
        if ($calcAbs) {
            $salary_day = $WORK_DAYS_MONTH ? ($base_salary / $WORK_DAYS_MONTH) : 0;
            $absence_ded = $absent_days * ($salary_day * $ABS_MULTIPLIER);
        }

        // التأمينات
        $gosi_ded = $calcGosi ? get_employee_gosi_deduction($conn, $empId, $month, $year) : 0.0;

        // المرضية (خصم 25% لأيام p75 و100% لأيام unpaid)
        $impact     = get_sick_impact_monthly($conn, $empId, $month, $year);
        $dailyBasic = $WORK_DAYS_MONTH ? ($base_salary / $WORK_DAYS_MONTH) : 0;
        $sick_ded   = ($impact['p75'] * $dailyBasic * 0.25) + ($impact['unpaid'] * $dailyBasic);

        $gross = $base_salary + $allowance + $overtime_pay;
        $net   = $gross - $deduction - $late_ded - $absence_ded - $gosi_ded - $sick_ded;

        echo "<tr class='border-b'>";
        echo "<td>" . htmlspecialchars((string)($emp['name'] ?? '')) . "</td>";
        echo "<td>" . htmlspecialchars((string)($emp['job_title'] ?? '')) . "</td>";
        echo "<td><input type='number' step='0.01' name='payroll[{$empId}][base_salary]' value='".number_format($base_salary, 2, '.', '')."' class='w-32 border rounded px-2 py-1'></td>";
        echo "<td><input type='number' step='0.01' name='payroll[{$empId}][allowance]' value='".number_format($allowance, 2, '.', '')."' class='w-24 border rounded px-2 py-1'></td>";
        echo "<td><input type='number' step='0.01' name='payroll[{$empId}][deduction]' value='".number_format($deduction, 2, '.', '')."' class='w-24 border rounded px-2 py-1'></td>";
        echo "<td><input type='number' step='0.5'  name='payroll[{$empId}][absent_days]' value='".htmlspecialchars($absent_days)."' class='w-16 border rounded px-2 py-1'></td>";
        echo "<td>".number_format($gross, 2)."</td>";
        echo "<td class='col-late'>".number_format($late_ded, 2)."</td>";
        echo "<td class='col-ot'>".number_format($overtime_pay, 2)."</td>";
        echo "<td class='col-gosi'>".number_format($gosi_ded, 2)."<input type='hidden' name='payroll[{$empId}][gosi_deduction]' value='".number_format($gosi_ded,2,'.','')."'></td>";
        echo "<td class='col-sick'>".number_format($sick_ded, 2)."<input type='hidden' name='payroll[{$empId}][sick_deduction]' value='".number_format($sick_ded,2,'.','')."'></td>";
        echo "<td class='col-abs'>".number_format($absence_ded, 2)."</td>";
        echo "<td>".number_format($net, 2)."</td>";
        echo "</tr>";
    }
    exit;
}
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
  <meta charset="UTF-8">
  <title>إنشاء مسير رواتب جديد</title>
  <script src="https://cdn.tailwindcss.com"></script>
  <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>

  <!-- Select2 (لأنك تستخدمه بالنفbar) -->
  <link href="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/css/select2.min.css" rel="stylesheet" />
  <script src="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.min.js"></script>

  <style>
    body { background-color: <?= htmlspecialchars($userThemeColor) ?> !important; }
    nav, .card { background-color: <?= htmlspecialchars($userThemeNavColor) ?> !important; }
    .text, label, option, h2, select, a { color: <?= htmlspecialchars($userThemeTextColor) ?> !important; }
    .col-late { display: <?= $calcLate ? 'table-cell' : 'none' ?>; }
    .col-ot   { display: <?= $calcOT   ? 'table-cell' : 'none' ?>; }
    .col-gosi { display: <?= $calcGosi ? 'table-cell' : 'none' ?>; }
    .col-abs  { display: <?= $calcAbs  ? 'table-cell' : 'none' ?>; }
    .col-sick { display: table-cell; } /* تظهر دائمًا */
  </style>
</head>
<body class="bg-gray-100">

<?php include $_SERVER['DOCUMENT_ROOT'] . '/erp_task_system/navbar/navbar.php'; ?>
<div class="h-20"></div>

<!-- محتوى الصفحة -->
<div class="max-w-6xl mx-auto mt-10 p-6 bg-white rounded-xl shadow">
<style>.rtl-left{ margin-right:auto; }</style>

<div class="flex items-center gap-2 mb-4">
  <h1 class="text-2xl font-bold">إنشاء مسير رواتب جديد</h1>
  <a href="help_payroll_create.php" target="_blank"
     class="rtl-left inline-flex items-center gap-2 px-3 py-1.5 rounded-xl bg-gray-200 text-gray-900 hover:bg-gray-300">
❓ تعليمات
  </a>
</div>

  <?php if (isset($_SESSION['success'])): ?>
    <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded relative mb-4">
      <?= $_SESSION['success']; unset($_SESSION['success']); ?>
    </div>
  <?php endif; ?>

  <form method="POST" class="space-y-6">
    <!-- اختيار الشهر والسنة + أزرار الحساب -->
    <div class="flex flex-wrap items-end gap-4 mb-2">
      <div>
        <label for="month" class="block font-bold mb-1">الشهر:</label>
        <select id="month" name="month" class="border rounded px-2 py-1">
          <?php
          $months = [1=>'يناير',2=>'فبراير',3=>'مارس',4=>'أبريل',5=>'مايو',6=>'يونيو',7=>'يوليو',8=>'أغسطس',9=>'سبتمبر',10=>'أكتوبر',11=>'نوفمبر',12=>'ديسمبر'];
          foreach ($months as $num=>$name) {
              $sel = ($num == $selectedMonth) ? 'selected' : '';
              echo "<option value='$num' $sel>$name</option>";
          }
          ?>
        </select>
      </div>
      <div>
        <label for="year" class="block font-bold mb-1">السنة:</label>
        <select id="year" name="year" class="border rounded px-2 py-1">
          <?php
          $current = (int)date('Y');
          for ($y = $current; $y >= $current - 5; $y--) {
              $sel = ($y == $selectedYear) ? 'selected' : '';
              echo "<option value='$y' $sel>$y</option>";
          }
          ?>
        </select>
      </div>

      <!-- مفاتيح تشغيل/إيقاف -->
      <input type="hidden" id="calc_late" value="<?= (int)$calcLate ?>">
      <input type="hidden" id="calc_ot"   value="<?= (int)$calcOT   ?>">
      <input type="hidden" id="calc_abs"  value="<?= (int)$calcAbs  ?>">
      <input type="hidden" id="calc_gosi" value="<?= (int)$calcGosi ?>">

      <div class="flex flex-wrap items-center gap-2">
        <button type="button" id="btn-calc-late" class="px-3 py-2 rounded bg-emerald-600 text-white hover:bg-emerald-700">حساب التأخير</button>
        <button type="button" id="btn-no-late"   class="px-3 py-2 rounded bg-gray-600 text-white hover:bg-gray-700">بدون خصم التأخير</button>

        <button type="button" id="btn-calc-ot"   class="px-3 py-2 rounded bg-indigo-600 text-white hover:bg-indigo-700">حساب الإضافي</button>
        <button type="button" id="btn-no-ot"     class="px-3 py-2 rounded bg-gray-600 text-white hover:bg-gray-700">بدون إضافي</button>

        <button type="button" id="btn-calc-abs"  class="px-3 py-2 rounded bg-rose-600 text-white hover:bg-rose-700">حساب الغياب</button>
        <button type="button" id="btn-no-abs"    class="px-3 py-2 rounded bg-gray-600 text-white hover:bg-gray-700">بدون خصم الغياب</button>

        <button type="button" id="btn-calc-gosi" class="px-3 py-2 rounded bg-amber-600 text-white hover:bg-amber-700">حساب التأمينات</button>
        <button type="button" id="btn-no-gosi"   class="px-3 py-2 rounded bg-gray-600 text-white hover:bg-gray-700">بدون التأمينات</button>
      </div>
    </div>

    <!-- جدول الموظفين -->
    <div class="overflow-x-auto">
      <table class="w-full text-center border rounded-xl overflow-hidden">
        <thead>
          <tr class="bg-gray-100">
            <th>الموظف</th>
            <th>المسمى الوظيفي</th>
            <th>الراتب الأساسي</th>
            <th>البدلات</th>
            <th>الخصومات</th>
            <th>أيام غياب</th>
            <th>الإجمالي</th>
            <th class="col-late">خصم التأخير</th>
            <th class="col-ot">الإضافي</th>
            <th class="col-gosi">التأمينات</th>
            <th class="col-sick">خصم مرضي</th>
            <th class="col-abs">خصم الغياب</th>
            <th>صافي الراتب</th>
          </tr>
        </thead>
        <tbody id="payroll-table">
        <?php
        foreach ($employees as $emp):
          $empId       = (int)$emp['id'];
          $base_salary = get_employee_basic_salary($conn, $empId, $selectedMonth, $selectedYear);
          $allowance   = get_employee_component_sum($conn, $empId, 'allowance', $selectedMonth, $selectedYear);
          $deduction   = get_employee_component_sum($conn, $empId, 'deduction', $selectedMonth, $selectedYear);
          $absent_days = (float)(get_employee_absent_days($conn, $empId, $selectedMonth, $selectedYear, $HALF_DAY_MINUTES) ?? 0);

          // خصم التأخير
          $late_hours = $calcLate ? get_employee_late_hours($conn, $empId, $selectedMonth, $selectedYear) : 0;
          $late_ded   = $calcLate ? ($late_hours * $LATE_MULTIPLIER * (($base_salary / $WORK_DAYS_MONTH) / $WORK_HOURS_DAY)) : 0;

          // الإضافي
          $overtime_pay = $calcOT ? calculate_overtime_pay($conn, $empId, $selectedMonth, $selectedYear, $base_salary, $WORK_HOURS_DAY, $OT_RATE_WORKING, $OT_RATE_OFFDAY, $WORK_DAYS_MONTH) : 0;

          // خصم الغياب
          $absence_ded = 0.0;
          if ($calcAbs) {
            $salary_day  = $WORK_DAYS_MONTH ? ($base_salary / $WORK_DAYS_MONTH) : 0;
            $absence_ded = $absent_days * ($salary_day * $ABS_MULTIPLIER);
          }

          // التأمينات
          $gosi_ded = $calcGosi ? get_employee_gosi_deduction($conn, $empId, $selectedMonth, $selectedYear) : 0.0;

          // المرضية
          $impact     = get_sick_impact_monthly($conn, $empId, $selectedMonth, $selectedYear);
          $dailyBasic = $WORK_DAYS_MONTH ? ($base_salary / $WORK_DAYS_MONTH) : 0;
          $sick_ded   = ($impact['p75'] * $dailyBasic * 0.25) + ($impact['unpaid'] * $dailyBasic);

          $gross = $base_salary + $allowance + $overtime_pay;
          $net   = $gross - $deduction - $late_ded - $absence_ded - $gosi_ded - $sick_ded;
        ?>
          <tr class="border-b">
            <td><?= htmlspecialchars($emp['name'] ?? '') ?></td>
            <td><?= htmlspecialchars((string)($emp['job_title'] ?? '')) ?></td>
            <td><input type="number" step="0.01" name="payroll[<?= $empId ?>][base_salary]" value="<?= number_format($base_salary, 2, '.', '') ?>" class="w-32 border rounded px-2 py-1"></td>
            <td><input type="number" step="0.01" name="payroll[<?= $empId ?>][allowance]"   value="<?= number_format($allowance, 2, '.', '')   ?>" class="w-24 border rounded px-2 py-1"></td>
            <td><input type="number" step="0.01" name="payroll[<?= $empId ?>][deduction]"  value="<?= number_format($deduction, 2, '.', '')  ?>" class="w-24 border rounded px-2 py-1"></td>
            <td><input type="number" step="0.5"  name="payroll[<?= $empId ?>][absent_days]" value="<?= htmlspecialchars($absent_days) ?>" class="w-16 border rounded px-2 py-1"></td>
            <td><?= number_format($gross, 2) ?></td>
            <td class="col-late"><?= number_format($late_ded, 2) ?></td>
            <td class="col-ot"><?= number_format($overtime_pay, 2) ?></td>
            <td class="col-gosi">
              <?= number_format($gosi_ded, 2) ?>
              <input type="hidden" name="payroll[<?= $empId ?>][gosi_deduction]" value="<?= number_format($gosi_ded, 2, '.', '') ?>">
            </td>
            <td class="col-sick">
              <?= number_format($sick_ded, 2) ?>
              <input type="hidden" name="payroll[<?= $empId ?>][sick_deduction]" value="<?= number_format($sick_ded, 2, '.', '') ?>">
            </td>
            <td class="col-abs"><?= number_format($absence_ded, 2) ?></td>
            <td><?= number_format($net, 2) ?></td>
          </tr>
        <?php endforeach; ?>
        </tbody>
      </table>
    </div>

    <button type="submit" class="bg-blue-600 text-white px-8 py-2 rounded-xl hover:bg-blue-800 transition">
      حفظ المسير
    </button>
  </form>
</div>

<script>
  function toggleLateColumn(show){ if (show) $('.col-late').show(); else $('.col-late').hide(); }
  function toggleOTColumn(show){   if (show) $('.col-ot').show();   else $('.col-ot').hide();   }
  function toggleAbsColumn(show){  if (show) $('.col-abs').show();  else $('.col-abs').hide();  }
  function toggleGosiColumn(show){ if (show) $('.col-gosi').show(); else $('.col-gosi').hide(); }

  function refreshTable() {
    var month    = $('#month').val();
    var year     = $('#year').val();
    var calcLate = $('#calc_late').val();
    var calcOT   = $('#calc_ot').val();
    var calcAbs  = $('#calc_abs').val();
    var calcGosi = $('#calc_gosi').val();

    $.get(window.location.pathname, {
      ajax: 1, month: month, year: year,
      calc_late: calcLate, calc_ot: calcOT, calc_abs: calcAbs, calc_gosi: calcGosi,
      _: Date.now() // منع الكاش
    }, function(res) {
      $('#payroll-table').html(res);
      toggleLateColumn(calcLate === '1');
      toggleOTColumn(calcOT === '1');
      toggleAbsColumn(calcAbs === '1');
      toggleGosiColumn(calcGosi === '1');
    });
  }

  // تغييرات الشهر/السنة
  $('#month, #year').on('change', refreshTable);

  // أزرار التأخير
  $('#btn-calc-late').on('click', function(){ $('#calc_late').val('1'); toggleLateColumn(true);  refreshTable(); });
  $('#btn-no-late').on('click',  function(){ $('#calc_late').val('0'); toggleLateColumn(false); refreshTable(); });

  // أزرار الإضافي
  $('#btn-calc-ot').on('click',  function(){ $('#calc_ot').val('1');   toggleOTColumn(true);    refreshTable(); });
  $('#btn-no-ot').on('click',    function(){ $('#calc_ot').val('0');   toggleOTColumn(false);   refreshTable(); });

  // أزرار الغياب
  $('#btn-calc-abs').on('click', function(){ $('#calc_abs').val('1');  toggleAbsColumn(true);   refreshTable(); });
  $('#btn-no-abs').on('click',   function(){ $('#calc_abs').val('0');  toggleAbsColumn(false);  refreshTable(); });

  // أزرار التأمينات
  $('#btn-calc-gosi').on('click', function(){ $('#calc_gosi').val('1'); toggleGosiColumn(true);  refreshTable(); });
  $('#btn-no-gosi').on('click',   function(){ $('#calc_gosi').val('0'); toggleGosiColumn(false); refreshTable(); });

  // حالة البداية
  $(function(){
    toggleLateColumn($('#calc_late').val() === '1');
    toggleOTColumn($('#calc_ot').val()   === '1');
    toggleAbsColumn($('#calc_abs').val() === '1');
    toggleGosiColumn($('#calc_gosi').val() === '1');
  });
</script>
</body>
</html>
