<?php
/* /erp_task_system/users/settlements/api/create_settlement.php */
@ob_clean();
header('Content-Type: application/json; charset=utf-8');
date_default_timezone_set('Asia/Riyadh');

session_start();
require_once '../../../config/db.php';
require_once '../../../authorize.php'; // لو تستخدم تحقّق صلاحيات

if (!isset($_SESSION['user'])) { echo json_encode(['ok'=>false,'msg'=>'login']); exit; }

/* ⭐ إنشاء الجداول لو ما كانت موجودة (مرّة وحدة) */
try {
  $conn->exec("
    CREATE TABLE IF NOT EXISTS settlements (
      id INT AUTO_INCREMENT PRIMARY KEY,
      code VARCHAR(32) UNIQUE,
      employee_id INT NOT NULL,
      contract_id INT NULL,
      initiated_by ENUM('employee','employer','mutual') NOT NULL,
      termination_reason ENUM('resignation','contract_end','termination','redundancy','death','other') NOT NULL,
      notice_date DATE NULL,
      last_working_day DATE NOT NULL,
      basic_salary_snapshot DECIMAL(12,2) NOT NULL,
      daily_wage_snapshot DECIMAL(12,2) NOT NULL,
      service_days_total INT NOT NULL,
      pay_method ENUM('bank','cash','cheque') DEFAULT 'bank',
      reason_note TEXT NULL,
      with_deductions TINYINT(1) DEFAULT 1,
      gross_additions DECIMAL(12,2) DEFAULT 0,
      total_deductions DECIMAL(12,2) DEFAULT 0,
      net_payable DECIMAL(12,2) DEFAULT 0,
      created_by INT NULL,
      created_at DATETIME DEFAULT CURRENT_TIMESTAMP
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
  ");

  $conn->exec("
    CREATE TABLE IF NOT EXISTS settlement_items (
      id INT AUTO_INCREMENT PRIMARY KEY,
      settlement_id INT NOT NULL,
      item_code VARCHAR(64) NOT NULL,      -- eos, notice, unused_leave, last_salary_days, loan, penalties, other_deduction...
      item_label VARCHAR(255) NOT NULL,    -- نص عربي واضح للطباعة
      is_addition TINYINT(1) NOT NULL,     -- 1 إضافة، 0 خصم
      amount DECIMAL(12,2) NOT NULL,
      details JSON NULL,
      FOREIGN KEY (settlement_id) REFERENCES settlements(id) ON DELETE CASCADE
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
  ");

  $conn->exec("
    CREATE TABLE IF NOT EXISTS settlement_logs (
      id INT AUTO_INCREMENT PRIMARY KEY,
      settlement_id INT NOT NULL,
      action VARCHAR(64) NOT NULL,         -- created, updated, recomputed, printed...
      note TEXT NULL,
      created_by INT NULL,
      created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
      FOREIGN KEY (settlement_id) REFERENCES settlements(id) ON DELETE CASCADE
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
  ");
} catch (Throwable $e) {
  echo json_encode(['ok'=>false,'msg'=>'db_init: '.$e->getMessage()]); exit;
}

/* 📨 مدخلات */
$employee_id            = (int)($_POST['employee_id'] ?? 0);
$contract_id            = isset($_POST['contract_id']) && $_POST['contract_id']!=='' ? (int)$_POST['contract_id'] : null;
$initiated_by           = $_POST['initiated_by'] ?? '';
$termination_reason     = $_POST['termination_reason'] ?? '';
$notice_date            = $_POST['notice_date'] ?: null;
$last_working_day       = $_POST['last_working_day'] ?? '';
$basic_snapshot         = (float)($_POST['basic_salary_snapshot'] ?? 0);
$daily_snapshot         = (float)($_POST['daily_wage_snapshot'] ?? 0);
$service_days_total     = (int)($_POST['service_days_total'] ?? 0);
$pay_method             = $_POST['pay_method'] ?? 'bank';
$reason_note            = trim($_POST['reason_note'] ?? '');
$with_deductions        = (int)($_POST['with_deductions'] ?? 1);

/* (اختياري) باراميترات مساعدة لو تبغى تعبّيها من الواجهة مستقبلاً */
$unused_leave_days      = isset($_POST['unused_leave_days']) ? max(0,(int)$_POST['unused_leave_days']) : 0;
$worked_days_final_month= isset($_POST['worked_days_final_month']) ? max(0,(int)$_POST['worked_days_final_month']) : 0;

/* ✅ تحقّق أساسي */
if ($employee_id <= 0 || !$last_working_day || $basic_snapshot <= 0 || $daily_snapshot <= 0 || $service_days_total <= 0) {
  echo json_encode(['ok'=>false,'msg'=>'بيانات ناقصة']); exit;
}
if (!in_array($initiated_by, ['employee','employer','mutual'])) {
  echo json_encode(['ok'=>false,'msg'=>'initiated_by invalid']); exit;
}
if (!in_array($termination_reason, ['resignation','contract_end','termination','redundancy','death','other'])) {
  echo json_encode(['ok'=>false,'msg'=>'termination_reason invalid']); exit;
}
if (!in_array($pay_method, ['bank','cash','cheque'])) {
  $pay_method = 'bank';
}

/* 🧮 دوال الحساب */
function years_from_days($days){ return $days / 365.0; }

/**
 * مكافأة نهاية الخدمة (تقريبًا وفق الشائع):
 * - الأساس الشهري = آخر أجر شهري (نستخدم لقطة الراتب الأساسي هنا؛ لو تبغى تضيف بدلات ثابتة عدّل الأساس).
 * - القاعدة: نصف شهر لكل سنة من أول خمس سنوات + شهر لكل سنة بعد ذلك.
 * - الاستقالة (resignation): <2 سنوات = 0، 2-5 = ثلث، 5-10 = ثلثين، ≥10 = كامل.
 * - انتهاء عقد/إنهاء من المنشأة/ظروف/وفاة: عادة يُصرف كامل الاستحقاق.
 */
function calc_eos($monthly_base, $service_years, $termination_reason){
  if ($service_years <= 0 || $monthly_base <= 0) return 0.0;

  $first_part  = min($service_years, 5.0) * (0.5 * $monthly_base);
  $second_part = max($service_years - 5.0, 0.0) * (1.0 * $monthly_base);
  $full_entitlement = $first_part + $second_part;

  if ($termination_reason === 'resignation') {
    if ($service_years < 2)       return 0.0;
    elseif ($service_years < 5)   return $full_entitlement * (1.0/3.0);
    elseif ($service_years < 10)  return $full_entitlement * (2.0/3.0);
    else                          return $full_entitlement;
  }
  // غير الاستقالة: نصرف كامل الاستحقاق
  return $full_entitlement;
}

/**
 * بدل إشعار (تبسيط): إن ما تم الالتزام بالإشعار 30 يوم (أو أقل)
 * نعوّض الفارق × الأجر اليومي. تقدر تغيّر 30 إلى 60 بحسب نوع العقد عندك.
 */
function calc_notice_comp($notice_date, $last_day, $daily_wage, $initiated_by) {
  if (!$last_day || $daily_wage <= 0) return 0.0;
  $required = 30; // تقدر تربطه بنوع العقد
  if (!$notice_date) {
    // ما فيه إشعار
    return ($initiated_by === 'employer') ? ($required * $daily_wage) : 0.0;
  }
  $d1 = new DateTime($notice_date);
  $d2 = new DateTime($last_day);
  $diff = (int)$d1->diff($d2)->format('%a'); // فرق الأيام
  $shortage = max(0, $required - $diff);
  // غالبًا نطبقها إذا المنهي صاحب العمل
  return ($initiated_by === 'employer' && $shortage > 0) ? ($shortage * $daily_wage) : 0.0;
}

try {
  $conn->beginTransaction();

  /* 🔢 حساب الأساسات */
  $service_years = years_from_days($service_days_total);
  $monthly_base  = $basic_snapshot;           // لو تبغى تضيف بدلات ثابتة، اجمعها هنا
  $eos_amount    = calc_eos($monthly_base, $service_years, $termination_reason);
  $notice_comp   = calc_notice_comp($notice_date, $last_working_day, $daily_snapshot, $initiated_by);
  $unused_leave  = $unused_leave_days * $daily_snapshot;
  $last_salary   = $worked_days_final_month * $daily_snapshot;

  /* 🧾 خصومات (اختيارية) — افتراضياً صفر. تقدر تربطها لاحقاً بـ payroll/loans/penalties */
  $loan_remaining    = 0.0; // مثال: استعلام من loans WHERE user_id=? AND status='active'
  $penalties_total   = 0.0; // مثال: غياب/تأخيرات غير مبررة
  $other_deductions  = 0.0;

  if ($with_deductions) {
    // TODO: اربط هنا بالجدول/الجداول الفعلية عندك واستبدل الأصفار بالقيم المحسوبة
    // $loan_remaining   = fetchLoansRemaining($conn, $employee_id);
    // $penalties_total  = fetchPenalties($conn, $employee_id, $last_working_day);
    // $other_deductions = ...
  }

  /* ➕➖ التجميع */
  $additions = $eos_amount + $notice_comp + $unused_leave + $last_salary;
  $deducs    = $loan_remaining + $penalties_total + $other_deductions;
  $net       = max(0, round($additions - $deducs, 2));

  /* 🆔 كود المخالصة */
  $code = 'STL'.date('ymdHis').mt_rand(100,999);

  /* 📝 إدخال أصل المخالصة */
  $stmt = $conn->prepare("INSERT INTO settlements
    (code, employee_id, contract_id, initiated_by, termination_reason, notice_date, last_working_day,
     basic_salary_snapshot, daily_wage_snapshot, service_days_total, pay_method, reason_note,
     with_deductions, gross_additions, total_deductions, net_payable, created_by)
    VALUES
    (:code,:employee_id,:contract_id,:initiated_by,:termination_reason,:notice_date,:last_working_day,
     :basic,:daily,:service_days,:pay_method,:note,:with_deductions,:adds,:deducs,:net,:created_by)
  ");
  $stmt->execute([
    ':code'            => $code,
    ':employee_id'     => $employee_id,
    ':contract_id'     => $contract_id,
    ':initiated_by'    => $initiated_by,
    ':termination_reason'=> $termination_reason,
    ':notice_date'     => $notice_date,
    ':last_working_day'=> $last_working_day,
    ':basic'           => $basic_snapshot,
    ':daily'           => $daily_snapshot,
    ':service_days'    => $service_days_total,
    ':pay_method'      => $pay_method,
    ':note'            => $reason_note,
    ':with_deductions' => $with_deductions,
    ':adds'            => $additions,
    ':deducs'          => $deducs,
    ':net'             => $net,
    ':created_by'      => $_SESSION['user']['id'] ?? null,
  ]);
  $settlement_id = (int)$conn->lastInsertId();

  /* 📦 تفصيل البنود */
  $insItem = $conn->prepare("INSERT INTO settlement_items
    (settlement_id,item_code,item_label,is_addition,amount,details)
    VALUES (:sid,:code,:label,:is_add,:amount,:details)");

  // مكافأة نهاية الخدمة
  if ($eos_amount > 0) {
    $insItem->execute([
      ':sid'=>$settlement_id, ':code'=>'eos', ':label'=>'مكافأة نهاية الخدمة',
      ':is_add'=>1, ':amount'=>$eos_amount,
      ':details'=>json_encode(['years'=>$service_years,'monthly_base'=>$monthly_base], JSON_UNESCAPED_UNICODE)
    ]);
  }
  // بدل إشعار
  if ($notice_comp > 0) {
    $insItem->execute([
      ':sid'=>$settlement_id, ':code'=>'notice', ':label'=>'تعويض بدل إشعار',
      ':is_add'=>1, ':amount'=>$notice_comp,
      ':details'=>json_encode(['required_days'=>30], JSON_UNESCAPED_UNICODE)
    ]);
  }
  // رصيد إجازات غير مستخدم
  if ($unused_leave > 0) {
    $insItem->execute([
      ':sid'=>$settlement_id, ':code'=>'unused_leave', ':label'=>'صرف رصيد الإجازات',
      ':is_add'=>1, ':amount'=>$unused_leave,
      ':details'=>json_encode(['days'=>$unused_leave_days], JSON_UNESCAPED_UNICODE)
    ]);
  }
  // راتب الأيام الأخيرة
  if ($last_salary > 0) {
    $insItem->execute([
      ':sid'=>$settlement_id, ':code'=>'last_salary_days', ':label'=>'راتب عن أيام العمل الأخيرة',
      ':is_add'=>1, ':amount'=>$last_salary,
      ':details'=>json_encode(['days'=>$worked_days_final_month], JSON_UNESCAPED_UNICODE)
    ]);
  }
  // خصومات
  if ($with_deductions) {
    if ($loan_remaining > 0) {
      $insItem->execute([
        ':sid'=>$settlement_id, ':code'=>'loan', ':label'=>'متبقي سلف/قروض',
        ':is_add'=>0, ':amount'=>$loan_remaining, ':details'=>json_encode([], JSON_UNESCAPED_UNICODE)
      ]);
    }
    if ($penalties_total > 0) {
      $insItem->execute([
        ':sid'=>$settlement_id, ':code'=>'penalties', ':label'=>'جزاءات/غياب',
        ':is_add'=>0, ':amount'=>$penalties_total, ':details'=>json_encode([], JSON_UNESCAPED_UNICODE)
      ]);
    }
    if ($other_deductions > 0) {
      $insItem->execute([
        ':sid'=>$settlement_id, ':code'=>'other_deduction', ':label'=>'خصومات أخرى',
        ':is_add'=>0, ':amount'=>$other_deductions, ':details'=>json_encode([], JSON_UNESCAPED_UNICODE)
      ]);
    }
  }

  /* سجلّ */
  $lg = $conn->prepare("INSERT INTO settlement_logs (settlement_id, action, note, created_by)
                        VALUES (:sid,'created',:note,:uid)");
  $lg->execute([
    ':sid'=>$settlement_id,
    ':note'=>'إنشاء مخالصة أولية مع '.($with_deductions?'احتساب':'عدم احتساب').' الخصومات',
    ':uid'=>$_SESSION['user']['id'] ?? null
  ]);

  $conn->commit();

  echo json_encode(['ok'=>true,'id'=>$settlement_id,'code'=>$code,'net'=>$net]);
} catch (Throwable $e) {
  if ($conn->inTransaction()) $conn->rollBack();
  echo json_encode(['ok'=>false,'msg'=>$e->getMessage()]);
}
