<?php
// /erp_task_system/users/settlements/api/generate_items.php
if (session_status() === PHP_SESSION_NONE) session_start();
header('Content-Type: application/json; charset=utf-8');

require_once __DIR__ . '/../_helpers.php';
require_roles(['admin','manager','hr','HR']);

try {
  $uid = current_user_id();
  if (!$uid) throw new Exception('login');

  $sid = (int)($_POST['settlement_id'] ?? 0);
  if ($sid <= 0) throw new Exception('bad_id');

  $settlement = get_settlement_row($GLOBALS['conn'], $sid);
  assert_editable($settlement);

  $employeeId = (int)($settlement['employee_id'] ?? 0);
  if ($employeeId <= 0) throw new Exception('employee_missing');

  $includeDeductions = (int)($_POST['include_deductions'] ?? 0) === 1;

  $leaveDaysOpt      = isset($_POST['leave_days']) ? (float)$_POST['leave_days'] : null;
  $lastSalaryDaysOpt = isset($_POST['last_salary_days']) ? (int)$_POST['last_salary_days'] : null;
  $manualBase        = isset($_POST['eos_monthly_base']) ? (float)$_POST['eos_monthly_base'] : null;

  $lastDay = $settlement['last_working_day'] ?? ($settlement['last_day'] ?? ($settlement['end_date'] ?? null));
  if (!$lastDay) throw new Exception('last_day_missing');

  $reason    = $settlement['termination_reason'] ?? $settlement['reason'] ?? 'termination';
  $initiated = $settlement['initiated_by'] ?? 'employer';

  // بيانات الموظف
  $u = $GLOBALS['conn']->prepare("SELECT id, name, hire_date, leave_balance, annual_leave_days FROM users WHERE id=? LIMIT 1");
  $u->execute([$employeeId]);
  $userRow = $u->fetch(PDO::FETCH_ASSOC);
  if (!$userRow) throw new Exception('user_not_found');

  // بداية الخدمة: من العقد أولاً ثم users.hire_date
  $c = $GLOBALS['conn']->prepare("SELECT start_date FROM contracts WHERE user_id=? ORDER BY start_date ASC LIMIT 1");
  $c->execute([$employeeId]);
  $contractStart = $c->fetchColumn();
  $hireDate = $contractStart ?: ($userRow['hire_date'] ?? null);
  if (!$hireDate) throw new Exception('hire_date_missing');

  // أجور: شامل + تفصيل الأساس/البدلات
  $monthlyBase = get_monthly_wage_snapshot(
    $GLOBALS['conn'],
    $employeeId,
    ($manualBase && $manualBase>0) ? $manualBase : null,
    isset($settlement['basic_salary_snapshot']) ? (float)$settlement['basic_salary_snapshot'] : null
  );

  $fixed = get_fixed_components($GLOBALS['conn'], $employeeId);
  $basicMonthly = (float)$fixed['basic']; // الأساس الشهري فقط (بدون بدلات)
  $allowSum     = max(0.0, $fixed['total'] - $basicMonthly);

  $years = years_of_service($hireDate, $lastDay);
  $leaveDays = ($leaveDaysOpt !== null) ? max(0.0, (float)$leaveDaysOpt) : get_user_leave_balance($GLOBALS['conn'], $employeeId);
  $lastPay = calc_last_period_pay($monthlyBase, $lastDay, $lastSalaryDaysOpt);

  $GLOBALS['conn']->beginTransaction();

  // احذف البنود التلقائية السابقة
  $GLOBALS['conn']->prepare("DELETE FROM settlement_items WHERE settlement_id=? AND source='auto'")->execute([$sid]);

  $created = [];

  // 1) راتب الأيام الأخيرة (إضافة)
  if ($monthlyBase > 0 && $lastPay['amount'] > 0) {
    insert_auto_item(
      $GLOBALS['conn'], $sid, 'addition', 'LAST_SALARY',
      'راتب الأيام الأخيرة',
      $lastPay['amount'],
      [
        'monthly_wage_total' => $monthlyBase,
        'monthly_basic'      => $basicMonthly,
        'monthly_allowances' => $allowSum,
        'daily_rate_total'   => $lastPay['daily'],
        'days'               => $lastPay['days'],
        'last_day'           => $lastDay
      ]
    );
    $created[] = ['code'=>'LAST_SALARY','amount'=>$lastPay['amount']];
  }

  // 2) تعويض الإجازات (إضافة)
  if ($monthlyBase > 0 && $leaveDays > 0) {
    $daily = round($monthlyBase / 30.0, 4);
    $lpAmt = round($daily * $leaveDays, 2);
    insert_auto_item(
      $GLOBALS['conn'], $sid, 'addition', 'LEAVE_PAYOUT',
      'تعويض الإجازات السنوية',
      $lpAmt,
      [
        'monthly_wage_total' => $monthlyBase,
        'daily_rate'         => $daily,
        'leave_days'         => $leaveDays
      ]
    );
    $created[] = ['code'=>'LEAVE_PAYOUT','amount'=>$lpAmt];
  }

  // 3) مكافأة نهاية الخدمة (إضافة)
  if ($monthlyBase > 0 && $years > 0) {
    $eos = calc_eos_reward($monthlyBase, $years, (string)$reason);
    if ($eos > 0) {
      insert_auto_item(
        $GLOBALS['conn'], $sid, 'addition', 'EOS', 'مكافأة نهاية الخدمة',
        round($eos, 2),
        [
          'base_monthly_wage' => $monthlyBase,
          'years'             => round($years, 4),
          'reason'            => $reason,
          'initiated_by'      => $initiated,
          'formula'           => 'first5=0.5m/y, rest=1m/y; resignation scaling'
        ]
      );
      $created[] = ['code'=>'EOS','amount'=>round($eos,2)];
    }
  }

  // 4) الخصومات (اختياري)
  if ($includeDeductions) {
    // 4.1 GOSI على الجزء الأساسي من راتب الأيام الأخيرة (نسبة 10%)
    if ($basicMonthly > 0 && $lastPay['days'] > 0) {
      $basicDaily = $basicMonthly / 30.0;
      $gosiBase   = round($basicDaily * $lastPay['days'], 2);
      $gosiDed    = round($gosiBase * 0.10, 2); // 10% حصة الموظف
      if ($gosiDed > 0) {
        insert_auto_item(
          $GLOBALS['conn'], $sid, 'deduction', 'GOSI_DED',
          'خصم التأمينات الاجتماعية (GOSI) على راتب الأيام الأخيرة',
          $gosiDed,
          [
            'basic_monthly'  => $basicMonthly,
            'basic_daily'    => round($basicDaily,4),
            'last_days'      => $lastPay['days'],
            'gosi_rate'      => 0.10,
            'gosi_base_days' => $gosiBase
          ]
        );
        $created[] = ['code'=>'GOSI_DED','amount'=>$gosiDed];
      }
    }

    // 4.2 TODO: خصم السلف (لو عندنا جدول loans/advances)
    // $adv = calc_remaining_advance($employeeId); if ($adv>0) insert_auto_item(... 'ADVANCE_DED' ...)

    // 4.3 TODO: جزاءات/غيابات غير مخصومة (لو عندنا جدول penalties/attendance)
    // $pen = calc_pending_penalties($employeeId, $lastDay); if ($pen>0) insert_auto_item(... 'PENALTY' ...)
  }

  recalc_settlement_totals($GLOBALS['conn'], $sid);

  $GLOBALS['conn']->commit();
  echo json_encode(['ok'=>true, 'created'=>$created, 'with_deductions'=>$includeDeductions]);
} catch (Exception $e) {
  if ($GLOBALS['conn']->inTransaction()) $GLOBALS['conn']->rollBack();
  echo json_encode(['ok'=>false, 'msg'=>$e->getMessage()]);
}
